package message

import (
	"reflect"
	"testing"
)

func TestReadLDAPMessage(t *testing.T) {
	for i, test := range getLDAPMessageTestData() {
		message, err := ReadLDAPMessage(&test.bytes)
		if err != nil {
			t.Errorf("#%d failed reading bytes at offset %d (%s): %s", i, test.bytes.offset, test.bytes.DumpCurrentBytes(), err)
		} else if !reflect.DeepEqual(message, test.out) {
			t.Errorf("#%d:\nGOT:\n%#+v\nEXPECTED:\n%#+v", i, message, test.out)
		}
	}
}

type LDAPMessageTestData struct {
	bytes Bytes
	out   LDAPMessage
}

func getLDAPMessageTestData() (ret []LDAPMessageTestData) {
	return []LDAPMessageTestData{
		// Request 1: client => bind request
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c020101600702010304008000
					0x30, 0x0c,
					0x02, 0x01, 0x01,
					//        BindRequest ::= [APPLICATION 0] SEQUENCE {
					0x60, 0x07,
					//             version                 INTEGER (1 ..  127),
					0x02, 0x01, 0x03,
					//             name                    LDAPDN,
					0x04, 0x00,
					//             authentication          AuthenticationChoice }
					//        AuthenticationChoice ::= CHOICE {
					//             simple                  [0] OCTET STRING,
					0x80, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(1),
				protocolOp: BindRequest{
					version:        INTEGER(3),
					name:           LDAPDN(""),
					authentication: OCTETSTRING(""),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 2: server => bind response
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010161070a010004000400
					0x30, 0x0c,
					0x02, 0x01, 0x01,
					//        BindResponse ::= [APPLICATION 1] SEQUENCE {
					//             COMPONENTS OF LDAPResult,
					0x61, 0x07,
					//        		LDAPResult ::= SEQUENCE {
					//             		resultCode         ENUMERATED {
					//                 		 success                      (0),
					//                  		...  },
					0x0a, 0x01, 0x00,
					//             		matchedDN          LDAPDN,
					0x04, 0x00,
					//             		diagnosticMessage  LDAPString,
					0x04, 0x00,
					//             referral           [3] Referral OPTIONAL }
					//             serverSaslCreds    [7] OCTET STRING OPTIONAL }
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x01)),
				protocolOp: BindResponse{
					LDAPResult: LDAPResult{
						resultCode:        0,
						matchedDN:         LDAPDN(""),
						diagnosticMessage: LDAPString(""),
					},
				},
			},
		},

		// Request 3: client => search request
		// select "subschemaSubentry" from entries where "objectClass" is present
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3038020102633304000a01000a0103020100020100010100870b6f626a656374436c61737330130411737562736368656d61537562656e747279
					0x30, 0x38,
					0x02, 0x01, 0x02,
					//        SearchRequest ::= [APPLICATION 3] SEQUENCE {
					0x63, 0x33,
					//             baseObject      LDAPDN,
					0x04, 0x00,
					//             scope           ENUMERATED {
					//                  baseObject              (0),
					//                  ...  },
					0x0a, 0x01, 0x00,
					//             derefAliases    ENUMERATED {
					//                  derefAlways             (3) },
					0x0a, 0x01, 0x03,
					//             sizeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             timeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             typesOnly       BOOLEAN,
					0x01, 0x01, 0x00,
					//             filter          Filter,
					//        			Filter ::= CHOICE {
					//             			present         [7] AttributeDescription,
					//             			...  }
					//        			AttributeDescription ::= LDAPString
					//                  			              -- Constrained to <attributedescription>
					//                             			      -- [RFC4512]
					0x87, 0x0b,
					// "objectClass"
					0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
					//             attributes      AttributeSelection }
					//        AttributeSelection ::= SEQUENCE OF selector LDAPString
					//                       -- The LDAPString is constrained to
					//                       -- <attributeSelector> in Section 4.5.1.8
					0x30, 0x13,
					0x04, 0x11,
					// "subschemaSubentry"
					0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x53, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x02)),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN(""),
					scope:        SearchRequestScopeBaseObject,
					derefAliases: SearchRequetDerefAliasesDerefAlways,
					sizeLimit:    0,
					timeLimit:    0,
					typesOnly:    false,
					filter:       FilterPresent("objectClass"),
					attributes:   AttributeSelection([]LDAPString{"subschemaSubentry"}),
				},
			},
		},

		// Request 4: server => search result entry
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 302b02010264260400302230200411737562736368656d61537562656e747279310b0409636e3d736368656d61
					0x30, 0x2b,
					0x02, 0x01, 0x02,
					//        SearchResultEntry ::= [APPLICATION 4] SEQUENCE {
					0x64, 0x26,
					//             objectName      LDAPDN,
					0x04, 0x00,
					//             attributes      PartialAttributeList }
					//        			PartialAttributeList ::= SEQUENCE OF
					//                  			           partialAttribute PartialAttribute
					0x30, 0x22,
					//        				PartialAttribute ::= SEQUENCE {
					0x30, 0x20,
					//             				type       AttributeDescription,
					0x04, 0x11,
					// "subschemaSubentry"
					0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x53, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79,
					//             				vals       SET OF value AttributeValue }
					//        						AttributeValue ::= OCTET STRING
					0x31, 0x0b,
					0x04, 0x09,
					// "cn=schema"
					0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x02)),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN(""),
					attributes: PartialAttributeList(
						[]PartialAttribute{
							{
								type_: AttributeDescription("subschemaSubentry"),
								vals:  []AttributeValue{AttributeValue("cn=schema")},
							},
						},
					),
				},
			},
		},

		// Request 5: server => search result done
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010265070a010004000400
					0x30, 0x0c,
					0x02, 0x01, 0x02,
					//        SearchResultDone ::= [APPLICATION 5] LDAPResult
					//        LDAPResult ::= SEQUENCE {
					0x65, 0x07,
					//             resultCode         ENUMERATED {
					//                  success                      (0),
					//                  ...  },
					0x0a, 0x01, 0x00,
					//             matchedDN          LDAPDN,
					0x04, 0x00,
					//             diagnosticMessage  LDAPString,
					0x04, 0x00,
					//             referral           [3] Referral OPTIONAL }
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x02)),
				protocolOp: SearchResultDone(LDAPResult{
					resultCode:        ResultCodeSuccess,
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
				}),
			},
		},

		// Request 6: client => search request
		// select
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 305d02010363580409636e3d736368656d610a01000a0103020100020100010100a318040b6f626a656374436c6173730409737562736368656d613022040f63726561746554696d657374616d70040f6d6f6469667954696d657374616d70
					0x30, 0x5d,
					0x02, 0x01, 0x03,
					//        SearchRequest ::= [APPLICATION 3] SEQUENCE {
					0x63, 0x58,
					//             baseObject      LDAPDN,
					// "cn=schema"
					0x04, 0x09, 0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61,
					//             scope           ENUMERATED {
					//                  baseObject              (0),
					//                  ...  },
					0x0a, 0x01, 0x00,
					//             derefAliases    ENUMERATED {
					//                  derefAlways             (3) },
					0x0a, 0x01, 0x03,
					//             sizeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             timeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             typesOnly       BOOLEAN,
					0x01, 0x01, 0x00,
					//             filter          Filter,
					//        Filter ::= CHOICE {
					//             equalityMatch   [3] AttributeValueAssertion,
					//        			AttributeValueAssertion ::= SEQUENCE {
					0xa3, 0x18,
					//             			attributeDesc   AttributeDescription,
					// "objectClass"
					0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
					//             			assertionValue  AssertionValue }
					// "subschema"
					0x04, 0x09, 0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61,
					//             attributes      AttributeSelection }
					//        				AttributeSelection ::= SEQUENCE OF selector LDAPString
					//                      				 -- The LDAPString is constrained to
					//                       				-- <attributeSelector> in Section 4.5.1.8
					0x30, 0x22,
					// "createTimestamp"
					0x04, 0x0f, 0x63, 0x72, 0x65, 0x61, 0x74, 0x65, 0x54, 0x69, 0x6d, 0x65, 0x73, 0x74, 0x61, 0x6d, 0x70,
					// "modifyTimestamp"
					0x04, 0x0f, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x54, 0x69, 0x6d, 0x65, 0x73, 0x74, 0x61, 0x6d, 0x70,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x03)),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("cn=schema"),
					scope:        SearchRequestScopeBaseObject,
					derefAliases: SearchRequetDerefAliasesDerefAlways,
					sizeLimit:    0,
					timeLimit:    0,
					typesOnly:    false,
					filter: FilterEqualityMatch(
						AttributeValueAssertion{
							attributeDesc:  AttributeDescription("objectClass"),
							assertionValue: AssertionValue("subschema"),
						}),
					attributes: AttributeSelection([]LDAPString{"createTimestamp", "modifyTimestamp"}),
				},
			},
		},

		// Request 7: server => search result entry
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 305e02010364590409636e3d736368656d61304c3024040f6d6f6469667954696d657374616d703111040f32303039303831383032323733335a3024040f63726561746554696d657374616d703111040f32303039303831383032323733335a
					0x30, 0x5e,
					0x02, 0x01, 0x03,
					//        SearchResultEntry ::= [APPLICATION 4] SEQUENCE {
					0x64, 0x59,
					//             objectName      LDAPDN,
					// "cn=schema"
					0x04, 0x09, 0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61,
					//             attributes      PartialAttributeList }
					//        			PartialAttributeList ::= SEQUENCE OF
					//                  			           partialAttribute PartialAttribute
					0x30, 0x4c,
					//        				PartialAttribute ::= SEQUENCE {
					0x30, 0x24,
					//             				type       AttributeDescription,
					// "modifyTimestamp"
					0x04, 0x0f, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x54, 0x69, 0x6d, 0x65, 0x73, 0x74, 0x61, 0x6d, 0x70,
					//             				vals       SET OF value AttributeValue }
					0x31, 0x11,
					// "20090818022733Z"
					0x04, 0x0f, 0x32, 0x30, 0x30, 0x39, 0x30, 0x38, 0x31, 0x38, 0x30, 0x32, 0x32, 0x37, 0x33, 0x33, 0x5a,
					//        				PartialAttribute ::= SEQUENCE {
					0x30, 0x24,
					//             				type       AttributeDescription,
					// "createTimestamp"
					0x04, 0x0f, 0x63, 0x72, 0x65, 0x61, 0x74, 0x65, 0x54, 0x69, 0x6d, 0x65, 0x73, 0x74, 0x61, 0x6d, 0x70,
					//             				vals       SET OF value AttributeValue }
					0x31, 0x11,
					// "20090818022733Z"
					0x04, 0x0f, 0x32, 0x30, 0x30, 0x39, 0x30, 0x38, 0x31, 0x38, 0x30, 0x32, 0x32, 0x37, 0x33, 0x33, 0x5a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x03)),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("cn=schema"),
					attributes: PartialAttributeList(
						[]PartialAttribute{
							{
								type_: AttributeDescription("modifyTimestamp"),
								vals:  []AttributeValue{AttributeValue("20090818022733Z")},
							},
							{
								type_: AttributeDescription("createTimestamp"),
								vals:  []AttributeValue{AttributeValue("20090818022733Z")},
							},
						},
					),
				},
			},
		},

		// Request 8
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010365070a010004000400
					0x30, 0x0c,
					0x02, 0x01, 0x03,
					//        SearchResultDone ::= [APPLICATION 5] LDAPResult
					//        LDAPResult ::= SEQUENCE {
					0x65, 0x07,
					//             resultCode         ENUMERATED {
					//                  success                      (0),
					//                  ...  },
					0x0a, 0x01, 0x00,
					//             matchedDN          LDAPDN,
					0x04, 0x00,
					//             diagnosticMessage  LDAPString,
					0x04, 0x00,
					//             referral           [3] Referral OPTIONAL }
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x03)),
				protocolOp: SearchResultDone(LDAPResult{
					resultCode:        ResultCodeSuccess,
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
				}),
			},
		},

		// Request 9: client => search request
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081dd0201046381d704000a01000a0100020100020100010100870b6f626a656374436c6173733081b6040e6e616d696e67436f6e74657874730411737562736368656d61537562656e7472790414737570706f727465644c44415056657273696f6e0417737570706f727465645341534c4d656368616e69736d730412737570706f72746564457874656e73696f6e0410737570706f72746564436f6e74726f6c0411737570706f727465644665617475726573040a76656e646f724e616d65040d76656e646f7256657273696f6e04012b040b6f626a656374436c617373
					0x30, 0x81, 0xdd,
					0x02, 0x01, 0x04,
					//        SearchRequest ::= [APPLICATION 3] SEQUENCE {
					0x63, 0x81, 0xd7,
					//             baseObject      LDAPDN,
					0x04, 0x00,
					//             scope           ENUMERATED {
					//                  baseObject              (0),
					//                  ...  },
					0x0a, 0x01, 0x00,
					//             derefAliases    ENUMERATED {
					//                  neverDerefAliases             (0) },
					0x0a, 0x01, 0x00,
					//             sizeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             timeLimit       INTEGER (0 ..  maxInt),
					0x02, 0x01, 0x00,
					//             typesOnly       BOOLEAN,
					0x01, 0x01, 0x00,
					//             filter          Filter,
					//        Filter ::= CHOICE {
					//             present         [7] AttributeDescription,
					// "objetClass"
					0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
					//             attributes      AttributeSelection }
					//        				AttributeSelection ::= SEQUENCE OF selector LDAPString
					//                      				 -- The LDAPString is constrained to
					//                       				-- <attributeSelector> in Section 4.5.1.8
					0x30, 0x81, 0xb6,
					// namingContexts
					0x04, 0x0e, 0x6e, 0x61, 0x6d, 0x69, 0x6e, 0x67, 0x43, 0x6f, 0x6e, 0x74, 0x65, 0x78, 0x74, 0x73,
					// subschemaSubentry
					0x04, 0x11, 0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x53, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79,
					// supportedLDAPVersion
					0x04, 0x14, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x4c, 0x44, 0x41, 0x50, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
					// supportedSASLMechanisms
					0x04, 0x17, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x53, 0x41, 0x53, 0x4c, 0x4d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x73, 0x6d, 0x73,
					// supportedExtension
					0x04, 0x12, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x45, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e,
					// supportedControl
					0x04, 0x10, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c,
					// supportedFeatures
					0x04, 0x11, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x46, 0x65, 0x61, 0x74, 0x75, 0x72, 0x65, 0x73,
					// vendorName
					0x04, 0x0a, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x4e, 0x61, 0x6d, 0x65,
					// vendorVersion
					0x04, 0x0d, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
					// +
					0x04, 0x01, 0x2b,
					// objectClass
					0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x04)),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN(""),
					scope:        SearchRequestScopeBaseObject,
					derefAliases: SearchRequetDerefAliasesNeverDerefAliases,
					sizeLimit:    0,
					timeLimit:    0,
					typesOnly:    false,
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection([]LDAPString{
						"namingContexts",
						"subschemaSubentry",
						"supportedLDAPVersion",
						"supportedSASLMechanisms",
						"supportedExtension",
						"supportedControl",
						"supportedFeatures",
						"vendorName",
						"vendorVersion",
						"+",
						"objectClass",
					}),
				},
			},
		},

		// Request 10
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308203850201046482037e040030820378302a040a76656e646f724e616d65311c041a41706163686520536f66747761726520466f756e646174696f6e301c040d76656e646f7256657273696f6e310b0409322e302e302d4d31343026040b6f626a656374436c61737331170403746f700410657874656e7369626c654f626a65637430200411737562736368656d61537562656e747279310b0409636e3d736368656d61301b0414737570706f727465644c44415056657273696f6e31030401333082012e0410737570706f72746564436f6e74726f6c318201180417322e31362e3834302e312e3131333733302e332e342e330417312e332e362e312e342e312e343230332e312e31302e310417322e31362e3834302e312e3131333733302e332e342e320418312e332e362e312e342e312e343230332e312e392e312e340419312e332e362e312e342e312e34322e322e32372e382e352e310418312e332e362e312e342e312e343230332e312e392e312e310418312e332e362e312e342e312e343230332e312e392e312e330418312e332e362e312e342e312e343230332e312e392e312e320417312e332e362e312e342e312e31383036302e302e302e310417322e31362e3834302e312e3131333733302e332e342e370416312e322e3834302e3131333535362e312e342e3331393081910412737570706f72746564457874656e73696f6e317b0416312e332e362e312e342e312e313436362e32303033360416312e332e362e312e342e312e313436362e32303033370417312e332e362e312e342e312e31383036302e302e312e350417312e332e362e312e342e312e31383036302e302e312e330417312e332e362e312e342e312e343230332e312e31312e3130530417737570706f727465645341534c4d656368616e69736d73313804044e544c4d0406475353415049040a4753532d53504e45474f04084352414d2d4d4435040653494d504c45040a4449474553542d4d443530330409656e747279555549443126042466323930343235632d383237322d346536322d386136372d3932623036663338646266353046040e6e616d696e67436f6e7465787473313404096f753d73797374656d041164633d6578616d706c652c64633d636f6d04096f753d736368656d6104096f753d636f6e666967302d0411737570706f72746564466561747572657331180416312e332e362e312e342e312e343230332e312e352e31
					0x30, 0x82, 0x03, 0x85, 0x02, 0x01, 0x04, 0x64, 0x82, 0x03, 0x7e, 0x04, 0x00, 0x30, 0x82, 0x03, 0x78, 0x30, 0x2a, 0x04, 0x0a, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x4e, 0x61, 0x6d, 0x65, 0x31, 0x1c, 0x04, 0x1a, 0x41, 0x70, 0x61, 0x63, 0x68, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x46, 0x6f, 0x75, 0x6e, 0x64, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x30, 0x1c, 0x04, 0x0d, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x31, 0x0b, 0x04, 0x09, 0x32, 0x2e, 0x30, 0x2e, 0x30, 0x2d, 0x4d, 0x31, 0x34, 0x30, 0x26, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x17, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x10, 0x65, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x30, 0x20, 0x04, 0x11, 0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x53, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x31, 0x0b, 0x04, 0x09, 0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x30, 0x1b, 0x04, 0x14, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x4c, 0x44, 0x41, 0x50, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x31, 0x03, 0x04, 0x01, 0x33, 0x30, 0x82, 0x01, 0x2e, 0x04, 0x10, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x31, 0x82, 0x01, 0x18, 0x04, 0x17, 0x32, 0x2e, 0x31, 0x36, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x2e, 0x31, 0x31, 0x33, 0x37, 0x33, 0x30, 0x2e, 0x33, 0x2e, 0x34, 0x2e, 0x33, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x31, 0x30, 0x2e, 0x31, 0x04, 0x17, 0x32, 0x2e, 0x31, 0x36, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x2e, 0x31, 0x31, 0x33, 0x37, 0x33, 0x30, 0x2e, 0x33, 0x2e, 0x34, 0x2e, 0x32, 0x04, 0x18, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x39, 0x2e, 0x31, 0x2e, 0x34, 0x04, 0x19, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x2e, 0x32, 0x2e, 0x32, 0x37, 0x2e, 0x38, 0x2e, 0x35, 0x2e, 0x31, 0x04, 0x18, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x39, 0x2e, 0x31, 0x2e, 0x31, 0x04, 0x18, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x39, 0x2e, 0x31, 0x2e, 0x33, 0x04, 0x18, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x39, 0x2e, 0x31, 0x2e, 0x32, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x38, 0x30, 0x36, 0x30, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x04, 0x17, 0x32, 0x2e, 0x31, 0x36, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x2e, 0x31, 0x31, 0x33, 0x37, 0x33, 0x30, 0x2e, 0x33, 0x2e, 0x34, 0x2e, 0x37, 0x04, 0x16, 0x31, 0x2e, 0x32, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x31, 0x33, 0x35, 0x35, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x33, 0x31, 0x39, 0x30, 0x81, 0x91, 0x04, 0x12, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x45, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x31, 0x7b, 0x04, 0x16, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36, 0x2e, 0x32, 0x30, 0x30, 0x33, 0x36, 0x04, 0x16, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36, 0x2e, 0x32, 0x30, 0x30, 0x33, 0x37, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x38, 0x30, 0x36, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x2e, 0x35, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x38, 0x30, 0x36, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x2e, 0x33, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x31, 0x31, 0x2e, 0x31, 0x30, 0x53, 0x04, 0x17, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x53, 0x41, 0x53, 0x4c, 0x4d, 0x65, 0x63, 0x68, 0x61, 0x6e, 0x69, 0x73, 0x6d, 0x73, 0x31, 0x38, 0x04, 0x04, 0x4e, 0x54, 0x4c, 0x4d, 0x04, 0x06, 0x47, 0x53, 0x53, 0x41, 0x50, 0x49, 0x04, 0x0a, 0x47, 0x53, 0x53, 0x2d, 0x53, 0x50, 0x4e, 0x45, 0x47, 0x4f, 0x04, 0x08, 0x43, 0x52, 0x41, 0x4d, 0x2d, 0x4d, 0x44, 0x35, 0x04, 0x06, 0x53, 0x49, 0x4d, 0x50, 0x4c, 0x45, 0x04, 0x0a, 0x44, 0x49, 0x47, 0x45, 0x53, 0x54, 0x2d, 0x4d, 0x44, 0x35, 0x30, 0x33, 0x04, 0x09, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x55, 0x55, 0x49, 0x44, 0x31, 0x26, 0x04, 0x24, 0x66, 0x32, 0x39, 0x30, 0x34, 0x32, 0x35, 0x63, 0x2d, 0x38, 0x32, 0x37, 0x32, 0x2d, 0x34, 0x65, 0x36, 0x32, 0x2d, 0x38, 0x61, 0x36, 0x37, 0x2d, 0x39, 0x32, 0x62, 0x30, 0x36, 0x66, 0x33, 0x38, 0x64, 0x62, 0x66, 0x35, 0x30, 0x46, 0x04, 0x0e, 0x6e, 0x61, 0x6d, 0x69, 0x6e, 0x67, 0x43, 0x6f, 0x6e, 0x74, 0x65, 0x78, 0x74, 0x73, 0x31, 0x34, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x04, 0x11, 0x64, 0x63, 0x3d, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2d, 0x04, 0x11, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x46, 0x65, 0x61, 0x74, 0x75, 0x72, 0x65, 0x73, 0x31, 0x18, 0x04, 0x16, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x35, 0x2e, 0x31,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(4),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN(""),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("vendorName"),
							vals: []AttributeValue{
								AttributeValue("Apache Software Foundation"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("vendorVersion"),
							vals: []AttributeValue{
								AttributeValue("2.0.0-M14"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("objectClass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("extensibleObject"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("subschemaSubentry"),
							vals: []AttributeValue{
								AttributeValue("cn=schema"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("supportedLDAPVersion"),
							vals: []AttributeValue{
								AttributeValue("3"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("supportedControl"),
							vals: []AttributeValue{
								AttributeValue("2.16.840.1.113730.3.4.3"),
								AttributeValue("1.3.6.1.4.1.4203.1.10.1"),
								AttributeValue("2.16.840.1.113730.3.4.2"),
								AttributeValue("1.3.6.1.4.1.4203.1.9.1.4"),
								AttributeValue("1.3.6.1.4.1.42.2.27.8.5.1"),
								AttributeValue("1.3.6.1.4.1.4203.1.9.1.1"),
								AttributeValue("1.3.6.1.4.1.4203.1.9.1.3"),
								AttributeValue("1.3.6.1.4.1.4203.1.9.1.2"),
								AttributeValue("1.3.6.1.4.1.18060.0.0.1"),
								AttributeValue("2.16.840.1.113730.3.4.7"),
								AttributeValue("1.2.840.113556.1.4.319"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("supportedExtension"),
							vals: []AttributeValue{
								AttributeValue("1.3.6.1.4.1.1466.20036"),
								AttributeValue("1.3.6.1.4.1.1466.20037"),
								AttributeValue("1.3.6.1.4.1.18060.0.1.5"),
								AttributeValue("1.3.6.1.4.1.18060.0.1.3"),
								AttributeValue("1.3.6.1.4.1.4203.1.11.1"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("supportedSASLMechanisms"),
							vals: []AttributeValue{
								AttributeValue("NTLM"),
								AttributeValue("GSSAPI"),
								AttributeValue("GSS-SPNEGO"),
								AttributeValue("CRAM-MD5"),
								AttributeValue("SIMPLE"),
								AttributeValue("DIGEST-MD5"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("entryUUID"),
							vals: []AttributeValue{
								AttributeValue("f290425c-8272-4e62-8a67-92b06f38dbf5"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("namingContexts"),
							vals: []AttributeValue{
								AttributeValue("ou=system"),
								AttributeValue("dc=example,dc=com"),
								AttributeValue("ou=schema"),
								AttributeValue("ou=config"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("supportedFeatures"),
							vals: []AttributeValue{
								AttributeValue("1.3.6.1.4.1.4203.1.5.1"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 11
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010465070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x04, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(4),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 12
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3028020105632304000a01000a0100020100020100010100870b6f626a656374436c617373300304012a
					0x30, 0x28, 0x02, 0x01, 0x05, 0x63, 0x23, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x00, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x03, 0x04, 0x01, 0x2a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(5),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN(""),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(0),
					sizeLimit:    INTEGER(0),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("*"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 13
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3031020105642c040030283026040b6f626a656374436c61737331170403746f700410657874656e7369626c654f626a656374
					0x30, 0x31, 0x02, 0x01, 0x05, 0x64, 0x2c, 0x04, 0x00, 0x30, 0x28, 0x30, 0x26, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x17, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x10, 0x65, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x62, 0x6c, 0x65, 0x4f, 0x62, 0x6a, 0x65, 0x63, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(5),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN(""),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectClass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("extensibleObject"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 14
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010565070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x05, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(5),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 15
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c020106634704096f753d736368656d610a01000a0103020101020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x4c, 0x02, 0x01, 0x06, 0x63, 0x47, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(6),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=schema"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 16
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303c020106643704096f753d736368656d61302a3028040b6f626a656374636c617373311904126f7267616e697a6174696f6e616c556e69740403746f70
					0x30, 0x3c, 0x02, 0x01, 0x06, 0x64, 0x37, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74, 0x04, 0x03, 0x74, 0x6f, 0x70,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(6),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=schema"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("organizationalUnit"),
								AttributeValue("top"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 17
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010665070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x06, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(6),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 18
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c020107634704096f753d636f6e6669670a01000a0103020101020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x4c, 0x02, 0x01, 0x07, 0x63, 0x47, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(7),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=config"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 19
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303c020107643704096f753d636f6e666967302a3028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x3c, 0x02, 0x01, 0x07, 0x64, 0x37, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(7),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 20
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010765070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x07, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(7),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 21
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c020108634704096f753d73797374656d0a01000a0103020101020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x4c, 0x02, 0x01, 0x08, 0x63, 0x47, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(8),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=system"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 22
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303c020108643704096f753d73797374656d302a3028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x3c, 0x02, 0x01, 0x08, 0x64, 0x37, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(8),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=system"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 23
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010865070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x08, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(8),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 24
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c02010963470409636e3d736368656d610a01000a0103020101020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x4c, 0x02, 0x01, 0x09, 0x63, 0x47, 0x04, 0x09, 0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(9),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("cn=schema"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 25
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304e02010964490409636e3d736368656d61303c303a040b6f626a656374436c617373312b0403746f700409737562736368656d610408737562656e747279040f617061636865537562736368656d61
					0x30, 0x4e, 0x02, 0x01, 0x09, 0x64, 0x49, 0x04, 0x09, 0x63, 0x6e, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x30, 0x3c, 0x30, 0x3a, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x2b, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x09, 0x73, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x04, 0x08, 0x73, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x04, 0x0f, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x53, 0x75, 0x62, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(9),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("cn=schema"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectClass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("subschema"),
								AttributeValue("subentry"),
								AttributeValue("apacheSubschema"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 26
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010965070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x09, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(9),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 27
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 305402010a634f041164633d6578616d706c652c64633d636f6d0a01000a0103020101020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x54, 0x02, 0x01, 0x0a, 0x63, 0x4f, 0x04, 0x11, 0x64, 0x63, 0x3d, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(10),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("dc=example,dc=com"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 28
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303802010a6433041164633d6578616d706c652c64633d636f6d301e301c040b6f626a656374636c617373310d0406646f6d61696e0403746f70
					0x30, 0x38, 0x02, 0x01, 0x0a, 0x64, 0x33, 0x04, 0x11, 0x64, 0x63, 0x3d, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c, 0x64, 0x63, 0x3d, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x30, 0x1c, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x0d, 0x04, 0x06, 0x64, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x04, 0x03, 0x74, 0x6f, 0x70,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(10),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("dc=example,dc=com"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("domain"),
								AttributeValue("top"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 29
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010a65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0a, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(10),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 30
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304d02010b634804096f753d636f6e6669670a01010a0103020203e8020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x4d, 0x02, 0x01, 0x0b, 0x63, 0x48, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x01, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(11),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=config"),
					scope:        ENUMERATED(1),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 31
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306702010b646204286164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e66696730363034040b6f626a656374636c61737331250403746f7004146164732d6469726563746f72795365727669636504086164732d62617365
					0x30, 0x67, 0x02, 0x01, 0x0b, 0x64, 0x62, 0x04, 0x28, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x36, 0x30, 0x34, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x25, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x14, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(11),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("ads-directoryService"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 32
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010b65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0b, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(11),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 33
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303102010c632c04096f753d636f6e6669670a01000a0103020100020100010100870b6f626a656374436c617373300304012a
					0x30, 0x31, 0x02, 0x01, 0x0c, 0x63, 0x2c, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x03, 0x04, 0x01, 0x2a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(12),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=config"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(0),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("*"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 34
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c02010c644704096f753d636f6e666967303a300e04026f7531080406636f6e6669673028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x4c, 0x02, 0x01, 0x0c, 0x64, 0x47, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x3a, 0x30, 0x0e, 0x04, 0x02, 0x6f, 0x75, 0x31, 0x08, 0x04, 0x06, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(12),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("ou"),
							vals: []AttributeValue{
								AttributeValue("config"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 35
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010c65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0c, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(12),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 36
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306c02010d636704286164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01010a0103020203e8020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x6c, 0x02, 0x01, 0x0d, 0x63, 0x67, 0x04, 0x28, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x01, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(1),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 37
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 307b02010d647604456164732d6a6f75726e616c49643d64656661756c744a6f75726e616c2c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e666967302d302b040b6f626a656374636c617373311c0403746f70040b6164732d6a6f75726e616c04086164732d62617365
					0x30, 0x7b, 0x02, 0x01, 0x0d, 0x64, 0x76, 0x04, 0x45, 0x61, 0x64, 0x73, 0x2d, 0x6a, 0x6f, 0x75, 0x72, 0x6e, 0x61, 0x6c, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x4a, 0x6f, 0x75, 0x72, 0x6e, 0x61, 0x6c, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2d, 0x30, 0x2b, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x1c, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x0b, 0x61, 0x64, 0x73, 0x2d, 0x6a, 0x6f, 0x75, 0x72, 0x6e, 0x61, 0x6c, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-journalId=defaultJournal,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("ads-journal"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 38
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306b02010d646604386f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e666967302a3028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x6b, 0x02, 0x01, 0x0d, 0x64, 0x66, 0x04, 0x38, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 39
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306902010d646404366f753d706172746974696f6e732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e666967302a3028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x69, 0x02, 0x01, 0x0d, 0x64, 0x64, 0x04, 0x36, 0x6f, 0x75, 0x3d, 0x70, 0x61, 0x72, 0x74, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=partitions,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 40
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30818102010d647c04496164732d6368616e67654c6f6749643d64656661756c744368616e67654c6f672c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e666967302f302d040b6f626a656374636c617373311e0403746f7004086164732d62617365040d6164732d6368616e67654c6f67
					0x30, 0x81, 0x81, 0x02, 0x01, 0x0d, 0x64, 0x7c, 0x04, 0x49, 0x61, 0x64, 0x73, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x4c, 0x6f, 0x67, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x43, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x4c, 0x6f, 0x67, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2f, 0x30, 0x2d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x1e, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65, 0x04, 0x0d, 0x61, 0x64, 0x73, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x4c, 0x6f, 0x67,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-changeLogId=defaultChangeLog,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("ads-base"),
								AttributeValue("ads-changeLog"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 41
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306602010d646104336f753d736572766572732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e666967302a3028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x66, 0x02, 0x01, 0x0d, 0x64, 0x61, 0x04, 0x33, 0x6f, 0x75, 0x3d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x2a, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=servers,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 42
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010d65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0d, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 43
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 305002010e634b04286164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01000a0103020100020100010100870b6f626a656374436c617373300304012a
					0x30, 0x50, 0x02, 0x01, 0x0e, 0x63, 0x4b, 0x04, 0x28, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x03, 0x04, 0x01, 0x2a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(14),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(0),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("*"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 44
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3082017c02010e6482017504286164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e66696730820147302304166164732d6469726563746f72797365727669636569643109040764656661756c74302104166164732d647373796e63706572696f646d696c6c69733107040531353030303024041a6164732d6473616c6c6f77616e6f6e796d6f757361636365737331060404545255453016040f6164732d64737265706c696361696431030401313025041a6164732d6473616363657373636f6e74726f6c656e61626c65643107040546414c5345301f04146164732d647370617373776f726468696464656e3107040546414c5345302a041f6164732d647364656e6f726d616c697a656f706174747273656e61626c65643107040546414c53453015040b6164732d656e61626c656431060404545255453034040b6f626a656374636c61737331250403746f7004146164732d6469726563746f72795365727669636504086164732d62617365
					0x30, 0x82, 0x01, 0x7c, 0x02, 0x01, 0x0e, 0x64, 0x82, 0x01, 0x75, 0x04, 0x28, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x82, 0x01, 0x47, 0x30, 0x23, 0x04, 0x16, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x69, 0x64, 0x31, 0x09, 0x04, 0x07, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x30, 0x21, 0x04, 0x16, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x73, 0x79, 0x6e, 0x63, 0x70, 0x65, 0x72, 0x69, 0x6f, 0x64, 0x6d, 0x69, 0x6c, 0x6c, 0x69, 0x73, 0x31, 0x07, 0x04, 0x05, 0x31, 0x35, 0x30, 0x30, 0x30, 0x30, 0x24, 0x04, 0x1a, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x61, 0x6c, 0x6c, 0x6f, 0x77, 0x61, 0x6e, 0x6f, 0x6e, 0x79, 0x6d, 0x6f, 0x75, 0x73, 0x61, 0x63, 0x63, 0x65, 0x73, 0x73, 0x31, 0x06, 0x04, 0x04, 0x54, 0x52, 0x55, 0x45, 0x30, 0x16, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x72, 0x65, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x69, 0x64, 0x31, 0x03, 0x04, 0x01, 0x31, 0x30, 0x25, 0x04, 0x1a, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x61, 0x63, 0x63, 0x65, 0x73, 0x73, 0x63, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x64, 0x31, 0x07, 0x04, 0x05, 0x46, 0x41, 0x4c, 0x53, 0x45, 0x30, 0x1f, 0x04, 0x14, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x68, 0x69, 0x64, 0x64, 0x65, 0x6e, 0x31, 0x07, 0x04, 0x05, 0x46, 0x41, 0x4c, 0x53, 0x45, 0x30, 0x2a, 0x04, 0x1f, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x73, 0x64, 0x65, 0x6e, 0x6f, 0x72, 0x6d, 0x61, 0x6c, 0x69, 0x7a, 0x65, 0x6f, 0x70, 0x61, 0x74, 0x74, 0x72, 0x73, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x64, 0x31, 0x07, 0x04, 0x05, 0x46, 0x41, 0x4c, 0x53, 0x45, 0x30, 0x15, 0x04, 0x0b, 0x61, 0x64, 0x73, 0x2d, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x64, 0x31, 0x06, 0x04, 0x04, 0x54, 0x52, 0x55, 0x45, 0x30, 0x34, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x25, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x14, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(14),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("ads-directoryserviceid"),
							vals: []AttributeValue{
								AttributeValue("default"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dssyncperiodmillis"),
							vals: []AttributeValue{
								AttributeValue("15000"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dsallowanonymousaccess"),
							vals: []AttributeValue{
								AttributeValue("TRUE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dsreplicaid"),
							vals: []AttributeValue{
								AttributeValue("1"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dsaccesscontrolenabled"),
							vals: []AttributeValue{
								AttributeValue("FALSE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dspasswordhidden"),
							vals: []AttributeValue{
								AttributeValue("FALSE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-dsdenormalizeopattrsenabled"),
							vals: []AttributeValue{
								AttributeValue("FALSE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-enabled"),
							vals: []AttributeValue{
								AttributeValue("TRUE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("ads-directoryService"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 45
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010e65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0e, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(14),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 46
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 306002010f635b04386f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01000a0103020100020100010100870b6f626a656374436c617373300304012a
					0x30, 0x60, 0x02, 0x01, 0x0f, 0x63, 0x5b, 0x04, 0x38, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x03, 0x04, 0x01, 0x2a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(15),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=interceptors,ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(0),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("*"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 47
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30818102010f647c04386f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673040301404026f75310e040c696e746572636570746f72733028040b6f626a656374636c61737331190403746f7004126f7267616e697a6174696f6e616c556e6974
					0x30, 0x81, 0x81, 0x02, 0x01, 0x0f, 0x64, 0x7c, 0x04, 0x38, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x40, 0x30, 0x14, 0x04, 0x02, 0x6f, 0x75, 0x31, 0x0e, 0x04, 0x0c, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x30, 0x28, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x19, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x12, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x55, 0x6e, 0x69, 0x74,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(15),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("ou"),
							vals: []AttributeValue{
								AttributeValue("interceptors"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("organizationalUnit"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 48
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02010f65070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x0f, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(15),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 49
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 307c020110637704386f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01010a0103020203e8020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x7c, 0x02, 0x01, 0x10, 0x63, 0x77, 0x04, 0x38, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x01, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=interceptors,ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(1),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 50
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30819a020110648194045f6164732d696e746572636570746f7249643d657863657074696f6e496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x9a, 0x02, 0x01, 0x10, 0x64, 0x81, 0x94, 0x04, 0x5f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x65, 0x78, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=exceptionInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 51
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308196020110648190045b6164732d696e746572636570746f7249643d6576656e74496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x96, 0x02, 0x01, 0x10, 0x64, 0x81, 0x90, 0x04, 0x5b, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x65, 0x76, 0x65, 0x6e, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=eventInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 52
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a502011064819f046a6164732d696e746572636570746f7249643d6f7065726174696f6e616c417474726962757465496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa5, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9f, 0x04, 0x6a, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=operationalAttributeInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 53
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081be0201106481b804646164732d696e746572636570746f7249643d61757468656e7469636174696f6e496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673050304e040b6f626a656374636c617373313f040f6164732d696e746572636570746f720403746f7004086164732d62617365041d6164732d61757468656e7469636174696f6e496e746572636570746f72
					0x30, 0x81, 0xbe, 0x02, 0x01, 0x10, 0x64, 0x81, 0xb8, 0x04, 0x64, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x50, 0x30, 0x4e, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x3f, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65, 0x04, 0x1d, 0x61, 0x64, 0x73, 0x2d, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=authenticationInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
								AttributeValue("ads-authenticationInterceptor"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 54
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a202011064819c04676164732d696e746572636570746f7249643d616369417574686f72697a6174696f6e496e746572636570746f72322c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa2, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9c, 0x04, 0x67, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x32, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=aciAuthorizationInterceptor2,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 55
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a002011064819a04656164732d696e746572636570746f7249643d70617373776f726448617368696e67496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa0, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9a, 0x04, 0x65, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x48, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=passwordHashingInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 56
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308197020110648191045c6164732d696e746572636570746f7249643d736368656d61496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x97, 0x02, 0x01, 0x10, 0x64, 0x81, 0x91, 0x04, 0x5c, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=schemaInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 57
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a402011064819e04696164732d696e746572636570746f7249643d61646d696e697374726174697665506f696e74496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa4, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9e, 0x04, 0x69, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x64, 0x6d, 0x69, 0x6e, 0x69, 0x73, 0x74, 0x72, 0x61, 0x74, 0x69, 0x76, 0x65, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=administrativePointInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 58
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308199020110648193045e6164732d696e746572636570746f7249643d726566657272616c496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x99, 0x02, 0x01, 0x10, 0x64, 0x81, 0x93, 0x04, 0x5e, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x61, 0x6c, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=referralInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 59
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30819e02011064819804636164732d696e746572636570746f7249643d6b657944657269766174696f6e496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x9e, 0x02, 0x01, 0x10, 0x64, 0x81, 0x98, 0x04, 0x63, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x6b, 0x65, 0x79, 0x44, 0x65, 0x72, 0x69, 0x76, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=keyDerivationInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 60
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30819e02011064819804636164732d696e746572636570746f7249643d6e6f726d616c697a6174696f6e496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x9e, 0x02, 0x01, 0x10, 0x64, 0x81, 0x98, 0x04, 0x63, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x6e, 0x6f, 0x72, 0x6d, 0x61, 0x6c, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=normalizationInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 61
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308199020110648193045e6164732d696e746572636570746f7249643d737562656e747279496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x99, 0x02, 0x01, 0x10, 0x64, 0x81, 0x93, 0x04, 0x5e, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x73, 0x75, 0x62, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=subentryInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 62
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a502011064819f046a6164732d696e746572636570746f7249643d64656661756c74417574686f72697a6174696f6e496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa5, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9f, 0x04, 0x6a, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=defaultAuthorizationInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 63
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 308198020110648192045d6164732d696e746572636570746f7249643d74726967676572496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0x98, 0x02, 0x01, 0x10, 0x64, 0x81, 0x92, 0x04, 0x5d, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x74, 0x72, 0x69, 0x67, 0x67, 0x65, 0x72, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=triggerInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 64
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081a402011064819e04696164732d696e746572636570746f7249643d636f6c6c656374697665417474726962757465496e746572636570746f722c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669673031302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d62617365
					0x30, 0x81, 0xa4, 0x02, 0x01, 0x10, 0x64, 0x81, 0x9e, 0x04, 0x69, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x63, 0x6f, 0x6c, 0x6c, 0x65, 0x63, 0x74, 0x69, 0x76, 0x65, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x31, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=collectiveAttributeInterceptor,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 65
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011065070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x10, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(16),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 66
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30819002011163818a04676164732d696e746572636570746f7249643d616369417574686f72697a6174696f6e496e746572636570746f72322c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01000a0103020100020100010100870b6f626a656374436c617373300304012a
					0x30, 0x81, 0x90, 0x02, 0x01, 0x11, 0x63, 0x81, 0x8a, 0x04, 0x67, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x32, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x00, 0x0a, 0x01, 0x03, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x03, 0x04, 0x01, 0x2a,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(17),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ads-interceptorId=aciAuthorizationInterceptor2,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(0),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(0),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("*"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 67
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3082018d0201116482018604676164732d696e746572636570746f7249643d616369417574686f72697a6174696f6e496e746572636570746f72322c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e66696730820119302f040b6f626a656374636c6173733120040f6164732d696e746572636570746f720403746f7004086164732d626173653015040b6164732d656e61626c65643106040454525545306004186164732d696e746572636570746f72636c6173736e616d65314404426f72672e6170616368652e6469726563746f72792e7365727665722e636f72652e617574687a2e416369417574686f72697a6174696f6e496e746572636570746f72301b04146164732d696e746572636570746f726f726465723103040134305004116164732d696e746572636570746f726964313b041b616369417574686f72697a6174696f6e496e746572636570746f72041c616369417574686f72697a6174696f6e496e746572636570746f7232
					0x30, 0x82, 0x01, 0x8d, 0x02, 0x01, 0x11, 0x64, 0x82, 0x01, 0x86, 0x04, 0x67, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x32, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x30, 0x82, 0x01, 0x19, 0x30, 0x2f, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x20, 0x04, 0x0f, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x08, 0x61, 0x64, 0x73, 0x2d, 0x62, 0x61, 0x73, 0x65, 0x30, 0x15, 0x04, 0x0b, 0x61, 0x64, 0x73, 0x2d, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x64, 0x31, 0x06, 0x04, 0x04, 0x54, 0x52, 0x55, 0x45, 0x30, 0x60, 0x04, 0x18, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x6e, 0x61, 0x6d, 0x65, 0x31, 0x44, 0x04, 0x42, 0x6f, 0x72, 0x67, 0x2e, 0x61, 0x70, 0x61, 0x63, 0x68, 0x65, 0x2e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x2e, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2e, 0x63, 0x6f, 0x72, 0x65, 0x2e, 0x61, 0x75, 0x74, 0x68, 0x7a, 0x2e, 0x41, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x30, 0x1b, 0x04, 0x14, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x6f, 0x72, 0x64, 0x65, 0x72, 0x31, 0x03, 0x04, 0x01, 0x34, 0x30, 0x50, 0x04, 0x11, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x69, 0x64, 0x31, 0x3b, 0x04, 0x1b, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x04, 0x1c, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x32,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(17),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("ads-interceptorId=aciAuthorizationInterceptor2,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectclass"),
							vals: []AttributeValue{
								AttributeValue("ads-interceptor"),
								AttributeValue("top"),
								AttributeValue("ads-base"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-enabled"),
							vals: []AttributeValue{
								AttributeValue("TRUE"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-interceptorclassname"),
							vals: []AttributeValue{
								AttributeValue("org.apache.directory.server.core.authz.AciAuthorizationInterceptor"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-interceptororder"),
							vals: []AttributeValue{
								AttributeValue("4"),
							},
						},
						PartialAttribute{
							type_: AttributeDescription("ads-interceptorid"),
							vals: []AttributeValue{
								AttributeValue("aciAuthorizationInterceptor"),
								AttributeValue("aciAuthorizationInterceptor2"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 68
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011165070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x11, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(17),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 69
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081ac0201126381a604676164732d696e746572636570746f7249643d616369417574686f72697a6174696f6e496e746572636570746f72322c6f753d696e746572636570746f72732c6164732d6469726563746f72795365727669636549643d64656661756c742c6f753d636f6e6669670a01010a0103020203e8020100010100870b6f626a656374436c617373301e040f6861735375626f7264696e61746573040b6f626a656374436c617373
					0x30, 0x81, 0xac, 0x02, 0x01, 0x12, 0x63, 0x81, 0xa6, 0x04, 0x67, 0x61, 0x64, 0x73, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x49, 0x64, 0x3d, 0x61, 0x63, 0x69, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x32, 0x2c, 0x6f, 0x75, 0x3d, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x6f, 0x72, 0x73, 0x2c, 0x61, 0x64, 0x73, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x49, 0x64, 0x3d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x0a, 0x01, 0x01, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x1e, 0x04, 0x0f, 0x68, 0x61, 0x73, 0x53, 0x75, 0x62, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x61, 0x74, 0x65, 0x73, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(18),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ads-interceptorId=aciAuthorizationInterceptor2,ou=interceptors,ads-directoryServiceId=default,ou=config"),
					scope:        ENUMERATED(1),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("hasSubordinates"),
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 70
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011265070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x12, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(18),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 71
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30050201134200
					0x30, 0x05, 0x02, 0x01, 0x13, 0x42, 0x00,
				},
			},
			out: LDAPMessage{
				messageID:  MessageID(19),
				protocolOp: UnbindRequest{},
				controls:   (*Controls)(nil),
			},
		},

		// Request 72: CLIENT AddRequest
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3081b60201156881b0044a636e3d723030582b636f6d6d6f6e4e616d653d54686520723030582b6465736372697074696f6e3d41207465737420757365722c6f753d636f6e73756d6572732c6f753d73797374656d3062301c040b6465736372697074696f6e310d040b4120746573742075736572300c0402736e310604047230307830160402636e311004047230305804085468652072303058301c040b6f626a656374436c617373310d0406706572736f6e0403746f70
					0x30, 0x81, 0xb6, 0x02, 0x01, 0x15, 0x68, 0x81, 0xb0, 0x04, 0x4a, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x4e, 0x61, 0x6d, 0x65, 0x3d, 0x54, 0x68, 0x65, 0x20, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x41, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x30, 0x62, 0x30, 0x1c, 0x04, 0x0b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x0d, 0x04, 0x0b, 0x41, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x72, 0x30, 0x0c, 0x04, 0x02, 0x73, 0x6e, 0x31, 0x06, 0x04, 0x04, 0x72, 0x30, 0x30, 0x78, 0x30, 0x16, 0x04, 0x02, 0x63, 0x6e, 0x31, 0x10, 0x04, 0x04, 0x72, 0x30, 0x30, 0x58, 0x04, 0x08, 0x54, 0x68, 0x65, 0x20, 0x72, 0x30, 0x30, 0x58, 0x30, 0x1c, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x0d, 0x04, 0x06, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x04, 0x03, 0x74, 0x6f, 0x70,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(21),
				protocolOp: AddRequest{
					entry: LDAPDN("cn=r00X+commonName=The r00X+description=A test user,ou=consumers,ou=system"),
					attributes: AttributeList{
						Attribute{
							type_: AttributeDescription("description"),
							vals: []AttributeValue{
								AttributeValue("A test user"),
							},
						},
						Attribute{
							type_: AttributeDescription("sn"),
							vals: []AttributeValue{
								AttributeValue("r00x"),
							},
						},
						Attribute{
							type_: AttributeDescription("cn"),
							vals: []AttributeValue{
								AttributeValue("r00X"),
								AttributeValue("The r00X"),
							},
						},
						Attribute{
							type_: AttributeDescription("objectClass"),
							vals: []AttributeValue{
								AttributeValue("person"),
								AttributeValue("top"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 73: SERVER AddResponse
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011569070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x15, 0x69, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(21),
				protocolOp: AddResponse{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 74: CLIENT ModifyRequest
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30790201196674044a636e3d723030582b636f6d6d6f6e6e616d653d54686520723030582b6465736372697074696f6e3d41207465737420757365722c6f753d636f6e73756d6572732c6f753d73797374656d302630240a0100301f040f74656c6570686f6e654e756d626572310c040a30313233343536373839
					0x30, 0x79, 0x02, 0x01, 0x19, 0x66, 0x74, 0x04, 0x4a, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x54, 0x68, 0x65, 0x20, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x41, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x30, 0x26, 0x30, 0x24, 0x0a, 0x01, 0x00, 0x30, 0x1f, 0x04, 0x0f, 0x74, 0x65, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x31, 0x0c, 0x04, 0x0a, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(25),
				protocolOp: ModifyRequest{
					object: LDAPDN("cn=r00X+commonname=The r00X+description=A test user,ou=consumers,ou=system"),
					changes: []ModifyRequestChange{
						{
							operation: ENUMERATED(0),
							modification: PartialAttribute{
								type_: AttributeDescription("telephoneNumber"),
								vals: []AttributeValue{
									AttributeValue("0123456789"),
								},
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 75: SERVER ModifyResponse
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011967070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x19, 0x67, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(25),
				protocolOp: ModifyResponse{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 76: CLIENT ModifyDNrequest
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 307502011b6c70044a636e3d723030582b636f6d6d6f6e6e616d653d54686520723030582b6465736372697074696f6e3d41207465737420757365722c6f753d636f6e73756d6572732c6f753d73797374656d0407636e3d723030580101ff80166f753d636f6e73756d6572732c6f753d73797374656d
					0x30, 0x75, 0x02, 0x01, 0x1b, 0x6c, 0x70, 0x04, 0x4a, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x54, 0x68, 0x65, 0x20, 0x72, 0x30, 0x30, 0x58, 0x2b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x41, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x72, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x04, 0x07, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x01, 0x01, 0xff, 0x80, 0x16, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(27),
				protocolOp: ModifyDNRequest{
					entry:        LDAPDN("cn=r00X+commonname=The r00X+description=A test user,ou=consumers,ou=system"),
					newrdn:       RelativeLDAPDN("cn=r00X"),
					deleteoldrdn: BOOLEAN(true),
					newSuperior:  LDAPDN("ou=consumers,ou=system").Pointer(),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 77: SERVER ModifyDNResponse
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02011b6d070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x1b, 0x6d, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(27),
				protocolOp: ModifyDNResponse{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 78: CLIENT
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303b020121633604096f753d73797374656d0a01020a0103020203e8020100010100a30a0402636e040472303058300d040b6f626a656374436c617373
					0x30, 0x3b, 0x02, 0x01, 0x21, 0x63, 0x36, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x0a, 0x01, 0x02, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0xa3, 0x0a, 0x04, 0x02, 0x63, 0x6e, 0x04, 0x04, 0x72, 0x30, 0x30, 0x58, 0x30, 0x0d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(33),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=system"),
					scope:        ENUMERATED(2),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter: FilterEqualityMatch{
						attributeDesc:  AttributeDescription("cn"),
						assertionValue: AssertionValue("r00X"),
					},
					attributes: AttributeSelection{
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 79: SERVER
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30450201216440041e636e3d723030582c6f753d636f6e73756d6572732c6f753d73797374656d301e301c040b6f626a656374436c617373310d0403746f700406706572736f6e
					0x30, 0x45, 0x02, 0x01, 0x21, 0x64, 0x40, 0x04, 0x1e, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x30, 0x1e, 0x30, 0x1c, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x0d, 0x04, 0x03, 0x74, 0x6f, 0x70, 0x04, 0x06, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(33),
				protocolOp: SearchResultEntry{
					objectName: LDAPDN("cn=r00X,ou=consumers,ou=system"),
					attributes: PartialAttributeList{
						PartialAttribute{
							type_: AttributeDescription("objectClass"),
							vals: []AttributeValue{
								AttributeValue("top"),
								AttributeValue("person"),
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 80: SERVER
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c02012165070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x21, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(33),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 81: CLIENT SearchRequest with controls
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30819c020124633704096f753d73797374656d0a01020a0103020203e8020100010100870b6f626a656374436c617373300d040b6f626a656374436c617373a05e301e0417312e332e362e312e342e312e343230332e312e31302e3104030101ff30190417322e31362e3834302e312e3131333733302e332e342e3230210416312e322e3834302e3131333535362e312e342e333139040730050201030400
					0x30, 0x81, 0x9c, 0x02, 0x01, 0x24, 0x63, 0x37, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x0a, 0x01, 0x02, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0x87, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x30, 0x0d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0xa0, 0x5e, 0x30, 0x1e, 0x04, 0x17, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x32, 0x30, 0x33, 0x2e, 0x31, 0x2e, 0x31, 0x30, 0x2e, 0x31, 0x04, 0x03, 0x01, 0x01, 0xff, 0x30, 0x19, 0x04, 0x17, 0x32, 0x2e, 0x31, 0x36, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x2e, 0x31, 0x31, 0x33, 0x37, 0x33, 0x30, 0x2e, 0x33, 0x2e, 0x34, 0x2e, 0x32, 0x30, 0x21, 0x04, 0x16, 0x31, 0x2e, 0x32, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x31, 0x33, 0x35, 0x35, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x33, 0x31, 0x39, 0x04, 0x07, 0x30, 0x05, 0x02, 0x01, 0x03, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(36),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=system"),
					scope:        ENUMERATED(2),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter:       FilterPresent("objectClass"),
					attributes: AttributeSelection{
						LDAPString("objectClass"),
					},
				},
				controls: &Controls{
					Control{
						controlType:  LDAPOID("1.3.6.1.4.1.4203.1.10.1"),
						criticality:  BOOLEAN(false),
						controlValue: OCTETSTRING("\x01\x01\xff").Pointer(),
					},
					Control{
						controlType:  LDAPOID("2.16.840.1.113730.3.4.2"),
						criticality:  BOOLEAN(false),
						controlValue: (*OCTETSTRING)(nil),
					},
					Control{
						controlType:  LDAPOID("1.2.840.113556.1.4.319"),
						criticality:  BOOLEAN(false),
						controlValue: OCTETSTRING("0\x05\x02\x01\x03\x04\x00").Pointer(),
					},
				},
			},
		},

		// Request 82: SERVER SearchResultDone with Controls
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 303402012465070a010004000400a02630240416312e322e3834302e3131333535362e312e342e3331390101ff040730050201000400
					0x30, 0x34, 0x02, 0x01, 0x24, 0x65, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00, 0xa0, 0x26, 0x30, 0x24, 0x04, 0x16, 0x31, 0x2e, 0x32, 0x2e, 0x38, 0x34, 0x30, 0x2e, 0x31, 0x31, 0x33, 0x35, 0x35, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x33, 0x31, 0x39, 0x01, 0x01, 0xff, 0x04, 0x07, 0x30, 0x05, 0x02, 0x01, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(36),
				protocolOp: SearchResultDone{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: &Controls{
					Control{
						controlType:  LDAPOID("1.2.840.113556.1.4.319"),
						criticality:  BOOLEAN(true),
						controlValue: OCTETSTRING("0\x05\x02\x01\x00\x04\x00").Pointer(),
					},
				},
			},
		},

		// (|
		//     (&
		//         (cn=r00x)
		//         (telephoneNumber=*)
		//     )
		//     (cn~=The)
		//     (&
		//         (!(description=Toto))
		//         (ou=co*f*g*r*on)
		//     )
		// )
		// Request 83: CLIENT
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30818e02010d63818804096f753d73797374656d0a01020a0103020203e8020100010100a15ca01da30a0402636e040472303078870f74656c6570686f6e654e756d626572a8090402636e0403546865a030a215a313040b6465736372697074696f6e0404546f746fa41704026f7530118002636f81016681016781017282026f6e300d040b6f626a656374436c617373
					// 0x30, 0x81, 0x8e, 0x02, 0x01, 0x0d, 0x63, 0x81, 0x88, 0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x0a, 0x01, 0x02, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0xa1, 0x5c, 0xa0, 0x1d, 0xa3, 0x0a, 0x04, 0x02, 0x63, 0x6e, 0x04, 0x04, 0x72, 0x30, 0x30, 0x78, 0x87, 0x0f, 0x74, 0x65, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0xa8, 0x09, 0x04, 0x02, 0x63, 0x6e, 0x04, 0x03, 0x54, 0x68, 0x65, 0xa0, 0x30, 0xa2, 0x15, 0xa3, 0x13, 0x04, 0x0b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x04, 0x04, 0x54, 0x6f, 0x74, 0x6f, 0xa4, 0x17, 0x04, 0x02, 0x6f, 0x75, 0x30, 0x11, 0x80, 0x02, 0x63, 0x6f, 0x81, 0x01, 0x66, 0x81, 0x01, 0x67, 0x81, 0x01, 0x72, 0x82, 0x02, 0x6f, 0x6e, 0x30, 0x0d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
					0x30, 0x81, 0x8e,
					// messageID
					0x02, 0x01, 0x0d,
					// protocolOp
					0x63, 0x81, 0x88,
					// baseObject
					0x04, 0x09, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
					// scope
					0x0a, 0x01, 0x02,
					// derefAliases
					0x0a, 0x01, 0x03,
					// sizeLimit
					0x02, 0x02, 0x03, 0xe8,
					// timeLimit
					0x02, 0x01, 0x00,
					// typesOnly
					0x01, 0x01, 0x00,
					// filter
					// filterOr [1]
					0xa1, 0x5c,
					// 		filterAnd [0]
					0xa0, 0x1d,
					//			filterEqualityMatch [3]
					0xa3, 0x0a,
					//				cn
					0x04, 0x02, 0x63, 0x6e,
					//				r00x
					0x04, 0x04, 0x72, 0x30, 0x30, 0x78,
					//			filterPresent [7]
					//				telephoneNumber
					0x87, 0x0f, 0x74, 0x65, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72,
					//		filterApproxMatch
					0xa8, 0x09,
					//			cn
					0x04, 0x02, 0x63, 0x6e,
					//			the
					0x04, 0x03, 0x54, 0x68, 0x65,
					//		filterAnd [0]
					0xa0, 0x30,
					//			filterNot [2]
					0xa2, 0x15,
					//				FilterEqualityMatch [3]
					0xa3, 0x13,
					//					description
					0x04, 0x0b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e,
					//					Toto
					0x04, 0x04, 0x54, 0x6f, 0x74, 0x6f,
					//			filterSubstrings
					0xa4, 0x17,
					//				type = ou
					0x04, 0x02, 0x6f, 0x75,
					//				substrings
					0x30, 0x11,
					//					substringInitial = co
					0x80, 0x02, 0x63, 0x6f,
					//					substringAny = f
					0x81, 0x01, 0x66,
					//					substringAny = g
					0x81, 0x01, 0x67,
					//					substringAny = r
					0x81, 0x01, 0x72,
					//					substringFinal = on
					0x82, 0x02, 0x6f, 0x6e,
					// attributes
					0x30, 0x0d,
					// AttributeSelection = objectClass
					0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(13),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=system"),
					scope:        ENUMERATED(2),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter: FilterOr{
						FilterAnd{
							FilterEqualityMatch{
								attributeDesc:  AttributeDescription("cn"),
								assertionValue: AssertionValue("r00x"),
							},
							FilterPresent("telephoneNumber"),
						},
						FilterApproxMatch{
							attributeDesc:  AttributeDescription("cn"),
							assertionValue: AssertionValue("The"),
						},
						FilterAnd{
							FilterNot{
								Filter: FilterEqualityMatch{
									attributeDesc:  AttributeDescription("description"),
									assertionValue: AssertionValue("Toto"),
								},
							},
							FilterSubstrings{
								type_: AttributeDescription("ou"),
								substrings: []Substring{
									SubstringInitial("co"),
									SubstringAny("f"),
									SubstringAny("g"),
									SubstringAny("r"),
									SubstringFinal("on"),
								},
							},
						},
					},
					attributes: AttributeSelection{
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 84: CLIENT SearchRequest with FilterGreaterOrEqual and FilterLessOrEqual
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3072020119636d04000a01020a0103020203e8020100010100a04aa523041e6164732d636867507764506f6c69637950617373776f72644c656e677468040133a623041e6164732d636867507764506f6c69637950617373776f72644c656e677468040135300d040b6f626a656374436c617373
					0x30, 0x72, 0x02, 0x01, 0x19, 0x63, 0x6d, 0x04, 0x00, 0x0a, 0x01, 0x02, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0xa0, 0x4a, 0xa5, 0x23, 0x04, 0x1e, 0x61, 0x64, 0x73, 0x2d, 0x63, 0x68, 0x67, 0x50, 0x77, 0x64, 0x50, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x04, 0x01, 0x33, 0xa6, 0x23, 0x04, 0x1e, 0x61, 0x64, 0x73, 0x2d, 0x63, 0x68, 0x67, 0x50, 0x77, 0x64, 0x50, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x04, 0x01, 0x35, 0x30, 0x0d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(25),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN(""),
					scope:        ENUMERATED(2),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter: FilterAnd{
						FilterGreaterOrEqual{
							attributeDesc:  AttributeDescription("ads-chgPwdPolicyPasswordLength"),
							assertionValue: AssertionValue("3"),
						},
						FilterLessOrEqual{
							attributeDesc:  AttributeDescription("ads-chgPwdPolicyPasswordLength"),
							assertionValue: AssertionValue("5"),
						},
					},
					attributes: AttributeSelection{
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 85: CLIENT SearchRequest with FilterExtensibleMatch
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 3074020131636f04166f753d636f6e73756d6572732c6f753d73797374656d0a01020a0103020203e8020100010100a936811474656c6570686f6e654e756d6265724d61746368820f74656c6570686f6e654e756d626572830a303132333435363738398401ff300d040b6f626a656374436c617373
					0x30, 0x74, 0x02, 0x01, 0x31, 0x63, 0x6f, 0x04, 0x16, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x0a, 0x01, 0x02, 0x0a, 0x01, 0x03, 0x02, 0x02, 0x03, 0xe8, 0x02, 0x01, 0x00, 0x01, 0x01, 0x00, 0xa9, 0x36, 0x81, 0x14, 0x74, 0x65, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x4d, 0x61, 0x74, 0x63, 0x68, 0x82, 0x0f, 0x74, 0x65, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x83, 0x0a, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x84, 0x01, 0xff, 0x30, 0x0d, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(49),
				protocolOp: SearchRequest{
					baseObject:   LDAPDN("ou=consumers,ou=system"),
					scope:        ENUMERATED(2),
					derefAliases: ENUMERATED(3),
					sizeLimit:    INTEGER(1000),
					timeLimit:    INTEGER(0),
					typesOnly:    BOOLEAN(false),
					filter: FilterExtensibleMatch{
						matchingRule: MatchingRuleId("telephoneNumberMatch").Pointer(),
						type_:        AttributeDescription("telephoneNumber").Pointer(),
						matchValue:   AssertionValue("0123456789"),
						dnAttributes: BOOLEAN(true),
					},
					attributes: AttributeSelection{
						LDAPString("objectClass"),
					},
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 86: CLIENT
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 30400201274a3b636e3d4120636f6d706c657820706572736f6e5c2c207665727920636f6d706c657820212c6f753d636f6e73756d6572732c6f753d73797374656d
					0x30, 0x40, 0x02, 0x01, 0x27, 0x4a, 0x3b, 0x63, 0x6e, 0x3d, 0x41, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x78, 0x20, 0x70, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x5c, 0x2c, 0x20, 0x76, 0x65, 0x72, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x78, 0x20, 0x21, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
				},
			},
			out: LDAPMessage{
				messageID:  MessageID(39),
				protocolOp: DelRequest("cn=A complex person\\, very complex !,ou=consumers,ou=system"),
				controls:   (*Controls)(nil),
			},
		},

		// Request 87: SERVER
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 300c0201276b070a010004000400
					0x30, 0x0c, 0x02, 0x01, 0x27, 0x6b, 0x07, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(39),
				protocolOp: DelResponse{
					resultCode:        ENUMERATED(0),
					matchedDN:         LDAPDN(""),
					diagnosticMessage: LDAPString(""),
					referral:          (*Referral)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 88: CLIENT ExtendedRequest: Start TLS (OID 1.3.6.1.4.1.1466.20037)
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 301d02010177188016312e332e362e312e342e312e313436362e3230303337
					0x30, 0x1d, 0x02, 0x01, 0x01, 0x77, 0x18, 0x80, 0x16, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36, 0x2e, 0x32, 0x30, 0x30, 0x33, 0x37,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(1),
				protocolOp: ExtendedRequest{
					requestName:  LDAPOID("1.3.6.1.4.1.1466.20037"),
					requestValue: (*OCTETSTRING)(nil),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 89: SERVER ExtendedResponse: Start TLS (OID 1.3.6.1.4.1.1466.20037)
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 302602010178210a0100040004008a16312e332e362e312e342e312e313436362e32303033378b00
					0x30, 0x26,
					0x02, 0x01, 0x01,
					0x78, 0x21, 0x0a,
					0x01, 0x00,
					0x04, 0x00,
					0x04, 0x00,
					0x8a, 0x16, 0x31, 0x2e, 0x33, 0x2e, 0x36, 0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x31, 0x34, 0x36, 0x36, 0x2e, 0x32, 0x30, 0x30, 0x33, 0x37,
					0x8b, 0x00,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(1),
				protocolOp: ExtendedResponse{
					LDAPResult: LDAPResult{
						resultCode:        ENUMERATED(0),
						matchedDN:         LDAPDN(""),
						diagnosticMessage: LDAPString(""),
						referral:          (*Referral)(nil),
					},
					responseName:  LDAPOID("1.3.6.1.4.1.1466.20037").Pointer(),
					responseValue: OCTETSTRING("").Pointer(),
				},
				controls: (*Controls)(nil),
			},
		},

		// Request 90: A bind request with a simple login / password authentication
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					0x30, 0x1d,
					0x02, 0x01, 0x01, // messageID
					0x60, 0x18, // Application, tag 0 => this is a Bind request
					0x02, 0x01, 0x03, // Version 3
					0x04, 0x07, 0x6d, 0x79, 0x4c, 0x6f, 0x67, 0x69, 0x6e, // login = myLogin
					0x80, 0x0a, 0x6d, 0x79, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, // simple authentication: myPassword
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x01)),
				protocolOp: BindRequest{
					version:        0x03,
					name:           LDAPDN("myLogin"),
					authentication: OCTETSTRING([]byte("myPassword")),
				},
			},
		},
		// Request 91: A bind request with SASL (CRAM-MD5)
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					0x30, 0x16,
					0x02, 0x01, 0x01, // messageID
					0x60, 0x11,
					0x02, 0x01, 0x03, // version 3
					0x04, 0x00, // no login
					0xa3, 0x0a, 0x04, 0x08, 0x43, 0x52, 0x41, 0x4d, 0x2d, 0x4d, 0x44, 0x35, // SASL mechanism "CRAM-MD5", no credentials
				},
			},
			out: LDAPMessage{
				messageID: MessageID(int(0x01)),
				protocolOp: BindRequest{
					version: 0x03,
					name:    LDAPDN(""),
					authentication: SaslCredentials{
						mechanism: LDAPString("CRAM-MD5"),
					},
				},
			},
		},
		// Request 92: An abandon request
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					0x30, 0x06,
					0x02, 0x01, 0x0a, // messageID
					0x50, 0x01, 0x05, // Abandon request [APPLICATION 16] MessageID = 0x05
				},
			},
			out: LDAPMessage{
				messageID:  MessageID(int(0x0a)),
				protocolOp: AbandonRequest(0x05),
				controls:   (*Controls)(nil),
			},
		},
		// Request 93: CLIENT
		{
			bytes: Bytes{
				offset: 0,
				bytes: []byte{
					// 304c0201156647041e636e3d723030582c6f753d636f6e73756d6572732c6f753d73797374656d302530230a0102301e040b6465736372697074696f6e310f040d48656c6c6f2c20e4b896e7958c
					0x30, 0x4c, 0x02, 0x01, 0x15, 0x66, 0x47, 0x04, 0x1e, 0x63, 0x6e, 0x3d, 0x72, 0x30, 0x30, 0x58, 0x2c, 0x6f, 0x75, 0x3d, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x72, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x30, 0x25, 0x30, 0x23, 0x0a, 0x01, 0x02, 0x30, 0x1e, 0x04, 0x0b, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x0f, 0x04, 0x0d, 0x48, 0x65, 0x6c, 0x6c, 0x6f, 0x2c, 0x20, 0xe4, 0xb8, 0x96, 0xe7, 0x95, 0x8c,
				},
			},
			out: LDAPMessage{
				messageID: MessageID(21),
				protocolOp: ModifyRequest{
					object: LDAPDN("cn=r00X,ou=consumers,ou=system"),
					changes: []ModifyRequestChange{
						{
							operation: ENUMERATED(2),
							modification: PartialAttribute{
								type_: AttributeDescription("description"),
								vals: []AttributeValue{
									AttributeValue("Hello, 世界"),
								},
							},
						},
					},
				},
				controls: (*Controls)(nil),
			},
		},
	}
}
