package strutil_test

import (
	"testing"

	"github.com/gookit/goutil/strutil"
	"github.com/gookit/goutil/testutil/assert"
)

func TestIsInt(t *testing.T) {
	// IsInt
	assert.True(t, strutil.IsInt("123"))
	assert.True(t, strutil.IsInt("0"))
	assert.True(t, strutil.IsInt("456789"))
	assert.True(t, strutil.IsInt("-123"))

	assert.False(t, strutil.IsInt("123.45"))
	assert.False(t, strutil.IsInt("abc"))
	assert.False(t, strutil.IsInt("123abc"))
	assert.False(t, strutil.IsInt(""))

	// IsUint
	assert.True(t, strutil.IsUint("123"))
	assert.True(t, strutil.IsUint("0"))
	assert.True(t, strutil.IsUint("456789"))

	assert.False(t, strutil.IsUint("-123"))
	assert.False(t, strutil.IsUint("123.45"))
	assert.False(t, strutil.IsUint("ab3"))
	assert.False(t, strutil.IsUint(""))
}

func TestIsFloat(t *testing.T) {
	assert.True(t, strutil.IsFloat("123.45"))
	assert.True(t, strutil.IsFloat("-123.45"))
	assert.True(t, strutil.IsFloat("+123.45"))
	assert.True(t, strutil.IsFloat("0.123"))
	assert.True(t, strutil.IsFloat(".123"))
	assert.True(t, strutil.IsFloat("123.0"))
	assert.True(t, strutil.IsFloat("0"))

	assert.True(t, strutil.IsFloat("123"))
	assert.False(t, strutil.IsFloat("abc"))
	assert.False(t, strutil.IsFloat("123abc"))
	assert.False(t, strutil.IsFloat(""))
}

func TestIsNumeric(t *testing.T) {
	// IsNumeric
	assert.True(t, strutil.IsNumeric("0"))
	assert.True(t, strutil.IsNumeric("234"))
	assert.True(t, strutil.IsNumeric("-234"))
	assert.True(t, strutil.IsNumeric("23.4"))
	assert.True(t, strutil.IsNumeric("-23.4"))
	assert.False(t, strutil.IsNumeric(""))
	assert.False(t, strutil.IsNumeric("a34"))

	// IsPositiveNum
	assert.True(t, strutil.IsPositiveNum("0.0"))
	assert.True(t, strutil.IsPositiveNum("234"))
	assert.True(t, strutil.IsPositiveNum("23.4"))
	assert.False(t, strutil.IsPositiveNum("-23.4"))
	assert.False(t, strutil.IsPositiveNum("-234"))
	assert.False(t, strutil.IsPositiveNum("a34"))
	assert.False(t, strutil.IsPositiveNum(""))
}

func TestIsAlphabet(t *testing.T) {
	assert.True(t, strutil.IsNumChar('9'))
	assert.False(t, strutil.IsNumChar('A'))

	assert.False(t, strutil.IsAlphabet('9'))
	assert.False(t, strutil.IsAlphabet('+'))

	assert.True(t, strutil.IsAlphabet('A'))
	assert.True(t, strutil.IsAlphabet('a'))
	assert.True(t, strutil.IsAlphabet('Z'))
	assert.True(t, strutil.IsAlphabet('z'))
}

func TestIsAlphaNum(t *testing.T) {
	assert.False(t, strutil.IsAlphaNum('+'))

	assert.True(t, strutil.IsAlphaNum('9'))
	assert.True(t, strutil.IsAlphaNum('A'))
	assert.True(t, strutil.IsAlphaNum('a'))
	assert.True(t, strutil.IsAlphaNum('Z'))
	assert.True(t, strutil.IsAlphaNum('z'))
}

func TestNoCaseEq(t *testing.T) {
	assert.True(t, strutil.Equal("a", "a"))
	assert.True(t, strutil.NoCaseEq("A", "a"))
	assert.True(t, strutil.NoCaseEq("Ab", "aB"))
	assert.False(t, strutil.Equal("a", "b"))
}

func TestIsUpper(t *testing.T) {
	assert.True(t, strutil.IsUpper("ABC"))
	assert.False(t, strutil.IsUpper("aBC"))

	assert.False(t, strutil.IsLower("aBC"))
	assert.True(t, strutil.IsLower("abc"))
}

func TestStrPos(t *testing.T) {
	// StrPos
	assert.Eq(t, -1, strutil.StrPos("xyz", "a"))
	assert.Eq(t, 0, strutil.StrPos("xyz", "x"))
	assert.Eq(t, 2, strutil.StrPos("xyz", "z"))

	// RunePos
	assert.Eq(t, -1, strutil.RunePos("xyz", 'a'))
	assert.Eq(t, 0, strutil.RunePos("xyz", 'x'))
	assert.Eq(t, 2, strutil.RunePos("xyz", 'z'))
	assert.Eq(t, 5, strutil.RunePos("hi时间", '间'))

	// BytePos
	assert.Eq(t, -1, strutil.BytePos("xyz", 'a'))
	assert.Eq(t, 0, strutil.BytePos("xyz", 'x'))
	assert.Eq(t, 2, strutil.BytePos("xyz", 'z'))
	// assert.Eq(t, 2, strutil.BytePos("hi时间", '间')) // will build error
}

func TestIsStartOf(t *testing.T) {
	tests := []struct {
		give string
		sub  string
		want bool
	}{
		{"abc", "a", true},
		{"abc", "d", false},
	}

	for _, item := range tests {
		assert.Eq(t, item.want, strutil.HasPrefix(item.give, item.sub))
		assert.Eq(t, item.want, strutil.IsStartOf(item.give, item.sub))
	}

	assert.True(t, strutil.StartsWith("abc", "a"))

	assert.True(t, strutil.IsStartsOf("abc", []string{"a", "b"}))
	assert.True(t, strutil.StartsWithAny("abc", []string{"a", "c"}))
	assert.False(t, strutil.IsStartsOf("abc", []string{"d", "e"}))
	assert.False(t, strutil.StartsWithAny("abc", []string{"d", "e"}))
}

func TestIsEndOf(t *testing.T) {
	tests := []struct {
		give string
		sub  string
		want bool
	}{
		{"abc", "c", true},
		{"abc", "d", false},
		{"some.json", ".json", true},
	}

	for _, item := range tests {
		assert.Eq(t, item.want, strutil.HasSuffix(item.give, item.sub))
		assert.Eq(t, item.want, strutil.IsEndOf(item.give, item.sub))
	}
}

func TestIsSpace(t *testing.T) {
	assert.True(t, strutil.IsSpace(' '))
	assert.True(t, strutil.IsSpace('\n'))
	assert.True(t, strutil.IsSpaceRune('\n'))
	assert.True(t, strutil.IsSpaceRune('\t'))

	assert.False(t, strutil.IsBlank(" a "))
	assert.True(t, strutil.IsNotBlank(" a "))
	assert.False(t, strutil.IsEmpty(" "))
	assert.True(t, strutil.IsBlank(""))
	assert.True(t, strutil.IsBlank(" "))
	assert.True(t, strutil.IsBlank("   "))
	assert.False(t, strutil.IsNotBlank("   "))

	assert.False(t, strutil.IsBlankBytes([]byte(" a ")))
	assert.True(t, strutil.IsBlankBytes([]byte(" ")))
	assert.True(t, strutil.IsBlankBytes([]byte("   ")))
}

func TestIsSymbol(t *testing.T) {
	assert.False(t, strutil.IsSymbol('a'))
	assert.True(t, strutil.IsSymbol('●'))
}

func TestIsVersion(t *testing.T) {
	assert.False(t, strutil.IsVersion("abc"))
	assert.False(t, strutil.IsVersion(".2"))
	assert.False(t, strutil.IsVersion("a.2"))

	assert.True(t, strutil.IsVersion("0.1"))
	assert.True(t, strutil.IsVersion("0.1.0"))
	assert.True(t, strutil.IsVersion("1.2.0"))
	assert.True(t, strutil.IsVersion("1.2.0-beta"))
	assert.True(t, strutil.IsVersion("1.2.0-beta2"))
	assert.True(t, strutil.IsVersion("1.2.0-alpha1"))
}

func TestIsVarName(t *testing.T) {
	assert.True(t, strutil.IsVarName("a"))
	assert.True(t, strutil.IsVarName("a12"))
	assert.True(t, strutil.IsVarName("abc"))
	assert.True(t, strutil.IsVarName("Abc"))
	assert.True(t, strutil.IsVarName("ab_c"))

	assert.False(t, strutil.IsVarName(" abc"))
	assert.False(t, strutil.IsVarName("+"))
}

func TestIsEnvName(t *testing.T) {
	assert.False(t, strutil.IsEnvName("a"))
	assert.False(t, strutil.IsEnvName("APP_aa"))

	assert.True(t, strutil.IsEnvName("SHELL"))
	assert.True(t, strutil.IsEnvName("APP_EN"))
}

func TestIEqual(t *testing.T) {
	assert.False(t, strutil.IEqual("h3ab2c", "d"))
	assert.False(t, strutil.IEqual("ab", "ac"))
	assert.True(t, strutil.IEqual("ab", "AB"))
	assert.True(t, strutil.IEqual("ab", "Ab"))
	assert.True(t, strutil.IEqual("ab", "ab"))
}

func TestIContains(t *testing.T) {
	assert.False(t, strutil.IContains("h3ab2c", "d"))
	assert.True(t, strutil.IContains("h3ab2c", "AB"))
	assert.True(t, strutil.IContains("H3AB2C", "aB"))

	assert.True(t, strutil.ContainsByte("H3AB2C", 'A'))
	assert.False(t, strutil.ContainsByte("H3AB2C", 'a'))

	assert.True(t, strutil.ContainsByteOne("H3AB2C", []byte("A")))
	assert.True(t, strutil.ContainsByteOne("H3AB2C", []byte("DB")))
	assert.False(t, strutil.ContainsByteOne("H3AB2C", []byte("D")))
}

func TestHasOneSub(t *testing.T) {
	assert.False(t, strutil.ContainsOne("h3ab2c", []string{"d"}))
	assert.False(t, strutil.HasOneSub("h3ab2c", []string{"d"}))
	assert.False(t, strutil.HasOneSub("h3AB2c", []string{"ab"}))
	assert.True(t, strutil.HasOneSub("h3AB2c", []string{"AB"}))

	// ignore-case
	assert.True(t, strutil.IContainsOne("h3AB2c", []string{"ab"}))
	assert.False(t, strutil.IContainsOne("h3AB2c", []string{"NO"}))
}

func TestHasAllSubs(t *testing.T) {
	assert.False(t, strutil.HasAllSubs("h3ab2c", []string{"a", "d"}))
	assert.False(t, strutil.ContainsAll("h3AB2c", []string{"A", "b"}))
	assert.True(t, strutil.HasAllSubs("h3ab2c", []string{"a", "b"}))
	assert.True(t, strutil.ContainsAll("h3ab2c", []string{"a", "b"}))

	// ignore-case
	assert.True(t, strutil.IContainsAll("h3AB2c", []string{"A", "b"}))
	assert.False(t, strutil.IContainsAll("h3AB2c", []string{"A", "NO"}))
}

func TestCompare(t *testing.T) {
	versions := []struct{ a, b string }{
		{"1.0.221.9289", "1.05.00.0156"},
		// Go versions
		{"1", "1.0.1"},
		{"1.0.1", "1.0.2"},
		{"1.0.2", "1.0.3"},
		{"1.0.3", "1.1"},
		{"1.1", "1.1.1"},
		{"1.1.1", "1.1.2"},
		{"1.1.2", "1.2"},
	}
	for _, version := range versions {
		assert.True(t, strutil.Compare(version.a, version.b, "<"), version.a+"<"+version.b)
		assert.True(t, strutil.Compare(version.a, version.b, "<="), version.a+"<="+version.b)
		assert.True(t, strutil.Compare(version.b, version.a, ">"), version.a+">"+version.b)
		assert.True(t, strutil.Compare(version.b, version.a, ">="), version.a+">="+version.b)
	}

	assert.True(t, strutil.Compare("1.0", "1.0", ""))
	assert.True(t, strutil.Compare("1.0", "1.0", "="))

	assert.True(t, strutil.Compare("1.0", "2.0", "!="))
	assert.False(t, strutil.Compare("2020-12-16", "2021-12-17", ">="))

	// diff with VersionCompare
	assert.False(t, strutil.Compare("1.11", "1.2", ">"))
}

func TestVersionCompare(t *testing.T) {
	versions := []struct{ a, b string }{
		{"1.0.221.9289", "1.05.00.0156"},
		// Go versions
		{"1", "1.0.1"},
		{"1.0.1", "1.0.2"},
		{"1.0.2", "1.0.3"},
		{"1.0.3", "1.1"},
		{"1.1", "1.1.1"},
		{"1.1.1", "1.1.2"},
		{"1.1.2", "1.2"},
	}
	for _, version := range versions {
		assert.True(t, strutil.VersionCompare(version.a, version.b, "<"), version.a+"<"+version.b)
		assert.True(t, strutil.VersionCompare(version.a, version.b, "<="), version.a+"<="+version.b)
		assert.True(t, strutil.VersionCompare(version.b, version.a, ">"), version.a+">"+version.b)
		assert.True(t, strutil.VersionCompare(version.b, version.a, ">="), version.a+">="+version.b)
	}

	assert.True(t, strutil.VersionCompare("1.0", "1.0", "="))
	assert.True(t, strutil.VersionCompare("1.0", "2.0", "!="))
	assert.False(t, strutil.VersionCompare("1.0", "1.0", ""))

	assert.True(t, strutil.VersionCompare("1.11", "1.2", ">"))
	assert.True(t, strutil.VersionCompare("1.11.3", "1.2.34", ">"))
}

func TestGlobMatch(t *testing.T) {
	tests := []struct {
		p, s string
		want bool
	}{
		{"a*", "abc", true},
		{"a*", "ab.cd.ef", true},
		{"ab.*.ef", "ab.cd.ef", true},
		{"ab.*.ef", "ab.cd.efg", false},
		{"ab.*.*", "ab.cd.ef", true},
		{"ab.cd.*", "ab.cd.ef", true},
		{"ab.*", "ab.cd.ef", true},
		{"a*/b", "acd/b", true},
		// {"a*/b", "a/c/b", false},
		// {"a*", "a/c/b", false},
		// {"a**", "a/c/b", false},
	}

	for _, tt := range tests {
		assert.Eq(t, tt.want, strutil.GlobMatch(tt.p, tt.s), "case %v", tt)
	}

	assert.True(t, strutil.QuickMatch("ab", "abc"))
	assert.True(t, strutil.QuickMatch("abc", "abc"))
	assert.False(t, strutil.GlobMatch("ab", "abc"))
	assert.True(t, strutil.QuickMatch("ab*", "abc"))
	assert.True(t, strutil.PathMatch("ab*", "abc"))
}

func TestPathMatch_GlobMatch_diff(t *testing.T) {
	// different of the PathMatch and GlobMatch
	assert.True(t, strutil.GlobMatch("a*/b", "a/c/b"))
	assert.False(t, strutil.PathMatch("a*/b", "a/c/b"))

	assert.True(t, strutil.GlobMatch("a*", "a/c/b"))
	assert.False(t, strutil.PathMatch("a*", "a/c/b"))

	assert.True(t, strutil.GlobMatch("a**", "a/c/b"))
	assert.False(t, strutil.PathMatch("a**", "a/c/b"))
}

func TestMatchNodePath(t *testing.T) {
	tests := []struct {
		p, s string
		want bool
	}{
		{"a*", "abc", true},
		{"ab.*.ef", "ab.cd.ef", true},
		{"ab.*.*", "ab.cd.ef", true},
		{"ab.cd.*", "ab.cd.ef", true},
		{"a*.b", "acd.b", true},
		{"a**", "a.c.b", true},
		{"ab", "abc", false},
		{"a*", "ab.cd.ef", false},
		{"ab.*.ef", "ab.cd.efg", false},
		{"ab.*", "ab.cd.ef", false},
		{"a*.b", "a.c.b", false},
		{"a*", "a.c.b", false},
	}

	for i, tt := range tests {
		assert.Eq(t, tt.want, strutil.MatchNodePath(tt.p, tt.s, "."), "case#%d %v", i, tt)
	}
}

func TestHasEmpty(t *testing.T) {
	assert.False(t, strutil.HasEmpty("ab", "cd", "ef"))
	assert.True(t, strutil.HasEmpty("ab", "", "ef"))
	assert.False(t, strutil.IsAllEmpty("ab", "", "ef"))
	assert.True(t, strutil.IsAllEmpty("", ""))
}

func TestSimpleMatch(t *testing.T) {
	str := "hi inhere, age is 120"
	assert.True(t, strutil.SimpleMatch(str, []string{"hi", "inhere"}))
	assert.True(t, strutil.SimpleMatch(str, []string{"hi", "inhere", "120$"}))
	assert.False(t, strutil.SimpleMatch(str, []string{"hi", "^inhere"}))
	assert.False(t, strutil.SimpleMatch(str, []string{"hi", "inhere$"}))
}

func TestLikeMatch(t *testing.T) {
	tests := []struct {
		p, s string
		ok   bool
	}{
		{"a%", "abc", true},
		{"%a%", "abc", true},
		{"a%", "ab.cd.ef", true},
		{"%c", "abc", true},
		{"%c", "cdf", false},
		{"%c%", "cdf", true},
		{"%cd%", "cdf", true},
		{"%d%", "cdf", true},
		{"%df%", "cdf", true},
		{"%c", "", false},
		{"abc", "abc", true},
		{"abc", "def", false},
	}

	for _, tt := range tests {
		assert.Eq(t, tt.ok, strutil.LikeMatch(tt.p, tt.s))
	}

}
