// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package chat provides access to the Google Chat API.
//
// For product documentation, see: https://developers.google.com/workspace/chat
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/chat/v1"
//	...
//	ctx := context.Background()
//	chatService, err := chat.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	chatService, err := chat.NewService(ctx, option.WithScopes(chat.ChatUsersSpacesettingsScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	chatService, err := chat.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	chatService, err := chat.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package chat // import "google.golang.org/api/chat/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "chat:v1"
const apiName = "chat"
const apiVersion = "v1"
const basePath = "https://chat.googleapis.com/"
const basePathTemplate = "https://chat.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://chat.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Delete conversations and spaces owned by your organization and remove access
	// to associated files in Google Chat
	ChatAdminDeleteScope = "https://www.googleapis.com/auth/chat.admin.delete"

	// View, add, update and remove members and managers in conversations owned by
	// your organization
	ChatAdminMembershipsScope = "https://www.googleapis.com/auth/chat.admin.memberships"

	// View members and managers in conversations owned by your organization
	ChatAdminMembershipsReadonlyScope = "https://www.googleapis.com/auth/chat.admin.memberships.readonly"

	// View or edit display name, description, and other metadata for all Google
	// Chat conversations owned by your organization
	ChatAdminSpacesScope = "https://www.googleapis.com/auth/chat.admin.spaces"

	// View display name, description, and other metadata for all Google Chat
	// conversations owned by your organization
	ChatAdminSpacesReadonlyScope = "https://www.googleapis.com/auth/chat.admin.spaces.readonly"

	// On their own behalf, apps in Google Chat can delete conversations and spaces
	// and remove access to associated files
	ChatAppDeleteScope = "https://www.googleapis.com/auth/chat.app.delete"

	// On their own behalf, apps in Google Chat can see, add, update, and remove
	// members from conversations and spaces
	ChatAppMembershipsScope = "https://www.googleapis.com/auth/chat.app.memberships"

	// On their own behalf, apps in Google Chat can see all messages and their
	// associated reactions and message content
	ChatAppMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.app.messages.readonly"

	// On their own behalf, apps in Google Chat can create conversations and spaces
	// and see or update their metadata (including history settings and access
	// settings)
	ChatAppSpacesScope = "https://www.googleapis.com/auth/chat.app.spaces"

	// On their own behalf, apps in Google Chat can create conversations and spaces
	ChatAppSpacesCreateScope = "https://www.googleapis.com/auth/chat.app.spaces.create"

	// Private Service: https://www.googleapis.com/auth/chat.bot
	ChatBotScope = "https://www.googleapis.com/auth/chat.bot"

	// View, create, and delete custom emoji in Google Chat
	ChatCustomemojisScope = "https://www.googleapis.com/auth/chat.customemojis"

	// View custom emoji in Google Chat
	ChatCustomemojisReadonlyScope = "https://www.googleapis.com/auth/chat.customemojis.readonly"

	// Delete conversations and spaces and remove access to associated files in
	// Google Chat
	ChatDeleteScope = "https://www.googleapis.com/auth/chat.delete"

	// Import spaces, messages, and memberships into Google Chat.
	ChatImportScope = "https://www.googleapis.com/auth/chat.import"

	// See, add, update, and remove members from conversations and spaces in Google
	// Chat
	ChatMembershipsScope = "https://www.googleapis.com/auth/chat.memberships"

	// Add and remove itself from conversations and spaces in Google Chat
	ChatMembershipsAppScope = "https://www.googleapis.com/auth/chat.memberships.app"

	// View members in Google Chat conversations.
	ChatMembershipsReadonlyScope = "https://www.googleapis.com/auth/chat.memberships.readonly"

	// See, compose, send, update, and delete messages as well as their message
	// content; add, see, and delete reactions to messages.
	ChatMessagesScope = "https://www.googleapis.com/auth/chat.messages"

	// Compose and send messages in Google Chat
	ChatMessagesCreateScope = "https://www.googleapis.com/auth/chat.messages.create"

	// See, add, and delete reactions as well as their reaction content to messages
	// in Google Chat
	ChatMessagesReactionsScope = "https://www.googleapis.com/auth/chat.messages.reactions"

	// Add reactions to messages in Google Chat
	ChatMessagesReactionsCreateScope = "https://www.googleapis.com/auth/chat.messages.reactions.create"

	// View reactions as well as their reaction content to messages in Google Chat
	ChatMessagesReactionsReadonlyScope = "https://www.googleapis.com/auth/chat.messages.reactions.readonly"

	// See messages as well as their reactions and message content in Google Chat
	ChatMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.messages.readonly"

	// Create conversations and spaces and see or update metadata (including
	// history settings and access settings) in Google Chat
	ChatSpacesScope = "https://www.googleapis.com/auth/chat.spaces"

	// Create new conversations and spaces in Google Chat
	ChatSpacesCreateScope = "https://www.googleapis.com/auth/chat.spaces.create"

	// View chat and spaces in Google Chat
	ChatSpacesReadonlyScope = "https://www.googleapis.com/auth/chat.spaces.readonly"

	// View and modify last read time for Google Chat conversations
	ChatUsersReadstateScope = "https://www.googleapis.com/auth/chat.users.readstate"

	// View last read time for Google Chat conversations
	ChatUsersReadstateReadonlyScope = "https://www.googleapis.com/auth/chat.users.readstate.readonly"

	// Read and update your space settings
	ChatUsersSpacesettingsScope = "https://www.googleapis.com/auth/chat.users.spacesettings"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chat.admin.delete",
		"https://www.googleapis.com/auth/chat.admin.memberships",
		"https://www.googleapis.com/auth/chat.admin.memberships.readonly",
		"https://www.googleapis.com/auth/chat.admin.spaces",
		"https://www.googleapis.com/auth/chat.admin.spaces.readonly",
		"https://www.googleapis.com/auth/chat.app.delete",
		"https://www.googleapis.com/auth/chat.app.memberships",
		"https://www.googleapis.com/auth/chat.app.messages.readonly",
		"https://www.googleapis.com/auth/chat.app.spaces",
		"https://www.googleapis.com/auth/chat.app.spaces.create",
		"https://www.googleapis.com/auth/chat.bot",
		"https://www.googleapis.com/auth/chat.customemojis",
		"https://www.googleapis.com/auth/chat.customemojis.readonly",
		"https://www.googleapis.com/auth/chat.delete",
		"https://www.googleapis.com/auth/chat.import",
		"https://www.googleapis.com/auth/chat.memberships",
		"https://www.googleapis.com/auth/chat.memberships.app",
		"https://www.googleapis.com/auth/chat.memberships.readonly",
		"https://www.googleapis.com/auth/chat.messages",
		"https://www.googleapis.com/auth/chat.messages.create",
		"https://www.googleapis.com/auth/chat.messages.reactions",
		"https://www.googleapis.com/auth/chat.messages.reactions.create",
		"https://www.googleapis.com/auth/chat.messages.reactions.readonly",
		"https://www.googleapis.com/auth/chat.messages.readonly",
		"https://www.googleapis.com/auth/chat.spaces",
		"https://www.googleapis.com/auth/chat.spaces.create",
		"https://www.googleapis.com/auth/chat.spaces.readonly",
		"https://www.googleapis.com/auth/chat.users.readstate",
		"https://www.googleapis.com/auth/chat.users.readstate.readonly",
		"https://www.googleapis.com/auth/chat.users.spacesettings",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.CustomEmojis = NewCustomEmojisService(s)
	s.Media = NewMediaService(s)
	s.Spaces = NewSpacesService(s)
	s.Users = NewUsersService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	CustomEmojis *CustomEmojisService

	Media *MediaService

	Spaces *SpacesService

	Users *UsersService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewCustomEmojisService(s *Service) *CustomEmojisService {
	rs := &CustomEmojisService{s: s}
	return rs
}

type CustomEmojisService struct {
	s *Service
}

func NewMediaService(s *Service) *MediaService {
	rs := &MediaService{s: s}
	return rs
}

type MediaService struct {
	s *Service
}

func NewSpacesService(s *Service) *SpacesService {
	rs := &SpacesService{s: s}
	rs.Members = NewSpacesMembersService(s)
	rs.Messages = NewSpacesMessagesService(s)
	rs.SpaceEvents = NewSpacesSpaceEventsService(s)
	return rs
}

type SpacesService struct {
	s *Service

	Members *SpacesMembersService

	Messages *SpacesMessagesService

	SpaceEvents *SpacesSpaceEventsService
}

func NewSpacesMembersService(s *Service) *SpacesMembersService {
	rs := &SpacesMembersService{s: s}
	return rs
}

type SpacesMembersService struct {
	s *Service
}

func NewSpacesMessagesService(s *Service) *SpacesMessagesService {
	rs := &SpacesMessagesService{s: s}
	rs.Attachments = NewSpacesMessagesAttachmentsService(s)
	rs.Reactions = NewSpacesMessagesReactionsService(s)
	return rs
}

type SpacesMessagesService struct {
	s *Service

	Attachments *SpacesMessagesAttachmentsService

	Reactions *SpacesMessagesReactionsService
}

func NewSpacesMessagesAttachmentsService(s *Service) *SpacesMessagesAttachmentsService {
	rs := &SpacesMessagesAttachmentsService{s: s}
	return rs
}

type SpacesMessagesAttachmentsService struct {
	s *Service
}

func NewSpacesMessagesReactionsService(s *Service) *SpacesMessagesReactionsService {
	rs := &SpacesMessagesReactionsService{s: s}
	return rs
}

type SpacesMessagesReactionsService struct {
	s *Service
}

func NewSpacesSpaceEventsService(s *Service) *SpacesSpaceEventsService {
	rs := &SpacesSpaceEventsService{s: s}
	return rs
}

type SpacesSpaceEventsService struct {
	s *Service
}

func NewUsersService(s *Service) *UsersService {
	rs := &UsersService{s: s}
	rs.Spaces = NewUsersSpacesService(s)
	return rs
}

type UsersService struct {
	s *Service

	Spaces *UsersSpacesService
}

func NewUsersSpacesService(s *Service) *UsersSpacesService {
	rs := &UsersSpacesService{s: s}
	rs.SpaceNotificationSetting = NewUsersSpacesSpaceNotificationSettingService(s)
	rs.Threads = NewUsersSpacesThreadsService(s)
	return rs
}

type UsersSpacesService struct {
	s *Service

	SpaceNotificationSetting *UsersSpacesSpaceNotificationSettingService

	Threads *UsersSpacesThreadsService
}

func NewUsersSpacesSpaceNotificationSettingService(s *Service) *UsersSpacesSpaceNotificationSettingService {
	rs := &UsersSpacesSpaceNotificationSettingService{s: s}
	return rs
}

type UsersSpacesSpaceNotificationSettingService struct {
	s *Service
}

func NewUsersSpacesThreadsService(s *Service) *UsersSpacesThreadsService {
	rs := &UsersSpacesThreadsService{s: s}
	return rs
}

type UsersSpacesThreadsService struct {
	s *Service
}

// AccessSettings: Represents the access setting
// (https://support.google.com/chat/answer/11971020) of the space.
type AccessSettings struct {
	// AccessState: Output only. Indicates the access state of the space.
	//
	// Possible values:
	//   "ACCESS_STATE_UNSPECIFIED" - Access state is unknown or not supported in
	// this API.
	//   "PRIVATE" - Only users or Google Groups that have been individually added
	// or invited by other users or Google Workspace administrators can discover
	// and access the space.
	//   "DISCOVERABLE" - A space manager has granted a target audience access to
	// the space. Users or Google Groups that have been individually added or
	// invited to the space can also discover and access the space. To learn more,
	// see [Make a space discoverable to specific
	// users](https://developers.google.com/workspace/chat/space-target-audience).
	// Creating discoverable spaces requires [user
	// authentication](https://developers.google.com/workspace/chat/authenticate-aut
	// horize-chat-user).
	AccessState string `json:"accessState,omitempty"`
	// Audience: Optional. The resource name of the target audience
	// (https://support.google.com/a/answer/9934697) who can discover the space,
	// join the space, and preview the messages in the space. If unset, only users
	// or Google Groups who have been individually invited or added to the space
	// can access it. For details, see Make a space discoverable to a target
	// audience
	// (https://developers.google.com/workspace/chat/space-target-audience).
	// Format: `audiences/{audience}` To use the default target audience for the
	// Google Workspace organization, set to `audiences/default`. Reading the
	// target audience supports: - User authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
	// - App authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
	// with administrator approval (https://support.google.com/a?p=chat-app-auth)
	// with the `chat.app.spaces` scope. This field is not populated when using the
	// `chat.bot` scope with app authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
	// Setting the target audience requires user authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
	Audience string `json:"audience,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessState") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessState") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AccessSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessoryWidget: One or more interactive widgets that appear at the bottom
// of a message. For details, see Add interactive widgets at the bottom of a
// message
// (https://developers.google.com/workspace/chat/create-messages#add-accessory-widgets).
type AccessoryWidget struct {
	// ButtonList: A list of buttons.
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ButtonList") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessoryWidget) MarshalJSON() ([]byte, error) {
	type NoMethod AccessoryWidget
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ActionParameter: List of string parameters to supply when the action method
// is invoked. For example, consider three snooze buttons: snooze now, snooze
// one day, snooze next week. You might use `action method = snooze()`, passing
// the snooze type and snooze time in the list of string parameters.
type ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`
	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Key") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Key") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod ActionParameter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ActionResponse: Parameters that a Chat app can use to configure how its
// response is posted.
type ActionResponse struct {
	// DialogAction: Input only. A response to an interaction event related to a
	// dialog (https://developers.google.com/workspace/chat/dialogs). Must be
	// accompanied by `ResponseType.Dialog`.
	DialogAction *DialogAction `json:"dialogAction,omitempty"`
	// Type: Input only. The type of Chat app response.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default type that's handled as `NEW_MESSAGE`.
	//   "NEW_MESSAGE" - Post as a new message in the topic.
	//   "UPDATE_MESSAGE" - Update the Chat app's message. This is only permitted
	// on a `CARD_CLICKED` event where the message sender type is `BOT`.
	//   "UPDATE_USER_MESSAGE_CARDS" - Update the cards on a user's message. This
	// is only permitted as a response to a `MESSAGE` event with a matched url, or
	// a `CARD_CLICKED` event where the message sender type is `HUMAN`. Text is
	// ignored.
	//   "REQUEST_CONFIG" - Privately ask the user for additional authentication or
	// configuration.
	//   "DIALOG" - Presents a
	// [dialog](https://developers.google.com/workspace/chat/dialogs).
	//   "UPDATE_WIDGET" - Widget text autocomplete options query.
	Type string `json:"type,omitempty"`
	// UpdatedWidget: Input only. The response of the updated widget.
	UpdatedWidget *UpdatedWidget `json:"updatedWidget,omitempty"`
	// Url: Input only. URL for users to authenticate or configure. (Only for
	// `REQUEST_CONFIG` response types.)
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DialogAction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DialogAction") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ActionResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ActionResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ActionStatus: Represents the status for a request to either invoke or submit
// a dialog (https://developers.google.com/workspace/chat/dialogs).
type ActionStatus struct {
	// StatusCode: The status code.
	//
	// Possible values:
	//   "OK" - Not an error; returned on success. HTTP Mapping: 200 OK
	//   "CANCELLED" - The operation was cancelled, typically by the caller. HTTP
	// Mapping: 499 Client Closed Request
	//   "UNKNOWN" - Unknown error. For example, this error may be returned when a
	// `Status` value received from another address space belongs to an error space
	// that is not known in this address space. Also errors raised by APIs that do
	// not return enough error information may be converted to this error. HTTP
	// Mapping: 500 Internal Server Error
	//   "INVALID_ARGUMENT" - The client specified an invalid argument. Note that
	// this differs from `FAILED_PRECONDITION`. `INVALID_ARGUMENT` indicates
	// arguments that are problematic regardless of the state of the system (e.g.,
	// a malformed file name). HTTP Mapping: 400 Bad Request
	//   "DEADLINE_EXCEEDED" - The deadline expired before the operation could
	// complete. For operations that change the state of the system, this error may
	// be returned even if the operation has completed successfully. For example, a
	// successful response from a server could have been delayed long enough for
	// the deadline to expire. HTTP Mapping: 504 Gateway Timeout
	//   "NOT_FOUND" - Some requested entity (e.g., file or directory) was not
	// found. Note to server developers: if a request is denied for an entire class
	// of users, such as gradual feature rollout or undocumented allowlist,
	// `NOT_FOUND` may be used. If a request is denied for some users within a
	// class of users, such as user-based access control, `PERMISSION_DENIED` must
	// be used. HTTP Mapping: 404 Not Found
	//   "ALREADY_EXISTS" - The entity that a client attempted to create (e.g.,
	// file or directory) already exists. HTTP Mapping: 409 Conflict
	//   "PERMISSION_DENIED" - The caller does not have permission to execute the
	// specified operation. `PERMISSION_DENIED` must not be used for rejections
	// caused by exhausting some resource (use `RESOURCE_EXHAUSTED` instead for
	// those errors). `PERMISSION_DENIED` must not be used if the caller can not be
	// identified (use `UNAUTHENTICATED` instead for those errors). This error code
	// does not imply the request is valid or the requested entity exists or
	// satisfies other pre-conditions. HTTP Mapping: 403 Forbidden
	//   "UNAUTHENTICATED" - The request does not have valid authentication
	// credentials for the operation. HTTP Mapping: 401 Unauthorized
	//   "RESOURCE_EXHAUSTED" - Some resource has been exhausted, perhaps a
	// per-user quota, or perhaps the entire file system is out of space. HTTP
	// Mapping: 429 Too Many Requests
	//   "FAILED_PRECONDITION" - The operation was rejected because the system is
	// not in a state required for the operation's execution. For example, the
	// directory to be deleted is non-empty, an rmdir operation is applied to a
	// non-directory, etc. Service implementors can use the following guidelines to
	// decide between `FAILED_PRECONDITION`, `ABORTED`, and `UNAVAILABLE`: (a) Use
	// `UNAVAILABLE` if the client can retry just the failing call. (b) Use
	// `ABORTED` if the client should retry at a higher level. For example, when a
	// client-specified test-and-set fails, indicating the client should restart a
	// read-modify-write sequence. (c) Use `FAILED_PRECONDITION` if the client
	// should not retry until the system state has been explicitly fixed. For
	// example, if an "rmdir" fails because the directory is non-empty,
	// `FAILED_PRECONDITION` should be returned since the client should not retry
	// unless the files are deleted from the directory. HTTP Mapping: 400 Bad
	// Request
	//   "ABORTED" - The operation was aborted, typically due to a concurrency
	// issue such as a sequencer check failure or transaction abort. See the
	// guidelines above for deciding between `FAILED_PRECONDITION`, `ABORTED`, and
	// `UNAVAILABLE`. HTTP Mapping: 409 Conflict
	//   "OUT_OF_RANGE" - The operation was attempted past the valid range. E.g.,
	// seeking or reading past end-of-file. Unlike `INVALID_ARGUMENT`, this error
	// indicates a problem that may be fixed if the system state changes. For
	// example, a 32-bit file system will generate `INVALID_ARGUMENT` if asked to
	// read at an offset that is not in the range [0,2^32-1], but it will generate
	// `OUT_OF_RANGE` if asked to read from an offset past the current file size.
	// There is a fair bit of overlap between `FAILED_PRECONDITION` and
	// `OUT_OF_RANGE`. We recommend using `OUT_OF_RANGE` (the more specific error)
	// when it applies so that callers who are iterating through a space can easily
	// look for an `OUT_OF_RANGE` error to detect when they are done. HTTP Mapping:
	// 400 Bad Request
	//   "UNIMPLEMENTED" - The operation is not implemented or is not
	// supported/enabled in this service. HTTP Mapping: 501 Not Implemented
	//   "INTERNAL" - Internal errors. This means that some invariants expected by
	// the underlying system have been broken. This error code is reserved for
	// serious errors. HTTP Mapping: 500 Internal Server Error
	//   "UNAVAILABLE" - The service is currently unavailable. This is most likely
	// a transient condition, which can be corrected by retrying with a backoff.
	// Note that it is not always safe to retry non-idempotent operations. See the
	// guidelines above for deciding between `FAILED_PRECONDITION`, `ABORTED`, and
	// `UNAVAILABLE`. HTTP Mapping: 503 Service Unavailable
	//   "DATA_LOSS" - Unrecoverable data loss or corruption. HTTP Mapping: 500
	// Internal Server Error
	StatusCode string `json:"statusCode,omitempty"`
	// UserFacingMessage: The message to send users about the status of their
	// request. If unset, a generic message based on the `status_code` is sent.
	UserFacingMessage string `json:"userFacingMessage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "StatusCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "StatusCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ActionStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ActionStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Annotation: Output only. Annotations can be associated with the plain-text
// body of the message or with chips that link to Google Workspace resources
// like Google Docs or Sheets with `start_index` and `length` of 0. To add
// basic formatting to a text message, see Format text messages
// (https://developers.google.com/workspace/chat/format-messages). Example
// plain-text message body: ``` Hello @FooBot how are you!" ``` The
// corresponding annotations metadata: ``` "annotations":[{
// "type":"USER_MENTION", "startIndex":6, "length":7, "userMention": { "user":
// { "name":"users/{user}", "displayName":"FooBot",
// "avatarUrl":"https://goo.gl/aeDtrS", "type":"BOT" }, "type":"MENTION" } }]
// ```
type Annotation struct {
	// CustomEmojiMetadata: The metadata for a custom emoji.
	CustomEmojiMetadata *CustomEmojiMetadata `json:"customEmojiMetadata,omitempty"`
	// Length: Length of the substring in the plain-text message body this
	// annotation corresponds to. If not present, indicates a length of 0.
	Length int64 `json:"length,omitempty"`
	// RichLinkMetadata: The metadata for a rich link.
	RichLinkMetadata *RichLinkMetadata `json:"richLinkMetadata,omitempty"`
	// SlashCommand: The metadata for a slash command.
	SlashCommand *SlashCommandMetadata `json:"slashCommand,omitempty"`
	// StartIndex: Start index (0-based, inclusive) in the plain-text message body
	// this annotation corresponds to.
	StartIndex int64 `json:"startIndex,omitempty"`
	// Type: The type of this annotation.
	//
	// Possible values:
	//   "ANNOTATION_TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "USER_MENTION" - A user is mentioned.
	//   "SLASH_COMMAND" - A slash command is invoked.
	//   "RICH_LINK" - A rich link annotation.
	//   "CUSTOM_EMOJI" - A custom emoji annotation.
	Type string `json:"type,omitempty"`
	// UserMention: The metadata of user mention.
	UserMention *UserMentionMetadata `json:"userMention,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomEmojiMetadata") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomEmojiMetadata") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Annotation) MarshalJSON() ([]byte, error) {
	type NoMethod Annotation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AppCommandMetadata: Metadata about a Chat app command
// (https://developers.google.com/workspace/chat/commands).
type AppCommandMetadata struct {
	// AppCommandId: The ID for the command specified in the Chat API
	// configuration.
	AppCommandId int64 `json:"appCommandId,omitempty"`
	// AppCommandType: The type of Chat app command.
	//
	// Possible values:
	//   "APP_COMMAND_TYPE_UNSPECIFIED" - Default value. Unspecified.
	//   "SLASH_COMMAND" - A slash command. The user sends the command in a Chat
	// message.
	//   "QUICK_COMMAND" - A quick command. The user selects the command from the
	// Chat menu in the message reply area.
	AppCommandType string `json:"appCommandType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AppCommandId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AppCommandId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AppCommandMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod AppCommandMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AttachedGif: A GIF image that's specified by a URL.
type AttachedGif struct {
	// Uri: Output only. The URL that hosts the GIF image.
	Uri string `json:"uri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Uri") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Uri") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AttachedGif) MarshalJSON() ([]byte, error) {
	type NoMethod AttachedGif
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Attachment: An attachment in Google Chat.
type Attachment struct {
	// AttachmentDataRef: Optional. A reference to the attachment data. This field
	// is used to create or update messages with attachments, or with the media API
	// to download the attachment data.
	AttachmentDataRef *AttachmentDataRef `json:"attachmentDataRef,omitempty"`
	// ContentName: Output only. The original file name for the content, not the
	// full path.
	ContentName string `json:"contentName,omitempty"`
	// ContentType: Output only. The content type (MIME type) of the file.
	ContentType string `json:"contentType,omitempty"`
	// DownloadUri: Output only. The download URL which should be used to allow a
	// human user to download the attachment. Chat apps shouldn't use this URL to
	// download attachment content.
	DownloadUri string `json:"downloadUri,omitempty"`
	// DriveDataRef: Output only. A reference to the Google Drive attachment. This
	// field is used with the Google Drive API.
	DriveDataRef *DriveDataRef `json:"driveDataRef,omitempty"`
	// Name: Optional. Resource name of the attachment, in the form
	// `spaces/{space}/messages/{message}/attachments/{attachment}`.
	Name string `json:"name,omitempty"`
	// Source: Output only. The source of the attachment.
	//
	// Possible values:
	//   "SOURCE_UNSPECIFIED" - Reserved.
	//   "DRIVE_FILE" - The file is a Google Drive file.
	//   "UPLOADED_CONTENT" - The file is uploaded to Chat.
	Source string `json:"source,omitempty"`
	// ThumbnailUri: Output only. The thumbnail URL which should be used to preview
	// the attachment to a human user. Chat apps shouldn't use this URL to download
	// attachment content.
	ThumbnailUri string `json:"thumbnailUri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AttachmentDataRef") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttachmentDataRef") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Attachment) MarshalJSON() ([]byte, error) {
	type NoMethod Attachment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AttachmentDataRef: A reference to the attachment data.
type AttachmentDataRef struct {
	// AttachmentUploadToken: Optional. Opaque token containing a reference to an
	// uploaded attachment. Treated by clients as an opaque string and used to
	// create or update Chat messages with attachments.
	AttachmentUploadToken string `json:"attachmentUploadToken,omitempty"`
	// ResourceName: Optional. The resource name of the attachment data. This field
	// is used with the media API to download the attachment data.
	ResourceName string `json:"resourceName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AttachmentUploadToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttachmentUploadToken") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AttachmentDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod AttachmentDataRef
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Button: A button. Can be a text button or an image button.
type Button struct {
	// ImageButton: A button with image and `onclick` action.
	ImageButton *ImageButton `json:"imageButton,omitempty"`
	// TextButton: A button with text and `onclick` action.
	TextButton *TextButton `json:"textButton,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ImageButton") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ImageButton") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Button) MarshalJSON() ([]byte, error) {
	type NoMethod Button
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CalendarEventLinkData: Data for Calendar event links.
type CalendarEventLinkData struct {
	// CalendarId: The Calendar identifier
	// (https://developers.google.com/workspace/calendar/api/v3/reference/calendars)
	// of the linked Calendar.
	CalendarId string `json:"calendarId,omitempty"`
	// EventId: The Event identifier
	// (https://developers.google.com/workspace/calendar/api/v3/reference/events)
	// of the linked Calendar event.
	EventId string `json:"eventId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CalendarId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CalendarId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CalendarEventLinkData) MarshalJSON() ([]byte, error) {
	type NoMethod CalendarEventLinkData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Card: A card is a UI element that can contain UI widgets such as text and
// images.
type Card struct {
	// CardActions: The actions of this card.
	CardActions []*CardAction `json:"cardActions,omitempty"`
	// Header: The header of the card. A header usually contains a title and an
	// image.
	Header *CardHeader `json:"header,omitempty"`
	// Name: Name of the card.
	Name string `json:"name,omitempty"`
	// Sections: Sections are separated by a line divider.
	Sections []*Section `json:"sections,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CardActions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Card) MarshalJSON() ([]byte, error) {
	type NoMethod Card
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CardAction: A card action is the action associated with the card. For an
// invoice card, a typical action would be: delete invoice, email invoice or
// open the invoice in browser. Not supported by Google Chat apps.
type CardAction struct {
	// ActionLabel: The label used to be displayed in the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`
	// OnClick: The onclick action for this action item.
	OnClick *OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionLabel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod CardAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type CardHeader struct {
	// ImageStyle: The image's type (for example, square border or circular
	// border).
	//
	// Possible values:
	//   "IMAGE_STYLE_UNSPECIFIED"
	//   "IMAGE" - Square border.
	//   "AVATAR" - Circular border.
	ImageStyle string `json:"imageStyle,omitempty"`
	// ImageUrl: The URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`
	// Subtitle: The subtitle of the card header.
	Subtitle string `json:"subtitle,omitempty"`
	// Title: The title must be specified. The header has a fixed height: if both a
	// title and subtitle is specified, each takes up one line. If only the title
	// is specified, it takes up both lines.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ImageStyle") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ImageStyle") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod CardHeader
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CardWithId: A card
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/cards)
// in a Google Chat message. Only Chat apps can create cards. If your Chat app
// authenticates as a user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
// the message can't contain cards. Card builder
// (https://addons.gsuite.google.com/uikit/builder)
type CardWithId struct {
	// Card: A card. Maximum size is 32 KB.
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`
	// CardId: Required if the message contains multiple cards. A unique identifier
	// for a card in a message.
	CardId string `json:"cardId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Card") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Card") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CardWithId) MarshalJSON() ([]byte, error) {
	type NoMethod CardWithId
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChatAppLogEntry: JSON payload of error messages. If the Cloud Logging API is
// enabled, these error messages are logged to Google Cloud Logging
// (https://cloud.google.com/logging/docs).
type ChatAppLogEntry struct {
	// Deployment: The deployment that caused the error. For Chat apps built in
	// Apps Script, this is the deployment ID defined by Apps Script.
	Deployment string `json:"deployment,omitempty"`
	// DeploymentFunction: The unencrypted `callback_method` name that was running
	// when the error was encountered.
	DeploymentFunction string `json:"deploymentFunction,omitempty"`
	// Error: The error code and message.
	Error *Status `json:"error,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Deployment") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Deployment") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChatAppLogEntry) MarshalJSON() ([]byte, error) {
	type NoMethod ChatAppLogEntry
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChatClientDataSourceMarkup: For a `SelectionInput` widget that uses a
// multiselect menu, a data source from Google Chat. The data source populates
// selection items for the multiselect menu. For example, a user can select
// Google Chat spaces that they're a member of. Google Chat apps
// (https://developers.google.com/workspace/chat):
type ChatClientDataSourceMarkup struct {
	// SpaceDataSource: Google Chat spaces that the user is a member of.
	SpaceDataSource *SpaceDataSource `json:"spaceDataSource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "SpaceDataSource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SpaceDataSource") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChatClientDataSourceMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod ChatClientDataSourceMarkup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChatSpaceLinkData: Data for Chat space links.
type ChatSpaceLinkData struct {
	// Message: The message of the linked Chat space resource. Format:
	// `spaces/{space}/messages/{message}`
	Message string `json:"message,omitempty"`
	// Space: The space of the linked Chat space resource. Format: `spaces/{space}`
	Space string `json:"space,omitempty"`
	// Thread: The thread of the linked Chat space resource. Format:
	// `spaces/{space}/threads/{thread}`
	Thread string `json:"thread,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChatSpaceLinkData) MarshalJSON() ([]byte, error) {
	type NoMethod ChatSpaceLinkData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Color: Represents a color in the RGBA color space. This representation is
// designed for simplicity of conversion to and from color representations in
// various languages over compactness. For example, the fields of this
// representation can be trivially provided to the constructor of
// `java.awt.Color` in Java; it can also be trivially provided to UIColor's
// `+colorWithRed:green:blue:alpha` method in iOS; and, with just a little
// work, it can be easily formatted into a CSS `rgba()` string in JavaScript.
// This reference page doesn't have information about the absolute color space
// that should be used to interpret the RGB value—for example, sRGB, Adobe
// RGB, DCI-P3, and BT.2020. By default, applications should assume the sRGB
// color space. When color equality needs to be decided, implementations,
// unless documented otherwise, treat two colors as equal if all their red,
// green, blue, and alpha values each differ by at most `1e-5`. Example (Java):
// import com.google.type.Color; // ... public static java.awt.Color
// fromProto(Color protocolor) { float alpha = protocolor.hasAlpha() ?
// protocolor.getAlpha().getValue() : 1.0; return new java.awt.Color(
// protocolor.getRed(), protocolor.getGreen(), protocolor.getBlue(), alpha); }
// public static Color toProto(java.awt.Color color) { float red = (float)
// color.getRed(); float green = (float) color.getGreen(); float blue = (float)
// color.getBlue(); float denominator = 255.0; Color.Builder resultBuilder =
// Color .newBuilder() .setRed(red / denominator) .setGreen(green /
// denominator) .setBlue(blue / denominator); int alpha = color.getAlpha(); if
// (alpha != 255) { result.setAlpha( FloatValue .newBuilder()
// .setValue(((float) alpha) / denominator) .build()); } return
// resultBuilder.build(); } // ... Example (iOS / Obj-C): // ... static
// UIColor* fromProto(Color* protocolor) { float red = [protocolor red]; float
// green = [protocolor green]; float blue = [protocolor blue]; FloatValue*
// alpha_wrapper = [protocolor alpha]; float alpha = 1.0; if (alpha_wrapper !=
// nil) { alpha = [alpha_wrapper value]; } return [UIColor colorWithRed:red
// green:green blue:blue alpha:alpha]; } static Color* toProto(UIColor* color)
// { CGFloat red, green, blue, alpha; if (![color getRed:&red green:&green
// blue:&blue alpha:&alpha]) { return nil; } Color* result = [[Color alloc]
// init]; [result setRed:red]; [result setGreen:green]; [result setBlue:blue];
// if (alpha <= 0.9999) { [result setAlpha:floatWrapperWithValue(alpha)]; }
// [result autorelease]; return result; } // ... Example (JavaScript): // ...
// var protoToCssColor = function(rgb_color) { var redFrac = rgb_color.red ||
// 0.0; var greenFrac = rgb_color.green || 0.0; var blueFrac = rgb_color.blue
// || 0.0; var red = Math.floor(redFrac * 255); var green =
// Math.floor(greenFrac * 255); var blue = Math.floor(blueFrac * 255); if
// (!('alpha' in rgb_color)) { return rgbToCssColor(red, green, blue); } var
// alphaFrac = rgb_color.alpha.value || 0.0; var rgbParams = [red, green,
// blue].join(','); return ['rgba(', rgbParams, ',', alphaFrac, ')'].join(”);
// }; var rgbToCssColor = function(red, green, blue) { var rgbNumber = new
// Number((red << 16) | (green << 8) | blue); var hexString =
// rgbNumber.toString(16); var missingZeros = 6 - hexString.length; var
// resultBuilder = ['#']; for (var i = 0; i < missingZeros; i++) {
// resultBuilder.push('0'); } resultBuilder.push(hexString); return
// resultBuilder.join(”); }; // ...
type Color struct {
	// Alpha: The fraction of this color that should be applied to the pixel. That
	// is, the final pixel color is defined by the equation: `pixel color = alpha *
	// (this color) + (1.0 - alpha) * (background color)` This means that a value
	// of 1.0 corresponds to a solid color, whereas a value of 0.0 corresponds to a
	// completely transparent color. This uses a wrapper message rather than a
	// simple float scalar so that it is possible to distinguish between a default
	// value and the value being unset. If omitted, this color object is rendered
	// as a solid color (as if the alpha value had been explicitly given a value of
	// 1.0).
	Alpha float64 `json:"alpha,omitempty"`
	// Blue: The amount of blue in the color as a value in the interval [0, 1].
	Blue float64 `json:"blue,omitempty"`
	// Green: The amount of green in the color as a value in the interval [0, 1].
	Green float64 `json:"green,omitempty"`
	// Red: The amount of red in the color as a value in the interval [0, 1].
	Red float64 `json:"red,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Alpha") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Alpha") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Color) MarshalJSON() ([]byte, error) {
	type NoMethod Color
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Color) UnmarshalJSON(data []byte) error {
	type NoMethod Color
	var s1 struct {
		Alpha gensupport.JSONFloat64 `json:"alpha"`
		Blue  gensupport.JSONFloat64 `json:"blue"`
		Green gensupport.JSONFloat64 `json:"green"`
		Red   gensupport.JSONFloat64 `json:"red"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Alpha = float64(s1.Alpha)
	s.Blue = float64(s1.Blue)
	s.Green = float64(s1.Green)
	s.Red = float64(s1.Red)
	return nil
}

// CommonEventObject: The common event object is the portion of the overall
// event object that carries general, host-independent information to the
// add-on from the user's client. This information includes details such as the
// user's locale, host app, and platform. In addition to homepage and
// contextual triggers, add-ons construct and pass event objects to action
// callback functions
// (https://developers.google.com/workspace/add-ons/concepts/actions#callback_functions)
// when the user interacts with widgets. Your add-on's callback function can
// query the common event object to determine the contents of open widgets in
// the user's client. For example, your add-on can locate the text a user has
// entered into a TextInput
// (https://developers.google.com/apps-script/reference/card-service/text-input)
// widget in the `eventObject.commentEventObject.formInputs` object. For Chat
// apps, the name of the function that the user invoked when interacting with a
// widget.
type CommonEventObject struct {
	// FormInputs: A map containing the current values of the widgets in the
	// displayed card. The map keys are the string IDs assigned with each widget.
	// The structure of the map value object is dependent on the widget type:
	// **Note**: The following examples are formatted for Apps Script's V8 runtime.
	// If you're using Rhino runtime, you must add `[""]` after the value. For
	// example, instead of
	// `e.commonEventObject.formInputs.employeeName.stringInputs.value[0]`, format
	// the event object as
	// `e.commonEventObject.formInputs.employeeName[""].stringInputs.value[0]`. To
	// learn more about runtimes in Apps Script, see the V8 Runtime Overview
	// (https://developers.google.com/apps-script/guides/v8-runtime). *
	// Single-valued widgets (for example, a text box): a list of strings (only one
	// element). **Example**: for a text input widget with `employeeName` as its
	// ID, access the text input value with:
	// `e.commonEventObject.formInputs.employeeName.stringInputs.value[0]`. *
	// Multi-valued widgets (for example, checkbox groups): a list of strings.
	// **Example**: for a multi-value widget with `participants` as its ID, access
	// the value array with:
	// `e.commonEventObject.formInputs.participants.stringInputs.value`. * **A
	// date-time picker**: a `DateTimeInput object`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#date-time-input).
	// **Example**: For a picker with an ID of `myDTPicker`, access the
	// `DateTimeInput`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#date-time-input)
	// object using `e.commonEventObject.formInputs.myDTPicker.dateTimeInput`. *
	// **A date-only picker**: a `DateInput object`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#date-input).
	// **Example**: For a picker with an ID of `myDatePicker`, access the
	// `DateInput`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#date-input)
	// object using `e.commonEventObject.formInputs.myDatePicker.dateInput`. * **A
	// time-only picker**: a `TimeInput object`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#time-input).
	// **Example**: For a picker with an ID of `myTimePicker`, access the
	// `TimeInput`
	// (https://developers.google.com/workspace/add-ons/concepts/event-objects#time-input)
	// object using `e.commonEventObject.formInputs.myTimePicker.timeInput`.
	FormInputs map[string]Inputs `json:"formInputs,omitempty"`
	// HostApp: Indicates the host app the add-on is active in when the event
	// object is generated. Possible values include the following: * `GMAIL` *
	// `CALENDAR` * `DRIVE` * `DOCS` * `SHEETS` * `SLIDES` * `CHAT`
	//
	// Possible values:
	//   "UNSPECIFIED_HOST_APP" - Google can't identify a host app.
	//   "GMAIL" - The add-on launches from Gmail.
	//   "CALENDAR" - The add-on launches from Google Calendar.
	//   "DRIVE" - The add-on launches from Google Drive.
	//   "DEMO" - Not used.
	//   "DOCS" - The add-on launches from Google Docs.
	//   "MEET" - The add-on launches from Google Meet.
	//   "SHEETS" - The add-on launches from Google Sheets.
	//   "SLIDES" - The add-on launches from Google Slides.
	//   "DRAWINGS" - The add-on launches from Google Drawings.
	//   "CHAT" - A Google Chat app.
	HostApp string `json:"hostApp,omitempty"`
	// InvokedFunction: Name of the function to invoke. This field doesn't populate
	// for Google Workspace Add-ons that extend Google Chat. Instead, to receive
	// function data like identifiers, add-ons that extend Chat should use the
	// `parameters` field. See Build interactive interfaces for Chat apps
	// (https://developers.google.com/workspace/add-ons/chat/build).
	InvokedFunction string `json:"invokedFunction,omitempty"`
	// Parameters: Any additional parameters you supply to an action using
	// `actionParameters`
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#google.apps.card.v1.Action.ActionParameter)
	// or `Action.setParameters()`
	// (https://developers.google.com/apps-script/reference/card-service/action#setparametersparameters).
	// **Developer Preview:** For add-ons that extend Google Chat
	// (https://developers.google.com/workspace/add-ons/chat), to suggest items
	// based on what the users type in multiselect menus, use the value of the
	// "autocomplete_widget_query" key
	// (`event.commonEventObject.parameters["autocomplete_widget_query"]`). You can
	// use this value to query a database and suggest selectable items to users as
	// they type. For details, see Collect and process information from Google Chat
	// users
	// (https://developers.google.com/workspace/add-ons/chat/collect-information).
	Parameters map[string]string `json:"parameters,omitempty"`
	// Platform: The platform enum which indicates the platform where the event
	// originates (`WEB`, `IOS`, or `ANDROID`). Not supported by Chat apps.
	//
	// Possible values:
	//   "UNKNOWN_PLATFORM"
	//   "WEB"
	//   "IOS"
	//   "ANDROID"
	Platform string `json:"platform,omitempty"`
	// TimeZone: **Disabled by default.** The timezone ID and offset from
	// Coordinated Universal Time (UTC). To turn on this field, you must set
	// `addOns.common.useLocaleFromApp` to `true` in your add-on's manifest. Your
	// add-on's scope list must also include
	// `https://www.googleapis.com/auth/script.locale`. See Accessing user locale
	// and timezone
	// (https://developers.google.com/workspace/add-ons/how-tos/access-user-locale)
	// for more details. Only supported for the event types `CARD_CLICKED`
	// (https://developers.google.com/chat/api/reference/rest/v1/EventType#ENUM_VALUES.CARD_CLICKED)
	// and `SUBMIT_DIALOG`
	// (https://developers.google.com/chat/api/reference/rest/v1/DialogEventType#ENUM_VALUES.SUBMIT_DIALOG).
	TimeZone *TimeZone `json:"timeZone,omitempty"`
	// UserLocale: **Disabled by default.** The user's language and country/region
	// identifier in the format of ISO 639
	// (https://wikipedia.org/wiki/ISO_639_macrolanguage) language code-ISO 3166
	// (https://wikipedia.org/wiki/ISO_3166) country/region code. For example,
	// `en-US`. To turn on this field, you must set
	// `addOns.common.useLocaleFromApp` to `true` in your add-on's manifest. Your
	// add-on's scope list must also include
	// `https://www.googleapis.com/auth/script.locale`. See Accessing user locale
	// and timezone
	// (https://developers.google.com/workspace/add-ons/how-tos/access-user-locale)
	// for more details.
	UserLocale string `json:"userLocale,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FormInputs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FormInputs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommonEventObject) MarshalJSON() ([]byte, error) {
	type NoMethod CommonEventObject
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CompleteImportSpaceRequest: Request message for completing the import
// process for a space.
type CompleteImportSpaceRequest struct {
}

// CompleteImportSpaceResponse: Response message for completing the import
// process for a space.
type CompleteImportSpaceResponse struct {
	// Space: The import mode space.
	Space *Space `json:"space,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Space") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Space") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CompleteImportSpaceResponse) MarshalJSON() ([]byte, error) {
	type NoMethod CompleteImportSpaceResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomEmoji: Represents a custom emoji
// (https://support.google.com/chat/answer/12800149).
type CustomEmoji struct {
	// EmojiName: Optional. Immutable. User-provided name for the custom emoji,
	// which is unique within the organization. Required when the custom emoji is
	// created, output only otherwise. Emoji names must start and end with colons,
	// must be lowercase and can only contain alphanumeric characters, hyphens, and
	// underscores. Hyphens and underscores should be used to separate words and
	// cannot be used consecutively. Example: `:valid-emoji-name:`
	EmojiName string `json:"emojiName,omitempty"`
	// Name: Identifier. The resource name of the custom emoji, assigned by the
	// server. Format: `customEmojis/{customEmoji}`
	Name string `json:"name,omitempty"`
	// Payload: Optional. Input only. Payload data. Required when the custom emoji
	// is created.
	Payload *CustomEmojiPayload `json:"payload,omitempty"`
	// TemporaryImageUri: Output only. A temporary image URL for the custom emoji,
	// valid for at least 10 minutes. Note that this is not populated in the
	// response when the custom emoji is created.
	TemporaryImageUri string `json:"temporaryImageUri,omitempty"`
	// Uid: Output only. Unique key for the custom emoji resource.
	Uid string `json:"uid,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "EmojiName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EmojiName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomEmoji) MarshalJSON() ([]byte, error) {
	type NoMethod CustomEmoji
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomEmojiMetadata: Annotation metadata for custom emoji.
type CustomEmojiMetadata struct {
	// CustomEmoji: The custom emoji.
	CustomEmoji *CustomEmoji `json:"customEmoji,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomEmoji") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomEmoji") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomEmojiMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod CustomEmojiMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomEmojiPayload: Payload data for the custom emoji.
type CustomEmojiPayload struct {
	// FileContent: Required. Input only. The image used for the custom emoji. The
	// payload must be under 256 KB and the dimension of the image must be square
	// and between 64 and 500 pixels. The restrictions are subject to change.
	FileContent string `json:"fileContent,omitempty"`
	// Filename: Required. Input only. The image file name. Supported file
	// extensions: `.png`, `.jpg`, `.gif`.
	Filename string `json:"filename,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FileContent") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FileContent") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomEmojiPayload) MarshalJSON() ([]byte, error) {
	type NoMethod CustomEmojiPayload
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DateInput: Date input values.
type DateInput struct {
	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "MsSinceEpoch") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MsSinceEpoch") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DateInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DateTimeInput: Date and time input values.
type DateTimeInput struct {
	// HasDate: Whether the `datetime` input includes a calendar date.
	HasDate bool `json:"hasDate,omitempty"`
	// HasTime: Whether the `datetime` input includes a timestamp.
	HasTime bool `json:"hasTime,omitempty"`
	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "HasDate") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HasDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DateTimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateTimeInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DeletionMetadata: Information about a deleted message. A message is deleted
// when `delete_time` is set.
type DeletionMetadata struct {
	// DeletionType: Indicates who deleted the message.
	//
	// Possible values:
	//   "DELETION_TYPE_UNSPECIFIED" - This value is unused.
	//   "CREATOR" - User deleted their own message.
	//   "SPACE_OWNER" - An owner or manager deleted the message.
	//   "ADMIN" - A Google Workspace administrator deleted the message.
	// Administrators can delete any message in the space, including messages sent
	// by any space member or Chat app.
	//   "APP_MESSAGE_EXPIRY" - A Chat app deleted its own message when it expired.
	//   "CREATOR_VIA_APP" - A Chat app deleted the message on behalf of the
	// creator (using user authentication).
	//   "SPACE_OWNER_VIA_APP" - A Chat app deleted the message on behalf of a
	// space manager (using user authentication).
	//   "SPACE_MEMBER" - A member of the space deleted the message. Users can
	// delete messages sent by apps.
	DeletionType string `json:"deletionType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeletionType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeletionType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeletionMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod DeletionMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DeprecatedEvent:  A Google Chat app interaction event that represents and
// contains data about a user's interaction with a Chat app. To configure your
// Chat app to receive interaction events, see Receive and respond to user
// interactions
// (https://developers.google.com/workspace/chat/receive-respond-interactions).
// In addition to receiving events from user interactions, Chat apps can
// receive events about changes to spaces, such as when a new member is added
// to a space. To learn about space events, see Work with events from Google
// Chat (https://developers.google.com/workspace/chat/events-overview). Note:
// This event is only used for Chat interaction events
// (https://developers.google.com/workspace/chat/receive-respond-interactions).
// If your Chat app is built as a Google Workspace add-on
// (https://developers.google.com/workspace/add-ons/chat/build), see Chat event
// objects
// (https://developers.google.com/workspace/add-ons/concepts/event-objects#chat-event-object)
// in the add-ons documentation.
type DeprecatedEvent struct {
	// Action: For `CARD_CLICKED` interaction events, the form action data
	// associated when a user clicks a card or dialog. To learn more, see Read form
	// data input by users on cards
	// (https://developers.google.com/workspace/chat/read-form-data).
	Action *FormAction `json:"action,omitempty"`
	// AppCommandMetadata: Metadata about a Chat app command.
	AppCommandMetadata *AppCommandMetadata `json:"appCommandMetadata,omitempty"`
	// Common: Represents information about the user's client, such as locale, host
	// app, and platform. For Chat apps, `CommonEventObject` includes information
	// submitted by users interacting with dialogs
	// (https://developers.google.com/workspace/chat/dialogs), like data entered on
	// a card.
	Common *CommonEventObject `json:"common,omitempty"`
	// ConfigCompleteRedirectUrl: This URL is populated for `MESSAGE`,
	// `ADDED_TO_SPACE`, and `APP_COMMAND` interaction events. After completing an
	// authorization or configuration flow outside of Google Chat, users must be
	// redirected to this URL to signal to Google Chat that the authorization or
	// configuration flow was successful. For more information, see Connect a Chat
	// app with other services and tools
	// (https://developers.google.com/workspace/chat/connect-web-services-tools).
	ConfigCompleteRedirectUrl string `json:"configCompleteRedirectUrl,omitempty"`
	// DialogEventType: The type of dialog
	// (https://developers.google.com/workspace/chat/dialogs) interaction event
	// received.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value. Unspecified.
	//   "REQUEST_DIALOG" - A user opens a dialog.
	//   "SUBMIT_DIALOG" - A user clicks an interactive element of a dialog. For
	// example, a user fills out information in a dialog and clicks a button to
	// submit the information.
	//   "CANCEL_DIALOG" - A user closes a dialog without submitting information.
	// The Chat app only receives this interaction event when users click the close
	// icon in the top right corner of the dialog. When the user closes the dialog
	// by other means (such as refreshing the browser, clicking outside the dialog
	// box, or pressing the escape key), no event is sent.
	DialogEventType string `json:"dialogEventType,omitempty"`
	// EventTime: The timestamp indicating when the interaction event occurred.
	EventTime string `json:"eventTime,omitempty"`
	// IsDialogEvent: For `CARD_CLICKED` and `MESSAGE` interaction events, whether
	// the user is interacting with or about to interact with a dialog
	// (https://developers.google.com/workspace/chat/dialogs).
	IsDialogEvent bool `json:"isDialogEvent,omitempty"`
	// Message: For `ADDED_TO_SPACE`, `CARD_CLICKED`, and `MESSAGE` interaction
	// events, the message that triggered the interaction event, if applicable.
	Message *Message `json:"message,omitempty"`
	// Space: The space in which the user interacted with the Chat app.
	Space *Space `json:"space,omitempty"`
	// Thread: The thread in which the user interacted with the Chat app. This
	// could be in a new thread created by a newly sent message. This field is
	// populated if the interaction event is associated with a specific message or
	// thread.
	Thread *Thread `json:"thread,omitempty"`
	// ThreadKey: The Chat app-defined key for the thread related to the
	// interaction event. See `spaces.messages.thread.threadKey`
	// (/chat/api/reference/rest/v1/spaces.messages#Thread.FIELDS.thread_key) for
	// more information.
	ThreadKey string `json:"threadKey,omitempty"`
	// Token: A secret value that legacy Chat apps can use to verify if a request
	// is from Google. Google randomly generates the token, and its value remains
	// static. You can obtain, revoke, or regenerate the token from the Chat API
	// configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat)
	// in the Google Cloud Console. Modern Chat apps don't use this field. It is
	// absent from API responses and the Chat API configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat).
	Token string `json:"token,omitempty"`
	// Type: The type (/workspace/chat/api/reference/rest/v1/EventType) of user
	// interaction with the Chat app, such as `MESSAGE` or `ADDED_TO_SPACE`.
	//
	// Possible values:
	//   "UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "MESSAGE" - A user sends the Chat app a message, or invokes the Chat app
	// in a space. Examples of message events include: * Any message in a direct
	// message (DM) space with the Chat app. * A message in a multi-person space
	// where a person @mentions the Chat app, or uses one of its [slash
	// commands](https://developers.google.com/workspace/chat/commands#types). * If
	// you've configured link previews for your Chat app, a user posts a message
	// that contains a link that matches the configured URL pattern.
	//   "ADDED_TO_SPACE" - A user adds the Chat app to a space, or a Google
	// Workspace administrator installs the Chat app in direct message spaces for
	// users in their organization. Chat apps typically respond to this interaction
	// event by posting a welcome message in the space. When administrators install
	// Chat apps, the `space.adminInstalled` field is set to `true` and users can't
	// uninstall them. To learn about Chat apps installed by administrators, see
	// Google Workspace Admin Help's documentation, [Install Marketplace apps in
	// your domain](https://support.google.com/a/answer/172482).
	//   "REMOVED_FROM_SPACE" - A user removes the Chat app from a space, or a
	// Google Workspace administrator uninstalls the Chat app for a user in their
	// organization. Chat apps can't respond with messages to this event, because
	// they have already been removed. When administrators uninstall Chat apps, the
	// `space.adminInstalled` field is set to `false`. If a user installed the Chat
	// app before the administrator, the Chat app remains installed for the user
	// and the Chat app doesn't receive a `REMOVED_FROM_SPACE` interaction event.
	//   "CARD_CLICKED" - A user clicks an interactive element of a card or dialog
	// from a Chat app, such as a button. To receive an interaction event, the
	// button must trigger another interaction with the Chat app. For example, a
	// Chat app doesn't receive a `CARD_CLICKED` interaction event if a user clicks
	// a button that opens a link to a website, but receives interaction events in
	// the following examples: * The user clicks a `Send feedback` button on a
	// card, which opens a dialog for the user to input information. * The user
	// clicks a `Submit` button after inputting information into a card or dialog.
	// If a user clicks a button to open, submit, or cancel a dialog, the
	// `CARD_CLICKED` interaction event's `isDialogEvent` field is set to `true`
	// and includes a
	// [`DialogEventType`](https://developers.google.com/workspace/chat/api/referenc
	// e/rest/v1/DialogEventType).
	//   "WIDGET_UPDATED" - A user updates a widget in a card message or dialog.
	// This event is triggered when a user interacts with a widget that has an
	// associated action.
	//   "APP_COMMAND" - A user uses a Chat app
	// [command](https://developers.google.com/workspace/chat/commands#types),
	// including slash commands and quick commands.
	Type string `json:"type,omitempty"`
	// User: The user that interacted with the Chat app.
	User *User `json:"user,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeprecatedEvent) MarshalJSON() ([]byte, error) {
	type NoMethod DeprecatedEvent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Dialog: Wrapper around the card body of the dialog.
type Dialog struct {
	// Body: Input only. Body of the dialog, which is rendered in a modal. Google
	// Chat apps don't support the following card entities: `DateTimePicker`,
	// `OnChangeAction`.
	Body *GoogleAppsCardV1Card `json:"body,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Body") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Body") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Dialog) MarshalJSON() ([]byte, error) {
	type NoMethod Dialog
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DialogAction: Contains a dialog
// (https://developers.google.com/workspace/chat/dialogs) and request status
// code.
type DialogAction struct {
	// ActionStatus: Input only. Status for a request to either invoke or submit a
	// dialog (https://developers.google.com/workspace/chat/dialogs). Displays a
	// status and message to users, if necessary. For example, in case of an error
	// or success.
	ActionStatus *ActionStatus `json:"actionStatus,omitempty"`
	// Dialog: Input only. Dialog
	// (https://developers.google.com/workspace/chat/dialogs) for the request.
	Dialog *Dialog `json:"dialog,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionStatus") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionStatus") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DialogAction) MarshalJSON() ([]byte, error) {
	type NoMethod DialogAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveDataRef: A reference to the data of a drive attachment.
type DriveDataRef struct {
	// DriveFileId: The ID for the drive file. Use with the Drive API.
	DriveFileId string `json:"driveFileId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DriveFileId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DriveFileId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod DriveDataRef
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveLinkData: Data for Google Drive links.
type DriveLinkData struct {
	// DriveDataRef: A DriveDataRef
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages.attachments#drivedataref)
	// which references a Google Drive file.
	DriveDataRef *DriveDataRef `json:"driveDataRef,omitempty"`
	// MimeType: The mime type of the linked Google Drive resource.
	MimeType string `json:"mimeType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DriveDataRef") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DriveDataRef") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveLinkData) MarshalJSON() ([]byte, error) {
	type NoMethod DriveLinkData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Emoji: An emoji that is used as a reaction to a message.
type Emoji struct {
	// CustomEmoji: A custom emoji.
	CustomEmoji *CustomEmoji `json:"customEmoji,omitempty"`
	// Unicode: Optional. A basic emoji represented by a unicode string.
	Unicode string `json:"unicode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomEmoji") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomEmoji") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Emoji) MarshalJSON() ([]byte, error) {
	type NoMethod Emoji
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EmojiReactionSummary: The number of people who reacted to a message with a
// specific emoji.
type EmojiReactionSummary struct {
	// Emoji: Output only. Emoji associated with the reactions.
	Emoji *Emoji `json:"emoji,omitempty"`
	// ReactionCount: Output only. The total number of reactions using the
	// associated emoji.
	ReactionCount int64 `json:"reactionCount,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Emoji") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Emoji") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EmojiReactionSummary) MarshalJSON() ([]byte, error) {
	type NoMethod EmojiReactionSummary
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// FormAction: A form action describes the behavior when the form is submitted.
// For example, you can invoke Apps Script to handle the form.
type FormAction struct {
	// ActionMethodName: The method name is used to identify which part of the form
	// triggered the form submission. This information is echoed back to the Chat
	// app as part of the card click event. You can use the same method name for
	// several elements that trigger a common behavior.
	ActionMethodName string `json:"actionMethodName,omitempty"`
	// Parameters: List of action parameters.
	Parameters []*ActionParameter `json:"parameters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionMethodName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionMethodName") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FormAction) MarshalJSON() ([]byte, error) {
	type NoMethod FormAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Action: An action that describes the behavior when the form
// is submitted. For example, you can invoke an Apps Script script to handle
// the form. If the action is triggered, the form values are sent to the
// server. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Action struct {
	// AllWidgetsAreRequired: Optional. If this is true, then all widgets are
	// considered required by this action. Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	AllWidgetsAreRequired bool `json:"allWidgetsAreRequired,omitempty"`
	// Function: A custom function to invoke when the containing element is clicked
	// or otherwise activated. For example usage, see Read form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Function string `json:"function,omitempty"`
	// Interaction: Optional. Required when opening a dialog
	// (https://developers.google.com/workspace/chat/dialogs). What to do in
	// response to an interaction with a user, such as a user clicking a button in
	// a card message. If unspecified, the app responds by executing an
	// `action`—like opening a link or running a function—as normal. By
	// specifying an `interaction`, the app can respond in special interactive
	// ways. For example, by setting `interaction` to `OPEN_DIALOG`, the app can
	// open a dialog (https://developers.google.com/workspace/chat/dialogs). When
	// specified, a loading indicator isn't shown. If specified for an add-on, the
	// entire card is stripped and nothing is shown in the client. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	//
	// Possible values:
	//   "INTERACTION_UNSPECIFIED" - Default value. The `action` executes as
	// normal.
	//   "OPEN_DIALOG" - Opens a
	// [dialog](https://developers.google.com/workspace/chat/dialogs), a windowed,
	// card-based interface that Chat apps use to interact with users. Only
	// supported by Chat apps in response to button-clicks on card messages. If
	// specified for an add-on, the entire card is stripped and nothing is shown in
	// the client. [Google Chat
	// apps](https://developers.google.com/workspace/chat):
	Interaction string `json:"interaction,omitempty"`
	// LoadIndicator: Specifies the loading indicator that the action displays
	// while making the call to the action.
	//
	// Possible values:
	//   "SPINNER" - Displays a spinner to indicate that content is loading.
	//   "NONE" - Nothing is displayed.
	LoadIndicator string `json:"loadIndicator,omitempty"`
	// Parameters: List of action parameters.
	Parameters []*GoogleAppsCardV1ActionParameter `json:"parameters,omitempty"`
	// PersistValues: Indicates whether form values persist after the action. The
	// default value is `false`. If `true`, form values remain after the action is
	// triggered. To let the user make changes while the action is being processed,
	// set `LoadIndicator`
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `NONE`. For card messages
	// (https://developers.google.com/workspace/chat/api/guides/v1/messages/create#create)
	// in Chat apps, you must also set the action's `ResponseType`
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages#responsetype)
	// to `UPDATE_MESSAGE` and use the same `card_id`
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages#CardWithId)
	// from the card that contained the action. If `false`, the form values are
	// cleared when the action is triggered. To prevent the user from making
	// changes while the action is being processed, set `LoadIndicator`
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `SPINNER`.
	PersistValues bool `json:"persistValues,omitempty"`
	// RequiredWidgets: Optional. Fill this list with the names of widgets that
	// this Action needs for a valid submission. If the widgets listed here don't
	// have a value when this Action is invoked, the form submission is aborted.
	// Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	RequiredWidgets []string `json:"requiredWidgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AllWidgetsAreRequired") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllWidgetsAreRequired") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Action) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Action
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ActionParameter: List of string parameters to supply when
// the action method is invoked. For example, consider three snooze buttons:
// snooze now, snooze one day, or snooze next week. You might use `action
// method = snooze()`, passing the snooze type and snooze time in the list of
// string parameters. To learn more, see `CommonEventObject`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/Event#commoneventobject).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`
	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Key") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Key") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ActionParameter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1BorderStyle: The style options for the border of a card or
// widget, including the border type and color. Google Workspace add-ons and
// Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1BorderStyle struct {
	// CornerRadius: The corner radius for the border.
	CornerRadius int64 `json:"cornerRadius,omitempty"`
	// StrokeColor: The colors to use when the type is `BORDER_TYPE_STROKE`. To set
	// the stroke color, specify a value for the `red`, `green`, and `blue` fields.
	// The value must be a float number between 0 and 1 based on the RGB color
	// value, where `0` (0/255) represents the absence of color and `1` (255/255)
	// represents the maximum intensity of the color. For example, the following
	// sets the color to red at its maximum intensity: ``` "color": { "red": 1,
	// "green": 0, "blue": 0, } ``` The `alpha` field is unavailable for stroke
	// color. If specified, this field is ignored.
	StrokeColor *Color `json:"strokeColor,omitempty"`
	// Type: The border type.
	//
	// Possible values:
	//   "BORDER_TYPE_UNSPECIFIED" - Don't use. Unspecified.
	//   "NO_BORDER" - No border.
	//   "STROKE" - Default value. Outline.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CornerRadius") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CornerRadius") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1BorderStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1BorderStyle
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Button: A text, icon, or text and icon button that users can
// click. For an example in Google Chat apps, see Add a button
// (https://developers.google.com/workspace/chat/design-interactive-card-dialog#add_a_button).
// To make an image a clickable button, specify an `Image` (not an
// `ImageComponent`) and set an `onClick` action. Google Workspace add-ons and
// Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Button struct {
	// AltText: The alternative text that's used for accessibility. Set descriptive
	// text that lets users know what the button does. For example, if a button
	// opens a hyperlink, you might write: "Opens a new browser tab and navigates
	// to the Google Chat developer documentation at
	// https://developers.google.com/workspace/chat".
	AltText string `json:"altText,omitempty"`
	// Color: Optional. The color of the button. If set, the button `type` is set
	// to `FILLED` and the color of `text` and `icon` fields are set to a
	// contrasting color for readability. For example, if the button color is set
	// to blue, any text or icons in the button are set to white. To set the button
	// color, specify a value for the `red`, `green`, and `blue` fields. The value
	// must be a float number between 0 and 1 based on the RGB color value, where
	// `0` (0/255) represents the absence of color and `1` (255/255) represents the
	// maximum intensity of the color. For example, the following sets the color to
	// red at its maximum intensity: ``` "color": { "red": 1, "green": 0, "blue":
	// 0, } ``` The `alpha` field is unavailable for button color. If specified,
	// this field is ignored.
	Color *Color `json:"color,omitempty"`
	// Disabled: If `true`, the button is displayed in an inactive state and
	// doesn't respond to user actions.
	Disabled bool `json:"disabled,omitempty"`
	// Icon: An icon displayed inside the button. If both `icon` and `text` are
	// set, then the icon appears before the text.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`
	// OnClick: Required. The action to perform when a user clicks the button, such
	// as opening a hyperlink or running a custom function.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// Text: The text displayed inside the button.
	Text string `json:"text,omitempty"`
	// Type: Optional. The type of a button. If unset, button type defaults to
	// `OUTLINED`. If the `color` field is set, the button type is forced to
	// `FILLED` and any value set for this field is ignored.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Don't use. Unspecified.
	//   "OUTLINED" - Outlined buttons are medium-emphasis buttons. They usually
	// contain actions that are important, but aren’t the primary action in a
	// Chat app or an add-on.
	//   "FILLED" - A filled button has a container with a solid color. It has the
	// most visual impact and is recommended for the important and primary action
	// in a Chat app or an add-on.
	//   "FILLED_TONAL" - A filled tonal button is an alternative middle ground
	// between filled and outlined buttons. They’re useful in contexts where a
	// lower-priority button requires slightly more emphasis than an outline button
	// would give.
	//   "BORDERLESS" - A button does not have an invisible container in its
	// default state. It is often used for the lowest priority actions, especially
	// when presenting multiple options.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AltText") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Button) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Button
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ButtonList: A list of buttons layed out horizontally. For an
// example in Google Chat apps, see Add a button
// (https://developers.google.com/workspace/chat/design-interactive-card-dialog#add_a_button).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1ButtonList struct {
	// Buttons: An array of buttons.
	Buttons []*GoogleAppsCardV1Button `json:"buttons,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Buttons") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Buttons") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ButtonList) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ButtonList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Card: A card interface displayed in a Google Chat message or
// Google Workspace add-on. Cards support a defined layout, interactive UI
// elements like buttons, and rich media like images. Use cards to present
// detailed information, gather information from users, and guide users to take
// a next step. Card builder (https://addons.gsuite.google.com/uikit/builder)
// To learn how to build cards, see the following documentation: * For Google
// Chat apps, see Design the components of a card or dialog
// (https://developers.google.com/workspace/chat/design-components-card-dialog).
// * For Google Workspace add-ons, see Card-based interfaces
// (https://developers.google.com/apps-script/add-ons/concepts/cards). Note:
// You can add up to 100 widgets per card. Any widgets beyond this limit are
// ignored. This limit applies to both card messages and dialogs in Google Chat
// apps, and to cards in Google Workspace add-ons. **Example: Card message for
// a Google Chat app** !Example contact card
// (https://developers.google.com/workspace/chat/images/card_api_reference.png)
// To create the sample card message in Google Chat, use the following JSON:
// ``` { "cardsV2": [ { "cardId": "unique-card-id", "card": { "header": {
// "title": "Sasha", "subtitle": "Software Engineer", "imageUrl":
// "https://developers.google.com/workspace/chat/images/quickstart-app-avatar.pn
// g", "imageType": "CIRCLE", "imageAltText": "Avatar for Sasha" }, "sections":
// [ { "header": "Contact Info", "collapsible": true,
// "uncollapsibleWidgetsCount": 1, "widgets": [ { "decoratedText": {
// "startIcon": { "knownIcon": "EMAIL" }, "text": "sasha@example.com" } }, {
// "decoratedText": { "startIcon": { "knownIcon": "PERSON" }, "text": "Online"
// } }, { "decoratedText": { "startIcon": { "knownIcon": "PHONE" }, "text": "+1
// (555) 555-1234" } }, { "buttonList": { "buttons": [ { "text": "Share",
// "onClick": { "openLink": { "url": "https://example.com/share" } } }, {
// "text": "Edit", "onClick": { "action": { "function": "goToView",
// "parameters": [ { "key": "viewType", "value": "EDIT" } ] } } } ] } } ] } ] }
// } ] } ```
type GoogleAppsCardV1Card struct {
	// CardActions: The card's actions. Actions are added to the card's toolbar
	// menu. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons): For example, the
	// following JSON constructs a card action menu with `Settings` and `Send
	// Feedback` options: ``` "card_actions": [ { "actionLabel": "Settings",
	// "onClick": { "action": { "functionName": "goToView", "parameters": [ {
	// "key": "viewType", "value": "SETTING" } ], "loadIndicator":
	// "LoadIndicator.SPINNER" } } }, { "actionLabel": "Send Feedback", "onClick":
	// { "openLink": { "url": "https://example.com/feedback" } } } ] ```
	CardActions []*GoogleAppsCardV1CardAction `json:"cardActions,omitempty"`
	// DisplayStyle: In Google Workspace add-ons, sets the display properties of
	// the `peekCardHeader`. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons):
	//
	// Possible values:
	//   "DISPLAY_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "PEEK" - The header of the card appears at the bottom of the sidebar,
	// partially covering the current top card of the stack. Clicking the header
	// pops the card into the card stack. If the card has no header, a generated
	// header is used instead.
	//   "REPLACE" - Default value. The card is shown by replacing the view of the
	// top card in the card stack.
	DisplayStyle string `json:"displayStyle,omitempty"`
	// ExpressionData: The expression data for the card. Only supported by Google
	// Workspace Workflow, but not Google Chat apps or Google Workspace add-ons.
	ExpressionData []*GoogleAppsCardV1ExpressionData `json:"expressionData,omitempty"`
	// FixedFooter: The fixed footer shown at the bottom of this card. Setting
	// `fixedFooter` without specifying a `primaryButton` or a `secondaryButton`
	// causes an error. For Chat apps, you can use fixed footers in dialogs
	// (https://developers.google.com/workspace/chat/dialogs), but not card
	// messages
	// (https://developers.google.com/workspace/chat/create-messages#create).
	// Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	FixedFooter *GoogleAppsCardV1CardFixedFooter `json:"fixedFooter,omitempty"`
	// Header: The header of the card. A header usually contains a leading image
	// and a title. Headers always appear at the top of a card.
	Header *GoogleAppsCardV1CardHeader `json:"header,omitempty"`
	// Name: Name of the card. Used as a card identifier in card navigation. Google
	// Workspace add-ons (https://developers.google.com/workspace/add-ons):
	Name string `json:"name,omitempty"`
	// PeekCardHeader: When displaying contextual content, the peek card header
	// acts as a placeholder so that the user can navigate forward between the
	// homepage cards and the contextual cards. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons):
	PeekCardHeader *GoogleAppsCardV1CardHeader `json:"peekCardHeader,omitempty"`
	// SectionDividerStyle: The divider style between the header, sections and
	// footer.
	//
	// Possible values:
	//   "DIVIDER_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "SOLID_DIVIDER" - Default option. Render a solid divider.
	//   "NO_DIVIDER" - If set, no divider is rendered. This style completely
	// removes the divider from the layout. The result is equivalent to not adding
	// a divider at all.
	SectionDividerStyle string `json:"sectionDividerStyle,omitempty"`
	// Sections: Contains a collection of widgets. Each section has its own,
	// optional header. Sections are visually separated by a line divider. For an
	// example in Google Chat apps, see Define a section of a card
	// (https://developers.google.com/workspace/chat/design-components-card-dialog#define_a_section_of_a_card).
	Sections []*GoogleAppsCardV1Section `json:"sections,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CardActions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Card) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Card
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardAction: A card action is the action associated with the
// card. For example, an invoice card might include actions such as delete
// invoice, email invoice, or open the invoice in a browser. Google Workspace
// add-ons (https://developers.google.com/workspace/add-ons):
type GoogleAppsCardV1CardAction struct {
	// ActionLabel: The label that displays as the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`
	// OnClick: The `onClick` action for this action item.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionLabel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardFixedFooter: A persistent (sticky) footer that that
// appears at the bottom of the card. Setting `fixedFooter` without specifying
// a `primaryButton` or a `secondaryButton` causes an error. For Chat apps, you
// can use fixed footers in dialogs
// (https://developers.google.com/workspace/chat/dialogs), but not card
// messages
// (https://developers.google.com/workspace/chat/create-messages#create). For
// an example in Google Chat apps, see Add a persistent footer
// (https://developers.google.com/workspace/chat/design-components-card-dialog#add_a_persistent_footer).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1CardFixedFooter struct {
	// PrimaryButton: The primary button of the fixed footer. The button must be a
	// text button with text and color set.
	PrimaryButton *GoogleAppsCardV1Button `json:"primaryButton,omitempty"`
	// SecondaryButton: The secondary button of the fixed footer. The button must
	// be a text button with text and color set. If `secondaryButton` is set, you
	// must also set `primaryButton`.
	SecondaryButton *GoogleAppsCardV1Button `json:"secondaryButton,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PrimaryButton") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PrimaryButton") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CardFixedFooter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardFixedFooter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardHeader: Represents a card header. For an example in
// Google Chat apps, see Add a header
// (https://developers.google.com/workspace/chat/design-components-card-dialog#add_a_header).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1CardHeader struct {
	// ImageAltText: The alternative text of this image that's used for
	// accessibility.
	ImageAltText string `json:"imageAltText,omitempty"`
	// ImageType: The shape used to crop the image. Google Workspace add-ons and
	// Chat apps (https://developers.google.com/workspace/extend):
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For example,
	// a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3 image
	// becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`
	// ImageUrl: The HTTPS URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`
	// Subtitle: The subtitle of the card header. If specified, appears on its own
	// line below the `title`.
	Subtitle string `json:"subtitle,omitempty"`
	// Title: Required. The title of the card header. The header has a fixed
	// height: if both a title and subtitle are specified, each takes up one line.
	// If only the title is specified, it takes up both lines.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ImageAltText") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ImageAltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardHeader
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Carousel: A carousel, also known as a slider, rotates and
// displays a list of widgets in a slideshow format, with buttons navigating to
// the previous or next widget. For example, this is a JSON representation of a
// carousel that contains three text paragraph widgets. ``` { "carouselCards":
// [ { "widgets": [ { "textParagraph": { "text": "First text paragraph in
// carousel", } } ] }, { "widgets": [ { "textParagraph": { "text": "Second text
// paragraph in carousel", } } ] }, { "widgets": [ { "textParagraph": { "text":
// "Third text paragraph in carousel", } } ] } ] } ``` Google Chat apps
// (https://developers.google.com/workspace/chat):
type GoogleAppsCardV1Carousel struct {
	// CarouselCards: A list of cards included in the carousel.
	CarouselCards []*GoogleAppsCardV1CarouselCard `json:"carouselCards,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CarouselCards") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CarouselCards") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Carousel) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Carousel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CarouselCard: A card that can be displayed as a carousel
// item. Google Chat apps (https://developers.google.com/workspace/chat):
type GoogleAppsCardV1CarouselCard struct {
	// FooterWidgets: A list of widgets displayed at the bottom of the carousel
	// card. The widgets are displayed in the order that they are specified.
	FooterWidgets []*GoogleAppsCardV1NestedWidget `json:"footerWidgets,omitempty"`
	// Widgets: A list of widgets displayed in the carousel card. The widgets are
	// displayed in the order that they are specified.
	Widgets []*GoogleAppsCardV1NestedWidget `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FooterWidgets") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FooterWidgets") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CarouselCard) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CarouselCard
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Chip: A text, icon, or text and icon chip that users can
// click. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Chip struct {
	// AltText: The alternative text that's used for accessibility. Set descriptive
	// text that lets users know what the chip does. For example, if a chip opens a
	// hyperlink, write: "Opens a new browser tab and navigates to the Google Chat
	// developer documentation at https://developers.google.com/workspace/chat".
	AltText string `json:"altText,omitempty"`
	// Disabled: Whether the chip is in an inactive state and ignores user actions.
	// Defaults to `false`.
	Disabled bool `json:"disabled,omitempty"`
	// Enabled: Whether the chip is in an active state and responds to user
	// actions. Defaults to `true`. Deprecated. Use `disabled` instead.
	Enabled bool `json:"enabled,omitempty"`
	// Icon: The icon image. If both `icon` and `text` are set, then the icon
	// appears before the text.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`
	// Label: The text displayed inside the chip.
	Label string `json:"label,omitempty"`
	// OnClick: Optional. The action to perform when a user clicks the chip, such
	// as opening a hyperlink or running a custom function.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AltText") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Chip) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Chip
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ChipList: A list of chips layed out horizontally, which can
// either scroll horizontally or wrap to the next line. Google Workspace
// add-ons and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1ChipList struct {
	// Chips: An array of chips.
	Chips []*GoogleAppsCardV1Chip `json:"chips,omitempty"`
	// Layout: Specified chip list layout.
	//
	// Possible values:
	//   "LAYOUT_UNSPECIFIED" - Don't use. Unspecified.
	//   "WRAPPED" - Default value. The chip list wraps to the next line if there
	// isn't enough horizontal space.
	//   "HORIZONTAL_SCROLLABLE" - The chips scroll horizontally if they don't fit
	// in the available space.
	Layout string `json:"layout,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Chips") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Chips") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ChipList) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ChipList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CollapseControl: Represent an expand and collapse control.
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1CollapseControl struct {
	// CollapseButton: Optional. Define a customizable button to collapse the
	// section. Both expand_button and collapse_button field must be set. Only one
	// field set will not take into effect. If this field isn't set, the default
	// button is used.
	CollapseButton *GoogleAppsCardV1Button `json:"collapseButton,omitempty"`
	// ExpandButton: Optional. Define a customizable button to expand the section.
	// Both expand_button and collapse_button field must be set. Only one field set
	// will not take into effect. If this field isn't set, the default button is
	// used.
	ExpandButton *GoogleAppsCardV1Button `json:"expandButton,omitempty"`
	// HorizontalAlignment: The horizontal alignment of the expand and collapse
	// button.
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "START" - Default value. Aligns widgets to the start position of the
	// column. For left-to-right layouts, aligns to the left. For right-to-left
	// layouts, aligns to the right.
	//   "CENTER" - Aligns widgets to the center of the column.
	//   "END" - Aligns widgets to the end position of the column. For
	// left-to-right layouts, aligns widgets to the right. For right-to-left
	// layouts, aligns widgets to the left.
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CollapseButton") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CollapseButton") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CollapseControl) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CollapseControl
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Column: A column. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend)
type GoogleAppsCardV1Column struct {
	// HorizontalAlignment: Specifies whether widgets align to the left, right, or
	// center of a column.
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "START" - Default value. Aligns widgets to the start position of the
	// column. For left-to-right layouts, aligns to the left. For right-to-left
	// layouts, aligns to the right.
	//   "CENTER" - Aligns widgets to the center of the column.
	//   "END" - Aligns widgets to the end position of the column. For
	// left-to-right layouts, aligns widgets to the right. For right-to-left
	// layouts, aligns widgets to the left.
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`
	// HorizontalSizeStyle: Specifies how a column fills the width of the card.
	//
	// Possible values:
	//   "HORIZONTAL_SIZE_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "FILL_AVAILABLE_SPACE" - Default value. Column fills the available space,
	// up to 70% of the card's width. If both columns are set to
	// `FILL_AVAILABLE_SPACE`, each column fills 50% of the space.
	//   "FILL_MINIMUM_SPACE" - Column fills the least amount of space possible and
	// no more than 30% of the card's width.
	HorizontalSizeStyle string `json:"horizontalSizeStyle,omitempty"`
	// VerticalAlignment: Specifies whether widgets align to the top, bottom, or
	// center of a column.
	//
	// Possible values:
	//   "VERTICAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "CENTER" - Default value. Aligns widgets to the center of a column.
	//   "TOP" - Aligns widgets to the top of a column.
	//   "BOTTOM" - Aligns widgets to the bottom of a column.
	VerticalAlignment string `json:"verticalAlignment,omitempty"`
	// Widgets: An array of widgets included in a column. Widgets appear in the
	// order that they are specified.
	Widgets []*GoogleAppsCardV1Widgets `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "HorizontalAlignment") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HorizontalAlignment") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Column) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Column
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Columns: The `Columns` widget displays up to 2 columns in a
// card or dialog. You can add widgets to each column; the widgets appear in
// the order that they are specified. For an example in Google Chat apps, see
// Display cards and dialogs in columns
// (https://developers.google.com/workspace/chat/format-structure-card-dialog#display_cards_and_dialogs_in_columns).
// The height of each column is determined by the taller column. For example,
// if the first column is taller than the second column, both columns have the
// height of the first column. Because each column can contain a different
// number of widgets, you can't define rows or align widgets between the
// columns. Columns are displayed side-by-side. You can customize the width of
// each column using the `HorizontalSizeStyle` field. If the user's screen
// width is too narrow, the second column wraps below the first: * On web, the
// second column wraps if the screen width is less than or equal to 480 pixels.
// * On iOS devices, the second column wraps if the screen width is less than
// or equal to 300 pt. * On Android devices, the second column wraps if the
// screen width is less than or equal to 320 dp. To include more than two
// columns, or to use rows, use the `Grid` widget. Google Workspace add-ons and
// Chat apps (https://developers.google.com/workspace/extend): The add-on UIs
// that support columns include: * The dialog displayed when users open the
// add-on from an email draft. * The dialog displayed when users open the
// add-on from the **Add attachment** menu in a Google Calendar event.
type GoogleAppsCardV1Columns struct {
	// ColumnItems: An array of columns. You can include up to 2 columns in a card
	// or dialog.
	ColumnItems []*GoogleAppsCardV1Column `json:"columnItems,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ColumnItems") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ColumnItems") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Columns) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Columns
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CommonWidgetAction: Represents an action that is not
// specific to a widget. Only supported by Google Workspace Workflow, but not
// Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1CommonWidgetAction struct {
	// UpdateVisibilityAction: The action to update the visibility of a widget.
	UpdateVisibilityAction *GoogleAppsCardV1UpdateVisibilityAction `json:"updateVisibilityAction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "UpdateVisibilityAction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "UpdateVisibilityAction") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1CommonWidgetAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CommonWidgetAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Condition: Represents a condition that can be used to
// trigger an action. Only supported by Google Workspace Workflow, but not
// Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1Condition struct {
	// ActionRuleId: The unique identifier of the ActionRule.
	ActionRuleId string `json:"actionRuleId,omitempty"`
	// ExpressionDataCondition: The condition that is determined by the expression
	// data.
	ExpressionDataCondition *GoogleAppsCardV1ExpressionDataCondition `json:"expressionDataCondition,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionRuleId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionRuleId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Condition) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Condition
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DataSourceConfig: A configuration object that helps
// configure the data sources for a widget. Only supported by Google Workspace
// Workflow, but not Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1DataSourceConfig struct {
	// PlatformDataSource: The data is from a Google Workspace application.
	PlatformDataSource *GoogleAppsCardV1PlatformDataSource `json:"platformDataSource,omitempty"`
	// RemoteDataSource: The data is from a remote data provider.
	RemoteDataSource *GoogleAppsCardV1Action `json:"remoteDataSource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PlatformDataSource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PlatformDataSource") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1DataSourceConfig) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DataSourceConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DateTimePicker: Lets users input a date, a time, or both a
// date and a time. Supports form submission validation. When
// `Action.all_widgets_are_required` is set to `true` or this widget is
// specified in `Action.required_widgets`, the submission action is blocked
// unless a value is selected. For an example in Google Chat apps, see Let a
// user pick a date and time
// (https://developers.google.com/workspace/chat/design-interactive-card-dialog#let_a_user_pick_a_date_and_time).
// Users can input text or use the picker to select dates and times. If users
// input an invalid date or time, the picker shows an error that prompts users
// to input the information correctly. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1DateTimePicker struct {
	// HostAppDataSource: A data source that's unique to a Google Workspace host
	// application, such as Gmail emails, Google Calendar events, or Google Chat
	// messages. Only supported by Google Workspace Workflows, but not Google Chat
	// API or Google Workspace Add-ons.
	HostAppDataSource *HostAppDataSourceMarkup `json:"hostAppDataSource,omitempty"`
	// Label: The text that prompts users to input a date, a time, or a date and
	// time. For example, if users are scheduling an appointment, use a label such
	// as `Appointment date` or `Appointment date and time`.
	Label string `json:"label,omitempty"`
	// Name: The name by which the `DateTimePicker` is identified in a form input
	// event. For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Name string `json:"name,omitempty"`
	// OnChangeAction: Triggered when the user clicks **Save** or **Clear** from
	// the `DateTimePicker` interface.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`
	// TimezoneOffsetDate: The number representing the time zone offset from UTC,
	// in minutes. If set, the `value_ms_epoch` is displayed in the specified time
	// zone. If unset, the value defaults to the user's time zone setting.
	TimezoneOffsetDate int64 `json:"timezoneOffsetDate,omitempty"`
	// Type: Whether the widget supports inputting a date, a time, or the date and
	// time.
	//
	// Possible values:
	//   "DATE_AND_TIME" - Users input a date and time.
	//   "DATE_ONLY" - Users input a date.
	//   "TIME_ONLY" - Users input a time.
	Type string `json:"type,omitempty"`
	// ValueMsEpoch: The default value displayed in the widget, in milliseconds
	// since Unix epoch time (https://en.wikipedia.org/wiki/Unix_time). Specify the
	// value based on the type of picker (`DateTimePickerType`): * `DATE_AND_TIME`:
	// a calendar date and time in UTC. For example, to represent January 1, 2023
	// at 12:00 PM UTC, use `1672574400000`. * `DATE_ONLY`: a calendar date at
	// 00:00:00 UTC. For example, to represent January 1, 2023, use
	// `1672531200000`. * `TIME_ONLY`: a time in UTC. For example, to represent
	// 12:00 PM, use `43200000` (or `12 * 60 * 60 * 1000`).
	ValueMsEpoch int64 `json:"valueMsEpoch,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "HostAppDataSource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HostAppDataSource") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1DateTimePicker) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DateTimePicker
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DecoratedText: A widget that displays text with optional
// decorations such as a label above or below the text, an icon in front of the
// text, a selection widget, or a button after the text. For an example in
// Google Chat apps, see Display text with decorative text
// (https://developers.google.com/workspace/chat/add-text-image-card-dialog#display_text_with_decorative_elements).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1DecoratedText struct {
	// BottomLabel: The text that appears below `text`. Always wraps.
	BottomLabel string `json:"bottomLabel,omitempty"`
	// BottomLabelText: `TextParagraph` equivalent of `bottom_label`. Always wraps.
	// Allows for more complex formatting than `bottom_label`. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	BottomLabelText *GoogleAppsCardV1TextParagraph `json:"bottomLabelText,omitempty"`
	// Button: A button that a user can click to trigger an action.
	Button *GoogleAppsCardV1Button `json:"button,omitempty"`
	// ContentText: `TextParagraph` equivalent of `text`. Allows for more complex
	// formatting than `text`. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	ContentText *GoogleAppsCardV1TextParagraph `json:"contentText,omitempty"`
	// EndIcon: An icon displayed after the text. Supports built-in
	// (https://developers.google.com/workspace/chat/format-messages#builtinicons)
	// and custom
	// (https://developers.google.com/workspace/chat/format-messages#customicons)
	// icons.
	EndIcon *GoogleAppsCardV1Icon `json:"endIcon,omitempty"`
	// Icon: Deprecated in favor of `startIcon`.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`
	// OnClick: This action is triggered when users click `topLabel` or
	// `bottomLabel`.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// StartIcon: The icon displayed in front of the text.
	StartIcon *GoogleAppsCardV1Icon `json:"startIcon,omitempty"`
	// StartIconVerticalAlignment: Optional. Vertical alignment of the start icon.
	// If not set, the icon will be vertically centered. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	//
	// Possible values:
	//   "VERTICAL_ALIGNMENT_UNSPECIFIED" - Unspecified type. Do not use.
	//   "TOP" - Alignment to the top position.
	//   "MIDDLE" - Alignment to the middle position.
	//   "BOTTOM" - Alignment to the bottom position.
	StartIconVerticalAlignment string `json:"startIconVerticalAlignment,omitempty"`
	// SwitchControl: A switch widget that a user can click to change its state and
	// trigger an action.
	SwitchControl *GoogleAppsCardV1SwitchControl `json:"switchControl,omitempty"`
	// Text: Required. The primary text. Supports simple formatting. For more
	// information about formatting text, see Formatting text in Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Text string `json:"text,omitempty"`
	// TopLabel: The text that appears above `text`. Always truncates.
	TopLabel string `json:"topLabel,omitempty"`
	// TopLabelText: `TextParagraph` equivalent of `top_label`. Always truncates.
	// Allows for more complex formatting than `top_label`. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	TopLabelText *GoogleAppsCardV1TextParagraph `json:"topLabelText,omitempty"`
	// WrapText: The wrap text setting. If `true`, the text wraps and displays on
	// multiple lines. Otherwise, the text is truncated. Only applies to `text`,
	// not `topLabel` and `bottomLabel`.
	WrapText bool `json:"wrapText,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BottomLabel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1DecoratedText) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DecoratedText
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Divider: Displays a divider between widgets as a horizontal
// line. For an example in Google Chat apps, see Add a horizontal divider
// between widgets
// (https://developers.google.com/workspace/chat/format-structure-card-dialog#add_a_horizontal_divider_between_widgets).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend): For example, the following
// JSON creates a divider: ``` "divider": {} ```
type GoogleAppsCardV1Divider struct {
}

// GoogleAppsCardV1EventAction: Represents an actionthat can be performed on an
// ui element. Only supported by Google Workspace Workflow, but not Google Chat
// apps or Google Workspace add-ons.
type GoogleAppsCardV1EventAction struct {
	// ActionRuleId: The unique identifier of the ActionRule.
	ActionRuleId string `json:"actionRuleId,omitempty"`
	// CommonWidgetAction: Common widget action.
	CommonWidgetAction *GoogleAppsCardV1CommonWidgetAction `json:"commonWidgetAction,omitempty"`
	// PostEventTriggers: The list of triggers that will be triggered after the
	// EventAction is executed.
	PostEventTriggers []*GoogleAppsCardV1Trigger `json:"postEventTriggers,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionRuleId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionRuleId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1EventAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1EventAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ExpressionData: Represents the data that is used to evaluate
// an expression. Only supported by Google Workspace Workflow, but not Google
// Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1ExpressionData struct {
	// Conditions: The list of conditions that are determined by the expression
	// evaluation result.
	Conditions []*GoogleAppsCardV1Condition `json:"conditions,omitempty"`
	// EventActions: The list of actions that the ExpressionData can be used.
	EventActions []*GoogleAppsCardV1EventAction `json:"eventActions,omitempty"`
	// Expression: The uncompiled expression.
	Expression string `json:"expression,omitempty"`
	// Id: The unique identifier of the ExpressionData.
	Id string `json:"id,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Conditions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Conditions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ExpressionData) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ExpressionData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ExpressionDataCondition: Represents a condition that is
// evaluated using CEL. Only supported by Google Workspace Workflow, but not
// Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1ExpressionDataCondition struct {
	// ConditionType: The type of the condition.
	//
	// Possible values:
	//   "CONDITION_TYPE_UNSPECIFIED" - Unspecified condition type.
	//   "EXPRESSION_EVALUATION_SUCCESS" - The expression evaluation was
	// successful.
	//   "EXPRESSION_EVALUATION_FAILURE" - The expression evaluation was
	// unsuccessful.
	ConditionType string `json:"conditionType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ConditionType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ConditionType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ExpressionDataCondition) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ExpressionDataCondition
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Grid: Displays a grid with a collection of items. Items can
// only include text or images. For responsive columns, or to include more than
// text or images, use `Columns`. For an example in Google Chat apps, see
// Display a Grid with a collection of items
// (https://developers.google.com/workspace/chat/format-structure-card-dialog#display_a_grid_with_a_collection_of_items).
// A grid supports any number of columns and items. The number of rows is
// determined by items divided by columns. A grid with 10 items and 2 columns
// has 5 rows. A grid with 11 items and 2 columns has 6 rows. Google Workspace
// add-ons and Chat apps (https://developers.google.com/workspace/extend): For
// example, the following JSON creates a 2 column grid with a single item: ```
// "grid": { "title": "A fine collection of items", "columnCount": 2,
// "borderStyle": { "type": "STROKE", "cornerRadius": 4 }, "items": [ {
// "image": { "imageUri": "https://www.example.com/image.png", "cropStyle": {
// "type": "SQUARE" }, "borderStyle": { "type": "STROKE" } }, "title": "An
// item", "textAlignment": "CENTER" } ], "onClick": { "openLink": { "url":
// "https://www.example.com" } } } ```
type GoogleAppsCardV1Grid struct {
	// BorderStyle: The border style to apply to each grid item.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`
	// ColumnCount: The number of columns to display in the grid. A default value
	// is used if this field isn't specified, and that default value is different
	// depending on where the grid is shown (dialog versus companion).
	ColumnCount int64 `json:"columnCount,omitempty"`
	// Items: The items to display in the grid.
	Items []*GoogleAppsCardV1GridItem `json:"items,omitempty"`
	// OnClick: This callback is reused by each individual grid item, but with the
	// item's identifier and index in the items list added to the callback's
	// parameters.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// Title: The text that displays in the grid header.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BorderStyle") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BorderStyle") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Grid) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Grid
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1GridItem: Represents an item in a grid layout. Items can
// contain text, an image, or both text and an image. Google Workspace add-ons
// and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1GridItem struct {
	// Id: A user-specified identifier for this grid item. This identifier is
	// returned in the parent grid's `onClick` callback parameters.
	Id string `json:"id,omitempty"`
	// Image: The image that displays in the grid item.
	Image *GoogleAppsCardV1ImageComponent `json:"image,omitempty"`
	// Layout: The layout to use for the grid item.
	//
	// Possible values:
	//   "GRID_ITEM_LAYOUT_UNSPECIFIED" - Don't use. Unspecified.
	//   "TEXT_BELOW" - The title and subtitle are shown below the grid item's
	// image.
	//   "TEXT_ABOVE" - The title and subtitle are shown above the grid item's
	// image.
	Layout string `json:"layout,omitempty"`
	// Subtitle: The grid item's subtitle.
	Subtitle string `json:"subtitle,omitempty"`
	// Title: The grid item's title.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1GridItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1GridItem
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Icon: An icon displayed in a widget on a card. For an
// example in Google Chat apps, see Add an icon
// (https://developers.google.com/workspace/chat/add-text-image-card-dialog#add_an_icon).
// Supports built-in
// (https://developers.google.com/workspace/chat/format-messages#builtinicons)
// and custom
// (https://developers.google.com/workspace/chat/format-messages#customicons)
// icons. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Icon struct {
	// AltText: Optional. A description of the icon used for accessibility. If
	// unspecified, the default value `Button` is provided. As a best practice, you
	// should set a helpful description for what the icon displays, and if
	// applicable, what it does. For example, `A user's account portrait`, or
	// `Opens a new browser tab and navigates to the Google Chat developer
	// documentation at https://developers.google.com/workspace/chat`. If the icon
	// is set in a `Button`, the `altText` appears as helper text when the user
	// hovers over the button. However, if the button also sets `text`, the icon's
	// `altText` is ignored.
	AltText string `json:"altText,omitempty"`
	// IconUrl: Display a custom icon hosted at an HTTPS URL. For example: ```
	// "iconUrl":
	// "https://developers.google.com/workspace/chat/images/quickstart-app-avatar.pn
	// g" ``` Supported file types include `.png` and `.jpg`.
	IconUrl string `json:"iconUrl,omitempty"`
	// ImageType: The crop style applied to the image. In some cases, applying a
	// `CIRCLE` crop causes the image to be drawn larger than a built-in icon.
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For example,
	// a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3 image
	// becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`
	// KnownIcon: Display one of the built-in icons provided by Google Workspace.
	// For example, to display an airplane icon, specify `AIRPLANE`. For a bus,
	// specify `BUS`. For a full list of supported icons, see built-in icons
	// (https://developers.google.com/workspace/chat/format-messages#builtinicons).
	KnownIcon string `json:"knownIcon,omitempty"`
	// MaterialIcon: Display one of the Google Material Icons
	// (https://fonts.google.com/icons). For example, to display a checkbox icon
	// (https://fonts.google.com/icons?selected=Material%20Symbols%20Outlined%3Acheck_box%3AFILL%400%3Bwght%40400%3BGRAD%400%3Bopsz%4048),
	// use ``` "material_icon": { "name": "check_box" } ``` Google Chat apps
	// (https://developers.google.com/workspace/chat):
	MaterialIcon *GoogleAppsCardV1MaterialIcon `json:"materialIcon,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AltText") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Icon) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Icon
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Image: An image that is specified by a URL and can have an
// `onClick` action. For an example, see Add an image
// (https://developers.google.com/workspace/chat/add-text-image-card-dialog#add_an_image).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Image struct {
	// AltText: The alternative text of this image that's used for accessibility.
	AltText string `json:"altText,omitempty"`
	// ImageUrl: The HTTPS URL that hosts the image. For example: ```
	// https://developers.google.com/workspace/chat/images/quickstart-app-avatar.png
	// ```
	ImageUrl string `json:"imageUrl,omitempty"`
	// OnClick: When a user clicks the image, the click triggers this action.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AltText") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Image) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Image
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageComponent: Represents an image. Google Workspace
// add-ons and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1ImageComponent struct {
	// AltText: The accessibility label for the image.
	AltText string `json:"altText,omitempty"`
	// BorderStyle: The border style to apply to the image.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`
	// CropStyle: The crop style to apply to the image.
	CropStyle *GoogleAppsCardV1ImageCropStyle `json:"cropStyle,omitempty"`
	// ImageUri: The image URL.
	ImageUri string `json:"imageUri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AltText") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AltText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ImageComponent) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageComponent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageCropStyle: Represents the crop style applied to an
// image. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend): For example, here's how to
// apply a 16:9 aspect ratio: ``` cropStyle { "type": "RECTANGLE_CUSTOM",
// "aspectRatio": 16/9 } ```
type GoogleAppsCardV1ImageCropStyle struct {
	// AspectRatio: The aspect ratio to use if the crop type is `RECTANGLE_CUSTOM`.
	// For example, here's how to apply a 16:9 aspect ratio: ``` cropStyle {
	// "type": "RECTANGLE_CUSTOM", "aspectRatio": 16/9 } ```
	AspectRatio float64 `json:"aspectRatio,omitempty"`
	// Type: The crop type.
	//
	// Possible values:
	//   "IMAGE_CROP_TYPE_UNSPECIFIED" - Don't use. Unspecified.
	//   "SQUARE" - Default value. Applies a square crop.
	//   "CIRCLE" - Applies a circular crop.
	//   "RECTANGLE_CUSTOM" - Applies a rectangular crop with a custom aspect
	// ratio. Set the custom aspect ratio with `aspectRatio`.
	//   "RECTANGLE_4_3" - Applies a rectangular crop with a 4:3 aspect ratio.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AspectRatio") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1ImageCropStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *GoogleAppsCardV1ImageCropStyle) UnmarshalJSON(data []byte) error {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// GoogleAppsCardV1MaterialIcon: A Google Material Icon
// (https://fonts.google.com/icons), which includes over 2500+ options. For
// example, to display a checkbox icon
// (https://fonts.google.com/icons?selected=Material%20Symbols%20Outlined%3Acheck_box%3AFILL%400%3Bwght%40400%3BGRAD%400%3Bopsz%4048)
// with customized weight and grade, write the following: ``` { "name":
// "check_box", "fill": true, "weight": 300, "grade": -25 } ``` Google Chat
// apps (https://developers.google.com/workspace/chat):
type GoogleAppsCardV1MaterialIcon struct {
	// Fill: Whether the icon renders as filled. Default value is false. To preview
	// different icon settings, go to Google Font Icons
	// (https://fonts.google.com/icons) and adjust the settings under
	// **Customize**.
	Fill bool `json:"fill,omitempty"`
	// Grade: Weight and grade affect a symbol’s thickness. Adjustments to grade
	// are more granular than adjustments to weight and have a small impact on the
	// size of the symbol. Choose from {-25, 0, 200}. If absent, default value is
	// 0. If any other value is specified, the default value is used. To preview
	// different icon settings, go to Google Font Icons
	// (https://fonts.google.com/icons) and adjust the settings under
	// **Customize**.
	Grade int64 `json:"grade,omitempty"`
	// Name: The icon name defined in the Google Material Icon
	// (https://fonts.google.com/icons), for example, `check_box`. Any invalid
	// names are abandoned and replaced with empty string and results in the icon
	// failing to render.
	Name string `json:"name,omitempty"`
	// Weight: The stroke weight of the icon. Choose from {100, 200, 300, 400, 500,
	// 600, 700}. If absent, default value is 400. If any other value is specified,
	// the default value is used. To preview different icon settings, go to Google
	// Font Icons (https://fonts.google.com/icons) and adjust the settings under
	// **Customize**.
	Weight int64 `json:"weight,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Fill") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Fill") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1MaterialIcon) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1MaterialIcon
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1NestedWidget: A list of widgets that can be displayed in a
// containing layout, such as a `CarouselCard`. Google Chat apps
// (https://developers.google.com/workspace/chat):
type GoogleAppsCardV1NestedWidget struct {
	// ButtonList: A button list widget.
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`
	// Image: An image widget.
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`
	// TextParagraph: A text paragraph widget.
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ButtonList") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1NestedWidget) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1NestedWidget
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OnClick: Represents how to respond when users click an
// interactive element on a card, such as a button. Google Workspace add-ons
// and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1OnClick struct {
	// Action: If specified, an action is triggered by this `onClick`.
	Action *GoogleAppsCardV1Action `json:"action,omitempty"`
	// Card: A new card is pushed to the card stack after clicking if specified.
	// Google Workspace add-ons (https://developers.google.com/workspace/add-ons):
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`
	// OpenDynamicLinkAction: An add-on triggers this action when the action needs
	// to open a link. This differs from the `open_link` above in that this needs
	// to talk to server to get the link. Thus some preparation work is required
	// for web client to do before the open link action response comes back. Google
	// Workspace add-ons (https://developers.google.com/workspace/add-ons):
	OpenDynamicLinkAction *GoogleAppsCardV1Action `json:"openDynamicLinkAction,omitempty"`
	// OpenLink: If specified, this `onClick` triggers an open link action.
	OpenLink *GoogleAppsCardV1OpenLink `json:"openLink,omitempty"`
	// OverflowMenu: If specified, this `onClick` opens an overflow menu.
	OverflowMenu *GoogleAppsCardV1OverflowMenu `json:"overflowMenu,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OnClick
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OpenLink: Represents an `onClick` event that opens a
// hyperlink. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1OpenLink struct {
	// OnClose: Whether the client forgets about a link after opening it, or
	// observes it until the window closes. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons):
	//
	// Possible values:
	//   "NOTHING" - Default value. The card doesn't reload; nothing happens.
	//   "RELOAD" - Reloads the card after the child window closes. If used in
	// conjunction with
	// [`OpenAs.OVERLAY`](https://developers.google.com/workspace/add-ons/reference/
	// rpc/google.apps.card.v1#openas), the child window acts as a modal dialog and
	// the parent card is blocked until the child window closes.
	OnClose string `json:"onClose,omitempty"`
	// OpenAs: How to open a link. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons):
	//
	// Possible values:
	//   "FULL_SIZE" - The link opens as a full-size window (if that's the frame
	// used by the client).
	//   "OVERLAY" - The link opens as an overlay, such as a pop-up.
	OpenAs string `json:"openAs,omitempty"`
	// Url: The URL to open.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OnClose") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OnClose") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OpenLink
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OverflowMenu: A widget that presents a pop-up menu with one
// or more actions that users can invoke. For example, showing non-primary
// actions in a card. You can use this widget when actions don't fit in the
// available space. To use, specify this widget in the `OnClick` action of
// widgets that support it. For example, in a `Button`. Google Workspace
// add-ons and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1OverflowMenu struct {
	// Items: Required. The list of menu options.
	Items []*GoogleAppsCardV1OverflowMenuItem `json:"items,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1OverflowMenu) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OverflowMenu
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OverflowMenuItem: An option that users can invoke in an
// overflow menu. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1OverflowMenuItem struct {
	// Disabled: Whether the menu option is disabled. Defaults to false.
	Disabled bool `json:"disabled,omitempty"`
	// OnClick: Required. The action invoked when a menu option is selected. This
	// `OnClick` cannot contain an `OverflowMenu`, any specified `OverflowMenu` is
	// dropped and the menu item disabled.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`
	// StartIcon: The icon displayed in front of the text.
	StartIcon *GoogleAppsCardV1Icon `json:"startIcon,omitempty"`
	// Text: Required. The text that identifies or describes the item to users.
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Disabled") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Disabled") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1OverflowMenuItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OverflowMenuItem
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1PlatformDataSource: For a `SelectionInput` widget that uses
// a multiselect menu, a data source from Google Workspace. Used to populate
// items in a multiselect menu. Google Chat apps
// (https://developers.google.com/workspace/chat):
type GoogleAppsCardV1PlatformDataSource struct {
	// CommonDataSource: A data source shared by all Google Workspace applications,
	// such as users in a Google Workspace organization.
	//
	// Possible values:
	//   "UNKNOWN" - Default value. Don't use.
	//   "USER" - Google Workspace users. The user can only view and select users
	// from their Google Workspace organization.
	CommonDataSource string `json:"commonDataSource,omitempty"`
	// HostAppDataSource: A data source that's unique to a Google Workspace host
	// application, such spaces in Google Chat. This field supports the Google API
	// Client Libraries but isn't available in the Cloud Client Libraries. To learn
	// more, see Install the client libraries
	// (https://developers.google.com/workspace/chat/libraries).
	HostAppDataSource *HostAppDataSourceMarkup `json:"hostAppDataSource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CommonDataSource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CommonDataSource") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1PlatformDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1PlatformDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Section: A section contains a collection of widgets that are
// rendered vertically in the order that they're specified. Google Workspace
// add-ons and Chat apps (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Section struct {
	// CollapseControl: Optional. Define the expand and collapse button of the
	// section. This button will be shown only if the section is collapsible. If
	// this field isn't set, the default button is used.
	CollapseControl *GoogleAppsCardV1CollapseControl `json:"collapseControl,omitempty"`
	// Collapsible: Indicates whether this section is collapsible. Collapsible
	// sections hide some or all widgets, but users can expand the section to
	// reveal the hidden widgets by clicking **Show more**. Users can hide the
	// widgets again by clicking **Show less**. To determine which widgets are
	// hidden, specify `uncollapsibleWidgetsCount`.
	Collapsible bool `json:"collapsible,omitempty"`
	// Header: Text that appears at the top of a section. Supports simple HTML
	// formatted text. For more information about formatting text, see Formatting
	// text in Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Header string `json:"header,omitempty"`
	// Id: A unique ID assigned to the section that's used to identify the section
	// to be mutated. The ID has a character limit of 64 characters and should be
	// in the format of `[a-zA-Z0-9-]+`. Only supported by Google Workspace
	// Workflow, but not Google Chat apps or Google Workspace add-ons.
	Id string `json:"id,omitempty"`
	// UncollapsibleWidgetsCount: The number of uncollapsible widgets which remain
	// visible even when a section is collapsed. For example, when a section
	// contains five widgets and the `uncollapsibleWidgetsCount` is set to `2`, the
	// first two widgets are always shown and the last three are collapsed by
	// default. The `uncollapsibleWidgetsCount` is taken into account only when
	// `collapsible` is `true`.
	UncollapsibleWidgetsCount int64 `json:"uncollapsibleWidgetsCount,omitempty"`
	// Widgets: All the widgets in the section. Must contain at least one widget.
	Widgets []*GoogleAppsCardV1Widget `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CollapseControl") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CollapseControl") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Section) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Section
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionInput: A widget that creates one or more UI items
// that users can select. Supports form submission validation for `dropdown`
// and `multiselect` menus only. When `Action.all_widgets_are_required` is set
// to `true` or this widget is specified in `Action.required_widgets`, the
// submission action is blocked unless a value is selected. For example, a
// dropdown menu or checkboxes. You can use this widget to collect data that
// can be predicted or enumerated. For an example in Google Chat apps, see Add
// selectable UI elements
// (/workspace/chat/design-interactive-card-dialog#add_selectable_ui_elements).
// Chat apps can process the value of items that users select or input. For
// details about working with form inputs, see Receive form data
// (https://developers.google.com/workspace/chat/read-form-data). To collect
// undefined or abstract data from users, use the TextInput widget. Google
// Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1SelectionInput struct {
	// DataSourceConfigs: Optional. The data source configs for the selection
	// control. This field provides more fine-grained control over the data source.
	// If specified, the `multi_select_max_selected_items` field,
	// `multi_select_min_query_length` field, `external_data_source` field and
	// `platform_data_source` field are ignored. Only supported by Google Workspace
	// Workflow, but not Google Chat apps or Google Workspace add-ons.
	DataSourceConfigs []*GoogleAppsCardV1DataSourceConfig `json:"dataSourceConfigs,omitempty"`
	// ExternalDataSource: An external data source, such as a relational database.
	ExternalDataSource *GoogleAppsCardV1Action `json:"externalDataSource,omitempty"`
	// HintText: Optional. Text that appears below the selection input field meant
	// to assist users by prompting them to enter a certain value. This text is
	// always visible. Only supported by Google Workspace Workflows, but not Google
	// Chat API or Google Workspace Add-ons.
	HintText string `json:"hintText,omitempty"`
	// Items: An array of selectable items. For example, an array of radio buttons
	// or checkboxes. Supports up to 100 items.
	Items []*GoogleAppsCardV1SelectionItem `json:"items,omitempty"`
	// Label: The text that appears above the selection input field in the user
	// interface. Specify text that helps the user enter the information your app
	// needs. For example, if users are selecting the urgency of a work ticket from
	// a drop-down menu, the label might be "Urgency" or "Select urgency".
	Label string `json:"label,omitempty"`
	// MultiSelectMaxSelectedItems: For multiselect menus, the maximum number of
	// items that a user can select. Minimum value is 1 item. If unspecified,
	// defaults to 3 items.
	MultiSelectMaxSelectedItems int64 `json:"multiSelectMaxSelectedItems,omitempty"`
	// MultiSelectMinQueryLength: For multiselect menus, the number of text
	// characters that a user inputs before the menu returns suggested selection
	// items. If unset, the multiselect menu uses the following default values: *
	// If the menu uses a static array of `SelectionInput` items, defaults to 0
	// characters and immediately populates items from the array. * If the menu
	// uses a dynamic data source (`multi_select_data_source`), defaults to 3
	// characters before querying the data source to return suggested items.
	MultiSelectMinQueryLength int64 `json:"multiSelectMinQueryLength,omitempty"`
	// Name: Required. The name that identifies the selection input in a form input
	// event. For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Name string `json:"name,omitempty"`
	// OnChangeAction: If specified, the form is submitted when the selection
	// changes. If not specified, you must specify a separate button that submits
	// the form. For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`
	// PlatformDataSource: A data source from Google Workspace.
	PlatformDataSource *GoogleAppsCardV1PlatformDataSource `json:"platformDataSource,omitempty"`
	// Type: The type of items that are displayed to users in a `SelectionInput`
	// widget. Selection types support different types of interactions. For
	// example, users can select one or more checkboxes, but they can only select
	// one value from a dropdown menu.
	//
	// Possible values:
	//   "CHECK_BOX" - A set of checkboxes. Users can select one or more
	// checkboxes.
	//   "RADIO_BUTTON" - A set of radio buttons. Users can select one radio
	// button.
	//   "SWITCH" - A set of switches. Users can turn on one or more switches.
	//   "DROPDOWN" - A dropdown menu. Users can select one item from the menu.
	//   "MULTI_SELECT" - A menu with a text box. Users can type and select one or
	// more items. For Google Workspace add-ons, you must populate items using a
	// static array of `SelectionItem` objects. For Google Chat apps, you can also
	// populate items using a dynamic data source and autosuggest items as users
	// type in the menu. For example, users can start typing the name of a Google
	// Chat space and the widget autosuggests the space. To dynamically populate
	// items for a multiselect menu, use one of the following types of data
	// sources: * Google Workspace data: Items are populated using data from Google
	// Workspace, such as Google Workspace users or Google Chat spaces. * External
	// data: Items are populated from an external data source outside of Google
	// Workspace. For examples of how to implement multiselect menus for Chat apps,
	// see [Add a multiselect
	// menu](https://developers.google.com/workspace/chat/design-interactive-card-di
	// alog#multiselect-menu). [Google Workspace add-ons and Chat
	// apps](https://developers.google.com/workspace/extend):
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DataSourceConfigs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataSourceConfigs") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1SelectionInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionItem: An item that users can select in a selection
// input, such as a checkbox or switch. Supports up to 100 items. Google
// Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1SelectionItem struct {
	// BottomText: For multiselect menus, a text description or label that's
	// displayed below the item's `text` field.
	BottomText string `json:"bottomText,omitempty"`
	// Selected: Whether the item is selected by default. If the selection input
	// only accepts one value (such as for radio buttons or a dropdown menu), only
	// set this field for one item.
	Selected     bool   `json:"selected,omitempty"`
	StartIconUri string `json:"startIconUri,omitempty"`
	// Text: The text that identifies or describes the item to users.
	Text string `json:"text,omitempty"`
	// Value: The value associated with this item. The client should use this as a
	// form input value. For details about working with form inputs, see Receive
	// form data (https://developers.google.com/workspace/chat/read-form-data).
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BottomText") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BottomText") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1SelectionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionItem
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SuggestionItem: One suggested value that users can enter in
// a text input field. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1SuggestionItem struct {
	// Text: The value of a suggested input to a text input field. This is
	// equivalent to what users enter themselves.
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Text") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Text") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1SuggestionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SuggestionItem
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Suggestions: Suggested values that users can enter. These
// values appear when users click inside the text input field. As users type,
// the suggested values dynamically filter to match what the users have typed.
// For example, a text input field for programming language might suggest Java,
// JavaScript, Python, and C++. When users start typing `Jav`, the list of
// suggestions filters to show `Java` and `JavaScript`. Suggested values help
// guide users to enter values that your app can make sense of. When referring
// to JavaScript, some users might enter `javascript` and others `java script`.
// Suggesting `JavaScript` can standardize how users interact with your app.
// When specified, `TextInput.type` is always `SINGLE_LINE`, even if it's set
// to `MULTIPLE_LINE`. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Suggestions struct {
	// Items: A list of suggestions used for autocomplete recommendations in text
	// input fields.
	Items []*GoogleAppsCardV1SuggestionItem `json:"items,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Suggestions) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Suggestions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SwitchControl: Either a toggle-style switch or a checkbox
// inside a `decoratedText` widget. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend): Only supported in the
// `decoratedText` widget.
type GoogleAppsCardV1SwitchControl struct {
	// ControlType: How the switch appears in the user interface. Google Workspace
	// add-ons and Chat apps (https://developers.google.com/workspace/extend):
	//
	// Possible values:
	//   "SWITCH" - A toggle-style switch.
	//   "CHECKBOX" - Deprecated in favor of `CHECK_BOX`.
	//   "CHECK_BOX" - A checkbox.
	ControlType string `json:"controlType,omitempty"`
	// Name: The name by which the switch widget is identified in a form input
	// event. For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Name string `json:"name,omitempty"`
	// OnChangeAction: The action to perform when the switch state is changed, such
	// as what function to run.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`
	// Selected: When `true`, the switch is selected.
	Selected bool `json:"selected,omitempty"`
	// Value: The value entered by a user, returned as part of a form input event.
	// For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ControlType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ControlType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1SwitchControl) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SwitchControl
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextInput: A field in which users can enter text. Supports
// suggestions and on-change actions. Supports form submission validation. When
// `Action.all_widgets_are_required` is set to `true` or this widget is
// specified in `Action.required_widgets`, the submission action is blocked
// unless a value is entered. For an example in Google Chat apps, see Add a
// field in which a user can enter text
// (https://developers.google.com/workspace/chat/design-interactive-card-dialog#add_a_field_in_which_a_user_can_enter_text).
// Chat apps receive and can process the value of entered text during form
// input events. For details about working with form inputs, see Receive form
// data (https://developers.google.com/workspace/chat/read-form-data). When you
// need to collect undefined or abstract data from users, use a text input. To
// collect defined or enumerated data from users, use the SelectionInput
// widget. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1TextInput struct {
	// AutoCompleteAction: Optional. Specify what action to take when the text
	// input field provides suggestions to users who interact with it. If
	// unspecified, the suggestions are set by `initialSuggestions` and are
	// processed by the client. If specified, the app takes the action specified
	// here, such as running a custom function. Google Workspace add-ons
	// (https://developers.google.com/workspace/add-ons):
	AutoCompleteAction *GoogleAppsCardV1Action `json:"autoCompleteAction,omitempty"`
	// HintText: Text that appears below the text input field meant to assist users
	// by prompting them to enter a certain value. This text is always visible.
	// Required if `label` is unspecified. Otherwise, optional.
	HintText string `json:"hintText,omitempty"`
	// HostAppDataSource: A data source that's unique to a Google Workspace host
	// application, such as Gmail emails, Google Calendar events, or Google Chat
	// messages. Only supported by Google Workspace Workflow, but not Google Chat
	// apps or Google Workspace add-ons.
	HostAppDataSource *HostAppDataSourceMarkup `json:"hostAppDataSource,omitempty"`
	// InitialSuggestions: Suggested values that users can enter. These values
	// appear when users click inside the text input field. As users type, the
	// suggested values dynamically filter to match what the users have typed. For
	// example, a text input field for programming language might suggest Java,
	// JavaScript, Python, and C++. When users start typing `Jav`, the list of
	// suggestions filters to show just `Java` and `JavaScript`. Suggested values
	// help guide users to enter values that your app can make sense of. When
	// referring to JavaScript, some users might enter `javascript` and others
	// `java script`. Suggesting `JavaScript` can standardize how users interact
	// with your app. When specified, `TextInput.type` is always `SINGLE_LINE`,
	// even if it's set to `MULTIPLE_LINE`. Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	InitialSuggestions *GoogleAppsCardV1Suggestions `json:"initialSuggestions,omitempty"`
	// Label: The text that appears above the text input field in the user
	// interface. Specify text that helps the user enter the information your app
	// needs. For example, if you are asking someone's name, but specifically need
	// their surname, write `surname` instead of `name`. Required if `hintText` is
	// unspecified. Otherwise, optional.
	Label string `json:"label,omitempty"`
	// Name: The name by which the text input is identified in a form input event.
	// For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Name string `json:"name,omitempty"`
	// OnChangeAction: What to do when a change occurs in the text input field. For
	// example, a user adding to the field or deleting text. Examples of actions to
	// take include running a custom function or opening a dialog
	// (https://developers.google.com/workspace/chat/dialogs) in Google Chat.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`
	// PlaceholderText: Text that appears in the text input field when the field is
	// empty. Use this text to prompt users to enter a value. For example, `Enter a
	// number from 0 to 100`. Google Chat apps
	// (https://developers.google.com/workspace/chat):
	PlaceholderText string `json:"placeholderText,omitempty"`
	// Type: How a text input field appears in the user interface. For example,
	// whether the field is single or multi-line.
	//
	// Possible values:
	//   "SINGLE_LINE" - The text input field has a fixed height of one line.
	//   "MULTIPLE_LINE" - The text input field has a fixed height of multiple
	// lines.
	Type string `json:"type,omitempty"`
	// Validation: Specify the input format validation necessary for this text
	// field. Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	Validation *GoogleAppsCardV1Validation `json:"validation,omitempty"`
	// Value: The value entered by a user, returned as part of a form input event.
	// For details about working with form inputs, see Receive form data
	// (https://developers.google.com/workspace/chat/read-form-data).
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AutoCompleteAction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutoCompleteAction") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1TextInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextParagraph: A paragraph of text that supports formatting.
// For an example in Google Chat apps, see Add a paragraph of formatted text
// (https://developers.google.com/workspace/chat/add-text-image-card-dialog#add_a_paragraph_of_formatted_text).
// For more information about formatting text, see Formatting text in Google
// Chat apps
// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
// and Formatting text in Google Workspace add-ons
// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
// Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1TextParagraph struct {
	// MaxLines: The maximum number of lines of text that are displayed in the
	// widget. If the text exceeds the specified maximum number of lines, the
	// excess content is concealed behind a **show more** button. If the text is
	// equal or shorter than the specified maximum number of lines, a **show more**
	// button isn't displayed. The default value is 0, in which case all context is
	// displayed. Negative values are ignored.
	MaxLines int64 `json:"maxLines,omitempty"`
	// Text: The text that's shown in the widget.
	Text string `json:"text,omitempty"`
	// TextSyntax: The syntax of the text. If not set, the text is rendered as
	// HTML. Google Chat apps (https://developers.google.com/workspace/chat):
	//
	// Possible values:
	//   "TEXT_SYNTAX_UNSPECIFIED" - The text is rendered as HTML if unspecified.
	//   "HTML" - The text is rendered as HTML. This is the default value.
	//   "MARKDOWN" - The text is rendered as Markdown.
	TextSyntax string `json:"textSyntax,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MaxLines") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MaxLines") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextParagraph
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Trigger: Represents a trigger. Only supported by Google
// Workspace Workflow, but not Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1Trigger struct {
	// ActionRuleId: The unique identifier of the ActionRule.
	ActionRuleId string `json:"actionRuleId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ActionRuleId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActionRuleId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Trigger) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Trigger
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1UpdateVisibilityAction: Represents an action that updates
// the visibility of a widget. Only supported by Google Workspace Workflow, but
// not Google Chat apps or Google Workspace add-ons.
type GoogleAppsCardV1UpdateVisibilityAction struct {
	// Visibility: The new visibility.
	//
	// Possible values:
	//   "VISIBILITY_UNSPECIFIED" - Unspecified visibility. Do not use.
	//   "VISIBLE" - The UI element is visible.
	//   "HIDDEN" - The UI element is hidden.
	Visibility string `json:"visibility,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Visibility") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Visibility") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1UpdateVisibilityAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1UpdateVisibilityAction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Validation: Represents the necessary data for validating the
// widget it's attached to. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend):
type GoogleAppsCardV1Validation struct {
	// CharacterLimit: Specify the character limit for text input widgets. Note
	// that this is only used for text input and is ignored for other widgets.
	// Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend):
	CharacterLimit int64 `json:"characterLimit,omitempty"`
	// InputType: Specify the type of the input widgets. Google Workspace add-ons
	// and Chat apps (https://developers.google.com/workspace/extend):
	//
	// Possible values:
	//   "INPUT_TYPE_UNSPECIFIED" - Unspecified type. Do not use.
	//   "TEXT" - Regular text that accepts all characters.
	//   "INTEGER" - An integer value.
	//   "FLOAT" - A float value.
	//   "EMAIL" - An email address.
	//   "EMOJI_PICKER" - A emoji selected from system-provided emoji picker.
	InputType string `json:"inputType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CharacterLimit") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CharacterLimit") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Validation) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Validation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Widget: Each card is made up of widgets. A widget is a
// composite object that can represent one of text, images, buttons, and other
// object types.
type GoogleAppsCardV1Widget struct {
	// ButtonList: A list of buttons. For example, the following JSON creates two
	// buttons. The first is a blue text button and the second is an image button
	// that opens a link: ``` "buttonList": { "buttons": [ { "text": "Edit",
	// "color": { "red": 0, "green": 0, "blue": 1, }, "disabled": true, }, {
	// "icon": { "knownIcon": "INVITE", "altText": "check calendar" }, "onClick": {
	// "openLink": { "url": "https://example.com/calendar" } } } ] } ```
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`
	// Carousel: A carousel contains a collection of nested widgets. For example,
	// this is a JSON representation of a carousel that contains two text
	// paragraphs. ``` { "widgets": [ { "textParagraph": { "text": "First text
	// paragraph in the carousel." } }, { "textParagraph": { "text": "Second text
	// paragraph in the carousel." } } ] } ```
	Carousel *GoogleAppsCardV1Carousel `json:"carousel,omitempty"`
	// ChipList: A list of chips. For example, the following JSON creates two
	// chips. The first is a text chip and the second is an icon chip that opens a
	// link: ``` "chipList": { "chips": [ { "text": "Edit", "disabled": true, }, {
	// "icon": { "knownIcon": "INVITE", "altText": "check calendar" }, "onClick": {
	// "openLink": { "url": "https://example.com/calendar" } } } ] } ```
	ChipList *GoogleAppsCardV1ChipList `json:"chipList,omitempty"`
	// Columns: Displays up to 2 columns. To include more than 2 columns, or to use
	// rows, use the `Grid` widget. For example, the following JSON creates 2
	// columns that each contain text paragraphs: ``` "columns": { "columnItems": [
	// { "horizontalSizeStyle": "FILL_AVAILABLE_SPACE", "horizontalAlignment":
	// "CENTER", "verticalAlignment": "CENTER", "widgets": [ { "textParagraph": {
	// "text": "First column text paragraph" } } ] }, { "horizontalSizeStyle":
	// "FILL_AVAILABLE_SPACE", "horizontalAlignment": "CENTER",
	// "verticalAlignment": "CENTER", "widgets": [ { "textParagraph": { "text":
	// "Second column text paragraph" } } ] } ] } ```
	Columns *GoogleAppsCardV1Columns `json:"columns,omitempty"`
	// DateTimePicker: Displays a widget that lets users input a date, time, or
	// date and time. For example, the following JSON creates a date time picker to
	// schedule an appointment: ``` "dateTimePicker": { "name": "appointment_time",
	// "label": "Book your appointment at:", "type": "DATE_AND_TIME",
	// "valueMsEpoch": 796435200000 } ```
	DateTimePicker *GoogleAppsCardV1DateTimePicker `json:"dateTimePicker,omitempty"`
	// DecoratedText: Displays a decorated text item. For example, the following
	// JSON creates a decorated text widget showing email address: ```
	// "decoratedText": { "icon": { "knownIcon": "EMAIL" }, "topLabel": "Email
	// Address", "text": "sasha@example.com", "bottomLabel": "This is a new Email
	// address!", "switchControl": { "name": "has_send_welcome_email_to_sasha",
	// "selected": false, "controlType": "CHECKBOX" } } ```
	DecoratedText *GoogleAppsCardV1DecoratedText `json:"decoratedText,omitempty"`
	// Divider: Displays a horizontal line divider between widgets. For example,
	// the following JSON creates a divider: ``` "divider": { } ```
	Divider *GoogleAppsCardV1Divider `json:"divider,omitempty"`
	// EventActions: Specifies the event actions that can be performed on the
	// widget. Only supported by Google Workspace Workflow, but not Google Chat
	// apps or Google Workspace add-ons.
	EventActions []*GoogleAppsCardV1EventAction `json:"eventActions,omitempty"`
	// Grid: Displays a grid with a collection of items. A grid supports any number
	// of columns and items. The number of rows is determined by the upper bounds
	// of the number items divided by the number of columns. A grid with 10 items
	// and 2 columns has 5 rows. A grid with 11 items and 2 columns has 6 rows.
	// Google Workspace add-ons and Chat apps
	// (https://developers.google.com/workspace/extend): For example, the following
	// JSON creates a 2 column grid with a single item: ``` "grid": { "title": "A
	// fine collection of items", "columnCount": 2, "borderStyle": { "type":
	// "STROKE", "cornerRadius": 4 }, "items": [ { "image": { "imageUri":
	// "https://www.example.com/image.png", "cropStyle": { "type": "SQUARE" },
	// "borderStyle": { "type": "STROKE" } }, "title": "An item", "textAlignment":
	// "CENTER" } ], "onClick": { "openLink": { "url": "https://www.example.com" }
	// } } ```
	Grid *GoogleAppsCardV1Grid `json:"grid,omitempty"`
	// HorizontalAlignment: Specifies whether widgets align to the left, right, or
	// center of a column.
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "START" - Default value. Aligns widgets to the start position of the
	// column. For left-to-right layouts, aligns to the left. For right-to-left
	// layouts, aligns to the right.
	//   "CENTER" - Aligns widgets to the center of the column.
	//   "END" - Aligns widgets to the end position of the column. For
	// left-to-right layouts, aligns widgets to the right. For right-to-left
	// layouts, aligns widgets to the left.
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`
	// Id: A unique ID assigned to the widget that's used to identify the widget to
	// be mutated. The ID has a character limit of 64 characters and should be in
	// the format of `[a-zA-Z0-9-]+` and. Only supported by Google Workspace
	// Workflow, but not Google Chat apps or Google Workspace add-ons.
	Id string `json:"id,omitempty"`
	// Image: Displays an image. For example, the following JSON creates an image
	// with alternative text: ``` "image": { "imageUrl":
	// "https://developers.google.com/workspace/chat/images/quickstart-app-avatar.pn
	// g", "altText": "Chat app avatar" } ```
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`
	// SelectionInput: Displays a selection control that lets users select items.
	// Selection controls can be checkboxes, radio buttons, switches, or dropdown
	// menus. For example, the following JSON creates a dropdown menu that lets
	// users choose a size: ``` "selectionInput": { "name": "size", "label": "Size"
	// "type": "DROPDOWN", "items": [ { "text": "S", "value": "small", "selected":
	// false }, { "text": "M", "value": "medium", "selected": true }, { "text":
	// "L", "value": "large", "selected": false }, { "text": "XL", "value":
	// "extra_large", "selected": false } ] } ```
	SelectionInput *GoogleAppsCardV1SelectionInput `json:"selectionInput,omitempty"`
	// TextInput: Displays a text box that users can type into. For example, the
	// following JSON creates a text input for an email address: ``` "textInput": {
	// "name": "mailing_address", "label": "Mailing Address" } ``` As another
	// example, the following JSON creates a text input for a programming language
	// with static suggestions: ``` "textInput": { "name":
	// "preferred_programing_language", "label": "Preferred Language",
	// "initialSuggestions": { "items": [ { "text": "C++" }, { "text": "Java" }, {
	// "text": "JavaScript" }, { "text": "Python" } ] } } ```
	TextInput *GoogleAppsCardV1TextInput `json:"textInput,omitempty"`
	// TextParagraph: Displays a text paragraph. Supports simple HTML formatted
	// text. For more information about formatting text, see Formatting text in
	// Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	// For example, the following JSON creates a bolded text: ``` "textParagraph":
	// { "text": " *bold text*" } ```
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`
	// Visibility: Specifies whether the widget is visible or hidden. The default
	// value is `VISIBLE`. Only supported by Google Workspace Workflow, but not
	// Google Chat apps or Google Workspace add-ons.
	//
	// Possible values:
	//   "VISIBILITY_UNSPECIFIED" - Unspecified visibility. Do not use.
	//   "VISIBLE" - The UI element is visible.
	//   "HIDDEN" - The UI element is hidden.
	Visibility string `json:"visibility,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ButtonList") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Widget) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Widget
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Widgets: The supported widgets that you can include in a
// column. Google Workspace add-ons and Chat apps
// (https://developers.google.com/workspace/extend)
type GoogleAppsCardV1Widgets struct {
	// ButtonList: ButtonList widget.
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`
	// ChipList: ChipList widget.
	ChipList *GoogleAppsCardV1ChipList `json:"chipList,omitempty"`
	// DateTimePicker: DateTimePicker widget.
	DateTimePicker *GoogleAppsCardV1DateTimePicker `json:"dateTimePicker,omitempty"`
	// DecoratedText: DecoratedText widget.
	DecoratedText *GoogleAppsCardV1DecoratedText `json:"decoratedText,omitempty"`
	// Image: Image widget.
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`
	// SelectionInput: SelectionInput widget.
	SelectionInput *GoogleAppsCardV1SelectionInput `json:"selectionInput,omitempty"`
	// TextInput: TextInput widget.
	TextInput *GoogleAppsCardV1TextInput `json:"textInput,omitempty"`
	// TextParagraph: TextParagraph widget.
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ButtonList") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleAppsCardV1Widgets) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Widgets
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Group: A Google Group in Google Chat.
type Group struct {
	// Name: Resource name for a Google Group. Represents a group
	// (https://cloud.google.com/identity/docs/reference/rest/v1/groups) in Cloud
	// Identity Groups API. Format: groups/{group}
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Group) MarshalJSON() ([]byte, error) {
	type NoMethod Group
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HostAppDataSourceMarkup: A data source from a Google Workspace application.
// The data source populates available items for a widget.
type HostAppDataSourceMarkup struct {
	// ChatDataSource: A data source from Google Chat.
	ChatDataSource *ChatClientDataSourceMarkup `json:"chatDataSource,omitempty"`
	// WorkflowDataSource: A data source from Google Workflow.
	WorkflowDataSource *WorkflowDataSourceMarkup `json:"workflowDataSource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChatDataSource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChatDataSource") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HostAppDataSourceMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod HostAppDataSourceMarkup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Image: An image that's specified by a URL and can have an `onclick` action.
type Image struct {
	// AspectRatio: The aspect ratio of this image (width and height). This field
	// lets you reserve the right height for the image while waiting for it to
	// load. It's not meant to override the built-in aspect ratio of the image. If
	// unset, the server fills it by prefetching the image.
	AspectRatio float64 `json:"aspectRatio,omitempty"`
	// ImageUrl: The URL of the image.
	ImageUrl string `json:"imageUrl,omitempty"`
	// OnClick: The `onclick` action.
	OnClick *OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AspectRatio") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Image) MarshalJSON() ([]byte, error) {
	type NoMethod Image
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Image) UnmarshalJSON(data []byte) error {
	type NoMethod Image
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// ImageButton: An image button with an `onclick` action.
type ImageButton struct {
	// Icon: The icon specified by an `enum` that indices to an icon provided by
	// Chat API.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`
	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`
	// Name: The name of this `image_button` that's used for accessibility. Default
	// value is provided if this name isn't specified.
	Name string `json:"name,omitempty"`
	// OnClick: The `onclick` action.
	OnClick *OnClick `json:"onClick,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Icon") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Icon") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ImageButton) MarshalJSON() ([]byte, error) {
	type NoMethod ImageButton
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Inputs: Types of data that users can input on cards or dialogs
// (https://developers.google.com/chat/ui/read-form-data). The input type
// depends on the type of values that the widget accepts.
type Inputs struct {
	// DateInput: Date input values from a `DateTimePicker`
	// (https://developers.google.com/chat/api/reference/rest/v1/cards#DateTimePicker)
	// widget that only accepts date values.
	DateInput *DateInput `json:"dateInput,omitempty"`
	// DateTimeInput: Date and time input values from a `DateTimePicker`
	// (https://developers.google.com/chat/api/reference/rest/v1/cards#DateTimePicker)
	// widget that accepts both a date and time.
	DateTimeInput *DateTimeInput `json:"dateTimeInput,omitempty"`
	// StringInputs: A list of strings that represent the values that the user
	// inputs in a widget. If the widget only accepts one value, such as a
	// `TextInput`
	// (https://developers.google.com/chat/api/reference/rest/v1/cards#TextInput)
	// widget, the list contains one string object. If the widget accepts multiple
	// values, such as a `SelectionInput`
	// (https://developers.google.com/chat/api/reference/rest/v1/cards#selectioninput)
	// widget of checkboxes, the list contains a string object for each value that
	// the user inputs or selects.
	StringInputs *StringInputs `json:"stringInputs,omitempty"`
	// TimeInput: Time input values from a `DateTimePicker`
	// (https://developers.google.com/chat/api/reference/rest/v1/cards#DateTimePicker)
	// widget that only accepts time values.
	TimeInput *TimeInput `json:"timeInput,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateInput") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateInput") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Inputs) MarshalJSON() ([]byte, error) {
	type NoMethod Inputs
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// KeyValue: A UI element contains a key (label) and a value (content). This
// element can also contain some actions such as `onclick` button.
type KeyValue struct {
	// BottomLabel: The text of the bottom label. Formatted text supported. For
	// more information about formatting text, see Formatting text in Google Chat
	// apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	BottomLabel string `json:"bottomLabel,omitempty"`
	// Button: A button that can be clicked to trigger an action.
	Button *Button `json:"button,omitempty"`
	// Content: The text of the content. Formatted text supported and always
	// required. For more information about formatting text, see Formatting text in
	// Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Content string `json:"content,omitempty"`
	// ContentMultiline: If the content should be multiline.
	ContentMultiline bool `json:"contentMultiline,omitempty"`
	// Icon: An enum value that's replaced by the Chat API with the corresponding
	// icon image.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`
	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`
	// OnClick: The `onclick` action. Only the top label, bottom label, and content
	// region are clickable.
	OnClick *OnClick `json:"onClick,omitempty"`
	// TopLabel: The text of the top label. Formatted text supported. For more
	// information about formatting text, see Formatting text in Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	TopLabel string `json:"topLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BottomLabel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s KeyValue) MarshalJSON() ([]byte, error) {
	type NoMethod KeyValue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListCustomEmojisResponse: A response to list custom emojis.
type ListCustomEmojisResponse struct {
	// CustomEmojis: Unordered list. List of custom emojis.
	CustomEmojis []*CustomEmoji `json:"customEmojis,omitempty"`
	// NextPageToken: A token that you can send as `pageToken` to retrieve the next
	// page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CustomEmojis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomEmojis") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListCustomEmojisResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListCustomEmojisResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListMembershipsResponse: Response to list memberships of the space.
type ListMembershipsResponse struct {
	// Memberships: Unordered list. List of memberships in the requested (or first)
	// page.
	Memberships []*Membership `json:"memberships,omitempty"`
	// NextPageToken: A token that you can send as `pageToken` to retrieve the next
	// page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Memberships") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListMembershipsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMembershipsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListMessagesResponse: Response message for listing messages.
type ListMessagesResponse struct {
	// Messages: List of messages.
	Messages []*Message `json:"messages,omitempty"`
	// NextPageToken: You can send a token as `pageToken` to retrieve the next page
	// of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Messages") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Messages") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListMessagesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMessagesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListReactionsResponse: Response to a list reactions request.
type ListReactionsResponse struct {
	// NextPageToken: Continuation token to retrieve the next page of results. It's
	// empty for the last page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Reactions: List of reactions in the requested (or first) page.
	Reactions []*Reaction `json:"reactions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListReactionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListReactionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSpaceEventsResponse: Response message for listing space events.
type ListSpaceEventsResponse struct {
	// NextPageToken: Continuation token used to fetch more events. If this field
	// is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SpaceEvents: Results are returned in chronological order (oldest event
	// first). Note: The `permissionSettings` field is not returned in the Space
	// object for list requests.
	SpaceEvents []*SpaceEvent `json:"spaceEvents,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSpaceEventsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSpaceEventsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSpacesResponse: The response for a list spaces request.
type ListSpacesResponse struct {
	// NextPageToken: You can send a token as `pageToken` to retrieve the next page
	// of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Spaces: List of spaces in the requested (or first) page. Note: The
	// `permissionSettings` field is not returned in the Space object for list
	// requests.
	Spaces []*Space `json:"spaces,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSpacesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSpacesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MatchedUrl: A matched URL in a Chat message. Chat apps can preview matched
// URLs. For more information, see Preview links
// (https://developers.google.com/chat/how-tos/preview-links).
type MatchedUrl struct {
	// Url: Output only. The URL that was matched.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Url") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Url") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MatchedUrl) MarshalJSON() ([]byte, error) {
	type NoMethod MatchedUrl
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Media: Media resource.
type Media struct {
	// ResourceName: Name of the media resource.
	ResourceName string `json:"resourceName,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ResourceName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ResourceName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Media) MarshalJSON() ([]byte, error) {
	type NoMethod Media
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MeetSpaceLinkData: Data for Meet space links.
type MeetSpaceLinkData struct {
	// HuddleStatus: Optional. Output only. If the Meet is a Huddle, indicates the
	// status of the huddle. Otherwise, this is unset.
	//
	// Possible values:
	//   "HUDDLE_STATUS_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "STARTED" - The huddle has started.
	//   "ENDED" - The huddle has ended. In this case the Meet space URI and
	// identifiers will no longer be valid.
	//   "MISSED" - The huddle has been missed. In this case the Meet space URI and
	// identifiers will no longer be valid.
	HuddleStatus string `json:"huddleStatus,omitempty"`
	// MeetingCode: Meeting code of the linked Meet space.
	MeetingCode string `json:"meetingCode,omitempty"`
	// Type: Indicates the type of the Meet space.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "MEETING" - The Meet space is a meeting.
	//   "HUDDLE" - The Meet space is a huddle.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "HuddleStatus") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HuddleStatus") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MeetSpaceLinkData) MarshalJSON() ([]byte, error) {
	type NoMethod MeetSpaceLinkData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Membership: Represents a membership relation in Google Chat, such as whether
// a user or Chat app is invited to, part of, or absent from a space.
type Membership struct {
	// CreateTime: Optional. Immutable. The creation time of the membership, such
	// as when a member joined or was invited to join a space. This field is output
	// only, except when used to import historical memberships in import mode
	// spaces.
	CreateTime string `json:"createTime,omitempty"`
	// DeleteTime: Optional. Immutable. The deletion time of the membership, such
	// as when a member left or was removed from a space. This field is output
	// only, except when used to import historical memberships in import mode
	// spaces.
	DeleteTime string `json:"deleteTime,omitempty"`
	// GroupMember: Optional. The Google Group the membership corresponds to.
	// Reading or mutating memberships for Google Groups requires user
	// authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
	GroupMember *Group `json:"groupMember,omitempty"`
	// Member: Optional. The Google Chat user or app the membership corresponds to.
	// If your Chat app authenticates as a user
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
	// the output populates the user
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/User)
	// `name` and `type`.
	Member *User `json:"member,omitempty"`
	// Name: Identifier. Resource name of the membership, assigned by the server.
	// Format: `spaces/{space}/members/{member}`
	Name string `json:"name,omitempty"`
	// Role: Optional. User's role within a Chat space, which determines their
	// permitted actions in the space. This field can only be used as input in
	// `UpdateMembership`.
	//
	// Possible values:
	//   "MEMBERSHIP_ROLE_UNSPECIFIED" - Default value. For users: they aren't a
	// member of the space, but can be invited. For Google Groups: they're always
	// assigned this role (other enum values might be used in the future).
	//   "ROLE_MEMBER" - A member of the space. In the Chat UI, this role is called
	// Member. The user has basic permissions, like sending messages to the space.
	// Managers and owners can grant members additional permissions in a space,
	// including: - Add or remove members. - Modify space details. - Turn history
	// on or off. - Mention everyone in the space with `@all`. - Manage Chat apps
	// and webhooks installed in the space. In direct messages and unnamed group
	// conversations, everyone has this role.
	//   "ROLE_MANAGER" - A space owner. In the Chat UI, this role is called Owner.
	// The user has the complete set of space permissions to manage the space,
	// including: - Change the role of other members in the space to member,
	// manager, or owner. - Delete the space. Only supported in SpaceType.SPACE
	// (named spaces). To learn more, see [Learn more about your role as a space
	// owner or manager](https://support.google.com/chat/answer/11833441).
	//   "ROLE_ASSISTANT_MANAGER" - A space manager. In the Chat UI, this role is
	// called Manager. The user has all basic permissions of `ROLE_MEMBER`, and can
	// be granted a subset of administrative permissions by an owner. By default,
	// managers have all the permissions of an owner except for the ability to: -
	// Delete the space. - Make another space member an owner. - Change an owner's
	// role. By default, managers permissions include but aren't limited to: - Make
	// another member a manager. - Delete messages in the space. - Manage space
	// permissions. - Receive notifications for requests to join the space if the
	// manager has the "manage members" permission in the space settings. - Make a
	// space discoverable. Only supported in SpaceType.SPACE (named spaces). To
	// learn more, see [Manage space
	// settings](https://support.google.com/chat/answer/13340792).
	Role string `json:"role,omitempty"`
	// State: Output only. State of the membership.
	//
	// Possible values:
	//   "MEMBERSHIP_STATE_UNSPECIFIED" - Default value. Don't use.
	//   "JOINED" - The user is added to the space, and can participate in the
	// space.
	//   "INVITED" - The user is invited to join the space, but hasn't joined it.
	//   "NOT_A_MEMBER" - The user doesn't belong to the space and doesn't have a
	// pending invitation to join the space.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Membership) MarshalJSON() ([]byte, error) {
	type NoMethod Membership
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipBatchCreatedEventData: Event payload for multiple new memberships.
// Event type: `google.workspace.chat.membership.v1.batchCreated`
type MembershipBatchCreatedEventData struct {
	// Memberships: A list of new memberships.
	Memberships []*MembershipCreatedEventData `json:"memberships,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Memberships") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipBatchCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipBatchCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipBatchDeletedEventData: Event payload for multiple deleted
// memberships. Event type: `google.workspace.chat.membership.v1.batchDeleted`
type MembershipBatchDeletedEventData struct {
	// Memberships: A list of deleted memberships.
	Memberships []*MembershipDeletedEventData `json:"memberships,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Memberships") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipBatchDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipBatchDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipBatchUpdatedEventData: Event payload for multiple updated
// memberships. Event type: `google.workspace.chat.membership.v1.batchUpdated`
type MembershipBatchUpdatedEventData struct {
	// Memberships: A list of updated memberships.
	Memberships []*MembershipUpdatedEventData `json:"memberships,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Memberships") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipBatchUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipBatchUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipCount: Represents the count of memberships of a space, grouped
// into categories.
type MembershipCount struct {
	// JoinedDirectHumanUserCount: Output only. Count of human users that have
	// directly joined the space, not counting users joined by having membership in
	// a joined group.
	JoinedDirectHumanUserCount int64 `json:"joinedDirectHumanUserCount,omitempty"`
	// JoinedGroupCount: Output only. Count of all groups that have directly joined
	// the space.
	JoinedGroupCount int64 `json:"joinedGroupCount,omitempty"`
	// ForceSendFields is a list of field names (e.g. "JoinedDirectHumanUserCount")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "JoinedDirectHumanUserCount") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipCount) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipCount
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipCreatedEventData: Event payload for a new membership. Event type:
// `google.workspace.chat.membership.v1.created`.
type MembershipCreatedEventData struct {
	// Membership: The new membership.
	Membership *Membership `json:"membership,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Membership") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Membership") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipDeletedEventData: Event payload for a deleted membership. Event
// type: `google.workspace.chat.membership.v1.deleted`
type MembershipDeletedEventData struct {
	// Membership: The deleted membership. Only the `name` and `state` fields are
	// populated.
	Membership *Membership `json:"membership,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Membership") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Membership") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MembershipUpdatedEventData: Event payload for an updated membership. Event
// type: `google.workspace.chat.membership.v1.updated`
type MembershipUpdatedEventData struct {
	// Membership: The updated membership.
	Membership *Membership `json:"membership,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Membership") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Membership") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MembershipUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MembershipUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Message: A message in a Google Chat space.
type Message struct {
	// AccessoryWidgets: Optional. One or more interactive widgets that appear at
	// the bottom of a message. You can add accessory widgets to messages that
	// contain text, cards, or both text and cards. Not supported for messages that
	// contain dialogs. For details, see Add interactive widgets at the bottom of a
	// message
	// (https://developers.google.com/workspace/chat/create-messages#add-accessory-widgets).
	// Creating a message with accessory widgets requires [app authentication]
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
	AccessoryWidgets []*AccessoryWidget `json:"accessoryWidgets,omitempty"`
	// ActionResponse: Input only. Parameters that a Chat app can use to configure
	// how its response is posted.
	ActionResponse *ActionResponse `json:"actionResponse,omitempty"`
	// Annotations: Output only. Annotations can be associated with the plain-text
	// body of the message or with chips that link to Google Workspace resources
	// like Google Docs or Sheets with `start_index` and `length` of 0.
	Annotations []*Annotation `json:"annotations,omitempty"`
	// ArgumentText: Output only. Plain-text body of the message with all Chat app
	// mentions stripped out.
	ArgumentText string `json:"argumentText,omitempty"`
	// AttachedGifs: Output only. GIF images that are attached to the message.
	AttachedGifs []*AttachedGif `json:"attachedGifs,omitempty"`
	// Attachment: Optional. User-uploaded attachment.
	Attachment []*Attachment `json:"attachment,omitempty"`
	// Cards: Deprecated: Use `cards_v2` instead. Rich, formatted, and interactive
	// cards that you can use to display UI elements such as: formatted texts,
	// buttons, and clickable images. Cards are normally displayed below the
	// plain-text body of the message. `cards` and `cards_v2` can have a maximum
	// size of 32 KB.
	Cards []*Card `json:"cards,omitempty"`
	// CardsV2: Optional. An array of cards
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/cards).
	// Only Chat apps can create cards. If your Chat app authenticates as a user
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
	// the messages can't contain cards. To learn how to create a message that
	// contains cards, see Send a message
	// (https://developers.google.com/workspace/chat/create-messages). Card builder
	// (https://addons.gsuite.google.com/uikit/builder)
	CardsV2 []*CardWithId `json:"cardsV2,omitempty"`
	// ClientAssignedMessageId: Optional. A custom ID for the message. You can use
	// field to identify a message, or to get, delete, or update a message. To set
	// a custom ID, specify the `messageId`
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages/create#body.QUERY_PARAMETERS.message_id)
	// field when you create the message. For details, see Name a message
	// (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
	ClientAssignedMessageId string `json:"clientAssignedMessageId,omitempty"`
	// CreateTime: Optional. Immutable. For spaces created in Chat, the time at
	// which the message was created. This field is output only, except when used
	// in import mode spaces. For import mode spaces, set this field to the
	// historical timestamp at which the message was created in the source in order
	// to preserve the original creation time.
	CreateTime string `json:"createTime,omitempty"`
	// DeleteTime: Output only. The time at which the message was deleted in Google
	// Chat. If the message is never deleted, this field is empty.
	DeleteTime string `json:"deleteTime,omitempty"`
	// DeletionMetadata: Output only. Information about a deleted message. A
	// message is deleted when `delete_time` is set.
	DeletionMetadata *DeletionMetadata `json:"deletionMetadata,omitempty"`
	// EmojiReactionSummaries: Output only. The list of emoji reaction summaries on
	// the message.
	EmojiReactionSummaries []*EmojiReactionSummary `json:"emojiReactionSummaries,omitempty"`
	// FallbackText: Optional. A plain-text description of the message's cards,
	// used when the actual cards can't be displayed—for example, mobile
	// notifications.
	FallbackText string `json:"fallbackText,omitempty"`
	// FormattedText: Output only. Contains the message `text` with markups added
	// to communicate formatting. This field might not capture all formatting
	// visible in the UI, but includes the following: * Markup syntax
	// (https://developers.google.com/workspace/chat/format-messages) for bold,
	// italic, strikethrough, monospace, monospace block, and bulleted list. * User
	// mentions
	// (https://developers.google.com/workspace/chat/format-messages#messages-@mention)
	// using the format ``. * Custom hyperlinks using the format
	// `<{url}|{rendered_text}>` where the first string is the URL and the second
	// is the rendered text—for example, ``. * Custom emoji using the format
	// `:{emoji_name}:`—for example, `:smile:`. This doesn't apply to Unicode
	// emoji, such as `U+1F600` for a grinning face emoji. * Bullet list items
	// using asterisks (`*`)—for example, `* item`. For more information, see
	// View text formatting sent in a message
	// (https://developers.google.com/workspace/chat/format-messages#view_text_formatting_sent_in_a_message)
	FormattedText string `json:"formattedText,omitempty"`
	// LastUpdateTime: Output only. The time at which the message was last edited
	// by a user. If the message has never been edited, this field is empty.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`
	// MatchedUrl: Output only. A URL in `spaces.messages.text` that matches a link
	// preview pattern. For more information, see Preview links
	// (https://developers.google.com/workspace/chat/preview-links).
	MatchedUrl *MatchedUrl `json:"matchedUrl,omitempty"`
	// Name: Identifier. Resource name of the message. Format:
	// `spaces/{space}/messages/{message}` Where `{space}` is the ID of the space
	// where the message is posted and `{message}` is a system-assigned ID for the
	// message. For example, `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
	// If you set a custom ID when you create a message, you can use this ID to
	// specify the message in a request by replacing `{message}` with the value
	// from the `clientAssignedMessageId` field. For example,
	// `spaces/AAAAAAAAAAA/messages/client-custom-name`. For details, see Name a
	// message
	// (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
	Name string `json:"name,omitempty"`
	// PrivateMessageViewer: Optional. Immutable. Input for creating a message,
	// otherwise output only. The user that can view the message. When set, the
	// message is private and only visible to the specified user and the Chat app.
	// To include this field in your request, you must call the Chat API using app
	// authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
	// and omit the following: * Attachments
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages.attachments)
	// * Accessory widgets
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages#Message.AccessoryWidget)
	// For details, see Send a message privately
	// (https://developers.google.com/workspace/chat/create-messages#private).
	PrivateMessageViewer *User `json:"privateMessageViewer,omitempty"`
	// QuotedMessageMetadata: Optional. Information about a message that another
	// message quotes. When you create a message, you can quote messages within the
	// same thread, or quote a root message to create a new root message. However,
	// you can't quote a message reply from a different thread. When you update a
	// message, you can't add or replace the `quotedMessageMetadata` field, but you
	// can remove it. For example usage, see Quote another message
	// (https://developers.google.com/workspace/chat/create-messages#quote-a-message).
	QuotedMessageMetadata *QuotedMessageMetadata `json:"quotedMessageMetadata,omitempty"`
	// Sender: Output only. The user who created the message. If your Chat app
	// authenticates as a user
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
	// the output populates the user
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/User)
	// `name` and `type`.
	Sender *User `json:"sender,omitempty"`
	// SlashCommand: Output only. Slash command information, if applicable.
	SlashCommand *SlashCommand `json:"slashCommand,omitempty"`
	// Space: Output only. If your Chat app authenticates as a user
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
	// the output only populates the space
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces)
	// `name`.
	Space *Space `json:"space,omitempty"`
	// Text: Optional. Plain-text body of the message. The first link to an image,
	// video, or web page generates a preview chip
	// (https://developers.google.com/workspace/chat/preview-links). You can also
	// @mention a Google Chat user
	// (https://developers.google.com/workspace/chat/format-messages#messages-@mention),
	// or everyone in the space. To learn about creating text messages, see Send a
	// message (https://developers.google.com/workspace/chat/create-messages).
	Text string `json:"text,omitempty"`
	// Thread: The thread the message belongs to. For example usage, see Start or
	// reply to a message thread
	// (https://developers.google.com/workspace/chat/create-messages#create-message-thread).
	Thread *Thread `json:"thread,omitempty"`
	// ThreadReply: Output only. When `true`, the message is a response in a reply
	// thread. When `false`, the message is visible in the space's top-level
	// conversation as either the first message of a thread or a message with no
	// threaded replies. If the space doesn't support reply in threads, this field
	// is always `false`.
	ThreadReply bool `json:"threadReply,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessoryWidgets") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessoryWidgets") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Message) MarshalJSON() ([]byte, error) {
	type NoMethod Message
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageBatchCreatedEventData: Event payload for multiple new messages. Event
// type: `google.workspace.chat.message.v1.batchCreated`
type MessageBatchCreatedEventData struct {
	// Messages: A list of new messages.
	Messages []*MessageCreatedEventData `json:"messages,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Messages") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Messages") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageBatchCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageBatchCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageBatchDeletedEventData: Event payload for multiple deleted messages.
// Event type: `google.workspace.chat.message.v1.batchDeleted`
type MessageBatchDeletedEventData struct {
	// Messages: A list of deleted messages.
	Messages []*MessageDeletedEventData `json:"messages,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Messages") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Messages") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageBatchDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageBatchDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageBatchUpdatedEventData: Event payload for multiple updated messages.
// Event type: `google.workspace.chat.message.v1.batchUpdated`
type MessageBatchUpdatedEventData struct {
	// Messages: A list of updated messages.
	Messages []*MessageUpdatedEventData `json:"messages,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Messages") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Messages") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageBatchUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageBatchUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageCreatedEventData: Event payload for a new message. Event type:
// `google.workspace.chat.message.v1.created`
type MessageCreatedEventData struct {
	// Message: The new message.
	Message *Message `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageDeletedEventData: Event payload for a deleted message. Event type:
// `google.workspace.chat.message.v1.deleted`
type MessageDeletedEventData struct {
	// Message: The deleted message. Only the `name`, `createTime`, and
	// `deletionMetadata` fields are populated.
	Message *Message `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MessageUpdatedEventData: Event payload for an updated message. Event type:
// `google.workspace.chat.message.v1.updated`
type MessageUpdatedEventData struct {
	// Message: The updated message.
	Message *Message `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MessageUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod MessageUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OnClick: An `onclick` action (for example, open a link).
type OnClick struct {
	// Action: A form action is triggered by this `onclick` action if specified.
	Action *FormAction `json:"action,omitempty"`
	// OpenLink: This `onclick` action triggers an open link action if specified.
	OpenLink *OpenLink `json:"openLink,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod OnClick
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OpenLink: A link that opens a new window.
type OpenLink struct {
	// Url: The URL to open.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Url") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Url") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod OpenLink
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PermissionSetting: Represents a space permission setting.
type PermissionSetting struct {
	// AssistantManagersAllowed: Optional. Whether space managers
	// `ROLE_ASSISTANT_MANAGER`) have this permission.
	AssistantManagersAllowed bool `json:"assistantManagersAllowed,omitempty"`
	// ManagersAllowed: Optional. Whether space owners (`ROLE_MANAGER`) have this
	// permission.
	ManagersAllowed bool `json:"managersAllowed,omitempty"`
	// MembersAllowed: Optional. Whether basic space members (`ROLE_MEMBER`) have
	// this permission.
	MembersAllowed bool `json:"membersAllowed,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AssistantManagersAllowed")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AssistantManagersAllowed") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionSetting) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionSetting
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PermissionSettings: Permission settings
// (https://support.google.com/chat/answer/13340792) that you can specify when
// updating an existing named space. To set permission settings when creating a
// space, specify the `PredefinedPermissionSettings` field in your request.
type PermissionSettings struct {
	// ManageApps: Optional. Setting for managing apps in a space.
	ManageApps *PermissionSetting `json:"manageApps,omitempty"`
	// ManageMembersAndGroups: Optional. Setting for managing members and groups in
	// a space.
	ManageMembersAndGroups *PermissionSetting `json:"manageMembersAndGroups,omitempty"`
	// ManageWebhooks: Optional. Setting for managing webhooks in a space.
	ManageWebhooks *PermissionSetting `json:"manageWebhooks,omitempty"`
	// ModifySpaceDetails: Optional. Setting for updating space name, avatar,
	// description and guidelines.
	ModifySpaceDetails *PermissionSetting `json:"modifySpaceDetails,omitempty"`
	// PostMessages: Output only. Setting for posting messages in a space.
	PostMessages *PermissionSetting `json:"postMessages,omitempty"`
	// ReplyMessages: Optional. Setting for replying to messages in a space.
	ReplyMessages *PermissionSetting `json:"replyMessages,omitempty"`
	// ToggleHistory: Optional. Setting for toggling space history on and off.
	ToggleHistory *PermissionSetting `json:"toggleHistory,omitempty"`
	// UseAtMentionAll: Optional. Setting for using @all in a space.
	UseAtMentionAll *PermissionSetting `json:"useAtMentionAll,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ManageApps") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ManageApps") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionSettings) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QuotedMessageMetadata: Information about a message that another message
// quotes. When you create a message, you can quote messages within the same
// thread, or quote a root message to create a new root message. However, you
// can't quote a message reply from a different thread. When you update a
// message, you can't add or replace the `quotedMessageMetadata` field, but you
// can remove it. For example usage, see Quote another message
// (https://developers.google.com/workspace/chat/create-messages#quote-a-message).
type QuotedMessageMetadata struct {
	// LastUpdateTime: Required. The timestamp when the quoted message was created
	// or when the quoted message was last updated. If the message was edited, use
	// this field, `last_update_time`. If the message was never edited, use
	// `create_time`. If `last_update_time` doesn't match the latest version of the
	// quoted message, the request fails.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`
	// Name: Required. Resource name of the message that is quoted. Format:
	// `spaces/{space}/messages/{message}`
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LastUpdateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LastUpdateTime") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QuotedMessageMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod QuotedMessageMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Reaction: A reaction to a message.
type Reaction struct {
	// Emoji: Required. The emoji used in the reaction.
	Emoji *Emoji `json:"emoji,omitempty"`
	// Name: Identifier. The resource name of the reaction. Format:
	// `spaces/{space}/messages/{message}/reactions/{reaction}`
	Name string `json:"name,omitempty"`
	// User: Output only. The user who created the reaction.
	User *User `json:"user,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Emoji") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Emoji") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Reaction) MarshalJSON() ([]byte, error) {
	type NoMethod Reaction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReactionBatchCreatedEventData: Event payload for multiple new reactions.
// Event type: `google.workspace.chat.reaction.v1.batchCreated`
type ReactionBatchCreatedEventData struct {
	// Reactions: A list of new reactions.
	Reactions []*ReactionCreatedEventData `json:"reactions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reactions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reactions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReactionBatchCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod ReactionBatchCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReactionBatchDeletedEventData: Event payload for multiple deleted reactions.
// Event type: `google.workspace.chat.reaction.v1.batchDeleted`
type ReactionBatchDeletedEventData struct {
	// Reactions: A list of deleted reactions.
	Reactions []*ReactionDeletedEventData `json:"reactions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reactions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reactions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReactionBatchDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod ReactionBatchDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReactionCreatedEventData: Event payload for a new reaction. Event type:
// `google.workspace.chat.reaction.v1.created`
type ReactionCreatedEventData struct {
	// Reaction: The new reaction.
	Reaction *Reaction `json:"reaction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reaction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reaction") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReactionCreatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod ReactionCreatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReactionDeletedEventData: Event payload for a deleted reaction. Type:
// `google.workspace.chat.reaction.v1.deleted`
type ReactionDeletedEventData struct {
	// Reaction: The deleted reaction.
	Reaction *Reaction `json:"reaction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reaction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reaction") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReactionDeletedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod ReactionDeletedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RichLinkMetadata: A rich link to a resource. Rich links can be associated
// with the plain-text body of the message or represent chips that link to
// Google Workspace resources like Google Docs or Sheets with `start_index` and
// `length` of 0.
type RichLinkMetadata struct {
	// CalendarEventLinkData: Data for a Calendar event link.
	CalendarEventLinkData *CalendarEventLinkData `json:"calendarEventLinkData,omitempty"`
	// ChatSpaceLinkData: Data for a chat space link.
	ChatSpaceLinkData *ChatSpaceLinkData `json:"chatSpaceLinkData,omitempty"`
	// DriveLinkData: Data for a drive link.
	DriveLinkData *DriveLinkData `json:"driveLinkData,omitempty"`
	// MeetSpaceLinkData: Data for a Meet space link.
	MeetSpaceLinkData *MeetSpaceLinkData `json:"meetSpaceLinkData,omitempty"`
	// RichLinkType: The rich link type.
	//
	// Possible values:
	//   "RICH_LINK_TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "DRIVE_FILE" - A Google Drive rich link type.
	//   "CHAT_SPACE" - A Chat space rich link type. For example, a space smart
	// chip.
	//   "MEET_SPACE" - A Meet message rich link type. For example, a Meet chip.
	//   "CALENDAR_EVENT" - A Calendar message rich link type. For example, a
	// Calendar chip.
	RichLinkType string `json:"richLinkType,omitempty"`
	// Uri: The URI of this link.
	Uri string `json:"uri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CalendarEventLinkData") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CalendarEventLinkData") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RichLinkMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod RichLinkMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SearchSpacesResponse: Response with a list of spaces corresponding to the
// search spaces request.
type SearchSpacesResponse struct {
	// NextPageToken: A token that can be used to retrieve the next page. If this
	// field is empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Spaces: A page of the requested spaces.
	Spaces []*Space `json:"spaces,omitempty"`
	// TotalSize: The total number of spaces that match the query, across all
	// pages. If the result is over 10,000 spaces, this value is an estimate.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SearchSpacesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod SearchSpacesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Section: A section contains a collection of widgets that are rendered
// (vertically) in the order that they are specified. Across all platforms,
// cards have a narrow fixed width, so there's currently no need for layout
// properties (for example, float).
type Section struct {
	// Header: The header of the section. Formatted text is supported. For more
	// information about formatting text, see Formatting text in Google Chat apps
	// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Header string `json:"header,omitempty"`
	// Widgets: A section must contain at least one widget.
	Widgets []*WidgetMarkup `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Header") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Header") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Section) MarshalJSON() ([]byte, error) {
	type NoMethod Section
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SelectionItems: List of widget autocomplete results.
type SelectionItems struct {
	// Items: An array of the SelectionItem objects.
	Items []*GoogleAppsCardV1SelectionItem `json:"items,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SelectionItems) MarshalJSON() ([]byte, error) {
	type NoMethod SelectionItems
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetUpSpaceRequest: Request to create a space and add specified users to it.
type SetUpSpaceRequest struct {
	// Memberships: Optional. The Google Chat users or groups to invite to join the
	// space. Omit the calling user, as they are added automatically. The set
	// currently allows up to 49 memberships (in addition to the caller). For human
	// membership, the `Membership.member` field must contain a `user` with `name`
	// populated (format: `users/{user}`) and `type` set to `User.Type.HUMAN`. You
	// can only add human users when setting up a space (adding Chat apps is only
	// supported for direct message setup with the calling app). You can also add
	// members using the user's email as an alias for {user}. For example, the
	// `user.name` can be `users/example@gmail.com`. To invite Gmail users or users
	// from external Google Workspace domains, user's email must be used for
	// `{user}`. For Google group membership, the `Membership.group_member` field
	// must contain a `group` with `name` populated (format `groups/{group}`). You
	// can only add Google groups when setting `Space.spaceType` to `SPACE`.
	// Optional when setting `Space.spaceType` to `SPACE`. Required when setting
	// `Space.spaceType` to `GROUP_CHAT`, along with at least two memberships.
	// Required when setting `Space.spaceType` to `DIRECT_MESSAGE` with a human
	// user, along with exactly one membership. Must be empty when creating a 1:1
	// conversation between a human and the calling Chat app (when setting
	// `Space.spaceType` to `DIRECT_MESSAGE` and `Space.singleUserBotDm` to
	// `true`).
	Memberships []*Membership `json:"memberships,omitempty"`
	// RequestId: Optional. A unique identifier for this request. A random UUID is
	// recommended. Specifying an existing request ID returns the space created
	// with that ID instead of creating a new space. Specifying an existing request
	// ID from the same Chat app with a different authenticated user returns an
	// error.
	RequestId string `json:"requestId,omitempty"`
	// Space: Required. The `Space.spaceType` field is required. To create a space,
	// set `Space.spaceType` to `SPACE` and set `Space.displayName`. If you receive
	// the error message `ALREADY_EXISTS` when setting up a space, try a different
	// `displayName`. An existing space within the Google Workspace organization
	// might already use this display name. To create a group chat, set
	// `Space.spaceType` to `GROUP_CHAT`. Don't set `Space.displayName`. To create
	// a 1:1 conversation between humans, set `Space.spaceType` to `DIRECT_MESSAGE`
	// and set `Space.singleUserBotDm` to `false`. Don't set `Space.displayName` or
	// `Space.spaceDetails`. To create an 1:1 conversation between a human and the
	// calling Chat app, set `Space.spaceType` to `DIRECT_MESSAGE` and
	// `Space.singleUserBotDm` to `true`. Don't set `Space.displayName` or
	// `Space.spaceDetails`. If a `DIRECT_MESSAGE` space already exists, that space
	// is returned instead of creating a new space.
	Space *Space `json:"space,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Memberships") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SetUpSpaceRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetUpSpaceRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SlashCommand: Metadata about a slash command
// (https://developers.google.com/workspace/chat/commands) in Google Chat.
type SlashCommand struct {
	// CommandId: The ID of the slash command.
	CommandId int64 `json:"commandId,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "CommandId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CommandId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SlashCommand) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommand
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SlashCommandMetadata: Annotation metadata for slash commands (/).
type SlashCommandMetadata struct {
	// Bot: The Chat app whose command was invoked.
	Bot *User `json:"bot,omitempty"`
	// CommandId: The command ID of the invoked slash command.
	CommandId int64 `json:"commandId,omitempty,string"`
	// CommandName: The name of the invoked slash command.
	CommandName string `json:"commandName,omitempty"`
	// TriggersDialog: Indicates whether the slash command is for a dialog.
	TriggersDialog bool `json:"triggersDialog,omitempty"`
	// Type: The type of slash command.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "ADD" - Add Chat app to space.
	//   "INVOKE" - Invoke slash command in space.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Bot") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Bot") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SlashCommandMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommandMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Space: A space in Google Chat. Spaces are conversations between two or more
// users or 1:1 messages between a user and a Chat app.
type Space struct {
	// AccessSettings: Optional. Specifies the access setting
	// (https://support.google.com/chat/answer/11971020) of the space. Only
	// populated when the `space_type` is `SPACE`.
	AccessSettings *AccessSettings `json:"accessSettings,omitempty"`
	// AdminInstalled: Output only. For direct message (DM) spaces with a Chat app,
	// whether the space was created by a Google Workspace administrator.
	// Administrators can install and set up a direct message with a Chat app on
	// behalf of users in their organization. To support admin install, your Chat
	// app must feature direct messaging.
	AdminInstalled bool `json:"adminInstalled,omitempty"`
	// CreateTime: Optional. Immutable. For spaces created in Chat, the time the
	// space was created. This field is output only, except when used in import
	// mode spaces. For import mode spaces, set this field to the historical
	// timestamp at which the space was created in the source in order to preserve
	// the original creation time. Only populated in the output when `spaceType` is
	// `GROUP_CHAT` or `SPACE`.
	CreateTime string `json:"createTime,omitempty"`
	// Customer: Optional. Immutable. The customer id of the domain of the space.
	// Required only when creating a space with app authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
	// and `SpaceType` is `SPACE`, otherwise should not be set. In the format
	// `customers/{customer}`, where `customer` is the `id` from the Admin SDK
	// customer resource
	// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/customers).
	// Private apps can also use the `customers/my_customer` alias to create the
	// space in the same Google Workspace organization as the app. This field isn't
	// populated for direct messages (DMs) or when the space is created by
	// non-Google Workspace users.
	Customer string `json:"customer,omitempty"`
	// DisplayName: Optional. The space's display name. Required when creating a
	// space
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces/create)
	// with a `spaceType` of `SPACE`. If you receive the error message
	// `ALREADY_EXISTS` when creating a space or updating the `displayName`, try a
	// different `displayName`. An existing space within the Google Workspace
	// organization might already use this display name. For direct messages, this
	// field might be empty. Supports up to 128 characters.
	DisplayName string `json:"displayName,omitempty"`
	// ExternalUserAllowed: Optional. Immutable. Whether this space permits any
	// Google Chat user as a member. Input when creating a space in a Google
	// Workspace organization. Omit this field when creating spaces in the
	// following conditions: * The authenticated user uses a consumer account
	// (unmanaged user account). By default, a space created by a consumer account
	// permits any Google Chat user. For existing spaces, this field is output
	// only.
	ExternalUserAllowed bool `json:"externalUserAllowed,omitempty"`
	// ImportMode: Optional. Whether this space is created in `Import Mode` as part
	// of a data migration into Google Workspace. While spaces are being imported,
	// they aren't visible to users until the import is complete. Creating a space
	// in `Import Mode`requires user authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
	ImportMode bool `json:"importMode,omitempty"`
	// ImportModeExpireTime: Output only. The time when the space will be
	// automatically deleted by the system if it remains in import mode. Each space
	// created in import mode must exit this mode before this expire time using
	// `spaces.completeImport`. This field is only populated for spaces that were
	// created with import mode.
	ImportModeExpireTime string `json:"importModeExpireTime,omitempty"`
	// LastActiveTime: Output only. Timestamp of the last message in the space.
	LastActiveTime string `json:"lastActiveTime,omitempty"`
	// MembershipCount: Output only. The count of joined memberships grouped by
	// member type. Populated when the `space_type` is `SPACE`, `DIRECT_MESSAGE` or
	// `GROUP_CHAT`.
	MembershipCount *MembershipCount `json:"membershipCount,omitempty"`
	// Name: Identifier. Resource name of the space. Format: `spaces/{space}` Where
	// `{space}` represents the system-assigned ID for the space. You can obtain
	// the space ID by calling the `spaces.list()`
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces/list)
	// method or from the space URL. For example, if the space URL is
	// `https://mail.google.com/mail/u/0/#chat/space/AAAAAAAAA`, the space ID is
	// `AAAAAAAAA`.
	Name string `json:"name,omitempty"`
	// PermissionSettings: Optional. Space permission settings for existing spaces.
	// Input for updating exact space permission settings, where existing
	// permission settings are replaced. Output lists current permission settings.
	// Reading and updating permission settings supports: - App authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
	// with administrator approval (https://support.google.com/a?p=chat-app-auth)
	// with the `chat.app.spaces` scope. Only populated and settable when the Chat
	// app created the space. - User authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
	PermissionSettings *PermissionSettings `json:"permissionSettings,omitempty"`
	// PredefinedPermissionSettings: Optional. Input only. Predefined space
	// permission settings, input only when creating a space. If the field is not
	// set, a collaboration space is created. After you create the space, settings
	// are populated in the `PermissionSettings` field. Setting predefined
	// permission settings supports: - App authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
	// with administrator approval (https://support.google.com/a?p=chat-app-auth)
	// with the `chat.app.spaces` or `chat.app.spaces.create` scopes. - User
	// authentication
	// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
	//
	// Possible values:
	//   "PREDEFINED_PERMISSION_SETTINGS_UNSPECIFIED" - Unspecified. Don't use.
	//   "COLLABORATION_SPACE" - Setting to make the space a collaboration space
	// where all members can post messages.
	//   "ANNOUNCEMENT_SPACE" - Setting to make the space an announcement space
	// where only space managers can post messages.
	PredefinedPermissionSettings string `json:"predefinedPermissionSettings,omitempty"`
	// SingleUserBotDm: Optional. Whether the space is a DM between a Chat app and
	// a single human.
	SingleUserBotDm bool `json:"singleUserBotDm,omitempty"`
	// SpaceDetails: Optional. Details about the space including description and
	// rules.
	SpaceDetails *SpaceDetails `json:"spaceDetails,omitempty"`
	// SpaceHistoryState: Optional. The message history state for messages and
	// threads in this space.
	//
	// Possible values:
	//   "HISTORY_STATE_UNSPECIFIED" - Default value. Do not use.
	//   "HISTORY_OFF" - History off. [Messages and threads are kept for 24
	// hours](https://support.google.com/chat/answer/7664687).
	//   "HISTORY_ON" - History on. The organization's [Vault retention
	// rules](https://support.google.com/vault/answer/7657597) specify for how long
	// messages and threads are kept.
	SpaceHistoryState string `json:"spaceHistoryState,omitempty"`
	// SpaceThreadingState: Output only. The threading state in the Chat space.
	//
	// Possible values:
	//   "SPACE_THREADING_STATE_UNSPECIFIED" - Reserved.
	//   "THREADED_MESSAGES" - Named spaces that support message threads. When
	// users respond to a message, they can reply in-thread, which keeps their
	// response in the context of the original message.
	//   "GROUPED_MESSAGES" - Named spaces where the conversation is organized by
	// topic. Topics and their replies are grouped together.
	//   "UNTHREADED_MESSAGES" - Direct messages (DMs) between two people and group
	// conversations between 3 or more people.
	SpaceThreadingState string `json:"spaceThreadingState,omitempty"`
	// SpaceType: Optional. The type of space. Required when creating a space or
	// updating the space type of a space. Output only for other usage.
	//
	// Possible values:
	//   "SPACE_TYPE_UNSPECIFIED" - Reserved.
	//   "SPACE" - A place where people send messages, share files, and
	// collaborate. A `SPACE` can include Chat apps.
	//   "GROUP_CHAT" - Group conversations between 3 or more people. A
	// `GROUP_CHAT` can include Chat apps.
	//   "DIRECT_MESSAGE" - 1:1 messages between two humans or a human and a Chat
	// app.
	SpaceType string `json:"spaceType,omitempty"`
	// SpaceUri: Output only. The URI for a user to access the space.
	SpaceUri string `json:"spaceUri,omitempty"`
	// Threaded: Output only. Deprecated: Use `spaceThreadingState` instead.
	// Whether messages are threaded in this space.
	Threaded bool `json:"threaded,omitempty"`
	// Type: Output only. Deprecated: Use `space_type` instead. The type of a
	// space.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Reserved.
	//   "ROOM" - Conversations between two or more humans.
	//   "DM" - 1:1 Direct Message between a human and a Chat app, where all
	// messages are flat. Note that this doesn't include direct messages between
	// two humans.
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessSettings") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Space) MarshalJSON() ([]byte, error) {
	type NoMethod Space
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceBatchUpdatedEventData: Event payload for multiple updates to a space.
// Event type: `google.workspace.chat.space.v1.batchUpdated`
type SpaceBatchUpdatedEventData struct {
	// Spaces: A list of updated spaces.
	Spaces []*SpaceUpdatedEventData `json:"spaces,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Spaces") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Spaces") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceBatchUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceBatchUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceDataSource: A data source that populates Google Chat spaces as
// selection items for a multiselect menu. Only populates spaces that the user
// is a member of. Google Chat apps
// (https://developers.google.com/workspace/chat):
type SpaceDataSource struct {
	// DefaultToCurrentSpace: If set to `true`, the multiselect menu selects the
	// current Google Chat space as an item by default.
	DefaultToCurrentSpace bool `json:"defaultToCurrentSpace,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DefaultToCurrentSpace") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DefaultToCurrentSpace") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceDetails: Details about the space including description and rules.
type SpaceDetails struct {
	// Description: Optional. A description of the space. For example, describe the
	// space's discussion topic, functional purpose, or participants. Supports up
	// to 150 characters.
	Description string `json:"description,omitempty"`
	// Guidelines: Optional. The space's rules, expectations, and etiquette.
	// Supports up to 5,000 characters.
	Guidelines string `json:"guidelines,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceDetails) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceEvent: An event that represents a change or activity in a Google Chat
// space. To learn more, see Work with events from Google Chat
// (https://developers.google.com/workspace/chat/events-overview).
type SpaceEvent struct {
	// EventTime: Time when the event occurred.
	EventTime string `json:"eventTime,omitempty"`
	// EventType: Type of space event. Each event type has a batch version, which
	// represents multiple instances of the event type that occur in a short period
	// of time. For `spaceEvents.list()` requests, omit batch event types in your
	// query filter. By default, the server returns both event type and its batch
	// version. Supported event types for messages
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages):
	// * New message: `google.workspace.chat.message.v1.created` * Updated message:
	// `google.workspace.chat.message.v1.updated` * Deleted message:
	// `google.workspace.chat.message.v1.deleted` * Multiple new messages:
	// `google.workspace.chat.message.v1.batchCreated` * Multiple updated messages:
	// `google.workspace.chat.message.v1.batchUpdated` * Multiple deleted messages:
	// `google.workspace.chat.message.v1.batchDeleted` Supported event types for
	// memberships
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.members):
	// * New membership: `google.workspace.chat.membership.v1.created` * Updated
	// membership: `google.workspace.chat.membership.v1.updated` * Deleted
	// membership: `google.workspace.chat.membership.v1.deleted` * Multiple new
	// memberships: `google.workspace.chat.membership.v1.batchCreated` * Multiple
	// updated memberships: `google.workspace.chat.membership.v1.batchUpdated` *
	// Multiple deleted memberships:
	// `google.workspace.chat.membership.v1.batchDeleted` Supported event types for
	// reactions
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages.reactions):
	// * New reaction: `google.workspace.chat.reaction.v1.created` * Deleted
	// reaction: `google.workspace.chat.reaction.v1.deleted` * Multiple new
	// reactions: `google.workspace.chat.reaction.v1.batchCreated` * Multiple
	// deleted reactions: `google.workspace.chat.reaction.v1.batchDeleted`
	// Supported event types about the space
	// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces):
	// * Updated space: `google.workspace.chat.space.v1.updated` * Multiple space
	// updates: `google.workspace.chat.space.v1.batchUpdated`
	EventType string `json:"eventType,omitempty"`
	// MembershipBatchCreatedEventData: Event payload for multiple new memberships.
	// Event type: `google.workspace.chat.membership.v1.batchCreated`
	MembershipBatchCreatedEventData *MembershipBatchCreatedEventData `json:"membershipBatchCreatedEventData,omitempty"`
	// MembershipBatchDeletedEventData: Event payload for multiple deleted
	// memberships. Event type: `google.workspace.chat.membership.v1.batchDeleted`
	MembershipBatchDeletedEventData *MembershipBatchDeletedEventData `json:"membershipBatchDeletedEventData,omitempty"`
	// MembershipBatchUpdatedEventData: Event payload for multiple updated
	// memberships. Event type: `google.workspace.chat.membership.v1.batchUpdated`
	MembershipBatchUpdatedEventData *MembershipBatchUpdatedEventData `json:"membershipBatchUpdatedEventData,omitempty"`
	// MembershipCreatedEventData: Event payload for a new membership. Event type:
	// `google.workspace.chat.membership.v1.created`
	MembershipCreatedEventData *MembershipCreatedEventData `json:"membershipCreatedEventData,omitempty"`
	// MembershipDeletedEventData: Event payload for a deleted membership. Event
	// type: `google.workspace.chat.membership.v1.deleted`
	MembershipDeletedEventData *MembershipDeletedEventData `json:"membershipDeletedEventData,omitempty"`
	// MembershipUpdatedEventData: Event payload for an updated membership. Event
	// type: `google.workspace.chat.membership.v1.updated`
	MembershipUpdatedEventData *MembershipUpdatedEventData `json:"membershipUpdatedEventData,omitempty"`
	// MessageBatchCreatedEventData: Event payload for multiple new messages. Event
	// type: `google.workspace.chat.message.v1.batchCreated`
	MessageBatchCreatedEventData *MessageBatchCreatedEventData `json:"messageBatchCreatedEventData,omitempty"`
	// MessageBatchDeletedEventData: Event payload for multiple deleted messages.
	// Event type: `google.workspace.chat.message.v1.batchDeleted`
	MessageBatchDeletedEventData *MessageBatchDeletedEventData `json:"messageBatchDeletedEventData,omitempty"`
	// MessageBatchUpdatedEventData: Event payload for multiple updated messages.
	// Event type: `google.workspace.chat.message.v1.batchUpdated`
	MessageBatchUpdatedEventData *MessageBatchUpdatedEventData `json:"messageBatchUpdatedEventData,omitempty"`
	// MessageCreatedEventData: Event payload for a new message. Event type:
	// `google.workspace.chat.message.v1.created`
	MessageCreatedEventData *MessageCreatedEventData `json:"messageCreatedEventData,omitempty"`
	// MessageDeletedEventData: Event payload for a deleted message. Event type:
	// `google.workspace.chat.message.v1.deleted`
	MessageDeletedEventData *MessageDeletedEventData `json:"messageDeletedEventData,omitempty"`
	// MessageUpdatedEventData: Event payload for an updated message. Event type:
	// `google.workspace.chat.message.v1.updated`
	MessageUpdatedEventData *MessageUpdatedEventData `json:"messageUpdatedEventData,omitempty"`
	// Name: Resource name of the space event. Format:
	// `spaces/{space}/spaceEvents/{spaceEvent}`
	Name string `json:"name,omitempty"`
	// ReactionBatchCreatedEventData: Event payload for multiple new reactions.
	// Event type: `google.workspace.chat.reaction.v1.batchCreated`
	ReactionBatchCreatedEventData *ReactionBatchCreatedEventData `json:"reactionBatchCreatedEventData,omitempty"`
	// ReactionBatchDeletedEventData: Event payload for multiple deleted reactions.
	// Event type: `google.workspace.chat.reaction.v1.batchDeleted`
	ReactionBatchDeletedEventData *ReactionBatchDeletedEventData `json:"reactionBatchDeletedEventData,omitempty"`
	// ReactionCreatedEventData: Event payload for a new reaction. Event type:
	// `google.workspace.chat.reaction.v1.created`
	ReactionCreatedEventData *ReactionCreatedEventData `json:"reactionCreatedEventData,omitempty"`
	// ReactionDeletedEventData: Event payload for a deleted reaction. Event type:
	// `google.workspace.chat.reaction.v1.deleted`
	ReactionDeletedEventData *ReactionDeletedEventData `json:"reactionDeletedEventData,omitempty"`
	// SpaceBatchUpdatedEventData: Event payload for multiple updates to a space.
	// Event type: `google.workspace.chat.space.v1.batchUpdated`
	SpaceBatchUpdatedEventData *SpaceBatchUpdatedEventData `json:"spaceBatchUpdatedEventData,omitempty"`
	// SpaceUpdatedEventData: Event payload for a space update. Event type:
	// `google.workspace.chat.space.v1.updated`
	SpaceUpdatedEventData *SpaceUpdatedEventData `json:"spaceUpdatedEventData,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "EventTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EventTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceEvent) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceEvent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceNotificationSetting: The notification setting of a user in a space.
type SpaceNotificationSetting struct {
	// MuteSetting: The space notification mute setting.
	//
	// Possible values:
	//   "MUTE_SETTING_UNSPECIFIED" - Reserved.
	//   "UNMUTED" - The user will receive notifications for the space based on the
	// notification setting.
	//   "MUTED" - The user will not receive any notifications for the space,
	// regardless of the notification setting.
	MuteSetting string `json:"muteSetting,omitempty"`
	// Name: Identifier. The resource name of the space notification setting.
	// Format: `users/{user}/spaces/{space}/spaceNotificationSetting`.
	Name string `json:"name,omitempty"`
	// NotificationSetting: The notification setting.
	//
	// Possible values:
	//   "NOTIFICATION_SETTING_UNSPECIFIED" - Reserved.
	//   "ALL" - Notifications are triggered by @mentions, followed threads, first
	// message of new threads. All new threads are automatically followed, unless
	// manually unfollowed by the user.
	//   "MAIN_CONVERSATIONS" - The notification is triggered by @mentions,
	// followed threads, first message of new threads. Not available for 1:1 direct
	// messages.
	//   "FOR_YOU" - The notification is triggered by @mentions, followed threads.
	// Not available for 1:1 direct messages.
	//   "OFF" - Notification is off.
	NotificationSetting string `json:"notificationSetting,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "MuteSetting") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MuteSetting") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceNotificationSetting) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceNotificationSetting
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceReadState: A user's read state within a space, used to identify read
// and unread messages.
type SpaceReadState struct {
	// LastReadTime: Optional. The time when the user's space read state was
	// updated. Usually this corresponds with either the timestamp of the last read
	// message, or a timestamp specified by the user to mark the last read position
	// in a space.
	LastReadTime string `json:"lastReadTime,omitempty"`
	// Name: Resource name of the space read state. Format:
	// `users/{user}/spaces/{space}/spaceReadState`
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "LastReadTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LastReadTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceReadState) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceReadState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceUpdatedEventData: Event payload for an updated space. Event type:
// `google.workspace.chat.space.v1.updated`
type SpaceUpdatedEventData struct {
	// Space: The updated space.
	Space *Space `json:"space,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Space") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Space") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceUpdatedEventData) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceUpdatedEventData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each `Status` message contains three
// pieces of data: error code, error message, and error details. You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StringInputs: Input parameter for regular widgets. For single-valued
// widgets, it is a single value list. For multi-valued widgets, such as
// checkbox, all the values are presented.
type StringInputs struct {
	// Value: An list of strings entered by the user.
	Value []string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Value") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Value") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StringInputs) MarshalJSON() ([]byte, error) {
	type NoMethod StringInputs
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TextButton: A button with text and `onclick` action.
type TextButton struct {
	// OnClick: The `onclick` action of the button.
	OnClick *OnClick `json:"onClick,omitempty"`
	// Text: The text of the button.
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OnClick") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OnClick") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TextButton) MarshalJSON() ([]byte, error) {
	type NoMethod TextButton
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TextParagraph: A paragraph of text. Formatted text supported. For more
// information about formatting text, see Formatting text in Google Chat apps
// (https://developers.google.com/workspace/chat/format-messages#card-formatting)
// and Formatting text in Google Workspace Add-ons
// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
type TextParagraph struct {
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Text") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Text") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod TextParagraph
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Thread: A thread in a Google Chat space. For example usage, see Start or
// reply to a message thread
// (https://developers.google.com/workspace/chat/create-messages#create-message-thread).
// If you specify a thread when creating a message, you can set the
// `messageReplyOption`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.messages/create#messagereplyoption)
// field to determine what happens if no matching thread is found.
type Thread struct {
	// Name: Identifier. Resource name of the thread. Example:
	// `spaces/{space}/threads/{thread}`
	Name string `json:"name,omitempty"`
	// ThreadKey: Optional. Input for creating or updating a thread. Otherwise,
	// output only. ID for the thread. Supports up to 4000 characters. This ID is
	// unique to the Chat app that sets it. For example, if multiple Chat apps
	// create a message using the same thread key, the messages are posted in
	// different threads. To reply in a thread created by a person or another Chat
	// app, specify the thread `name` field instead.
	ThreadKey string `json:"threadKey,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Thread) MarshalJSON() ([]byte, error) {
	type NoMethod Thread
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ThreadReadState: A user's read state within a thread, used to identify read
// and unread messages.
type ThreadReadState struct {
	// LastReadTime: The time when the user's thread read state was updated.
	// Usually this corresponds with the timestamp of the last read message in a
	// thread.
	LastReadTime string `json:"lastReadTime,omitempty"`
	// Name: Resource name of the thread read state. Format:
	// `users/{user}/spaces/{space}/threads/{thread}/threadReadState`
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "LastReadTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LastReadTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ThreadReadState) MarshalJSON() ([]byte, error) {
	type NoMethod ThreadReadState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeInput: Time input values.
type TimeInput struct {
	// Hours: The hour on a 24-hour clock.
	Hours int64 `json:"hours,omitempty"`
	// Minutes: The number of minutes past the hour. Valid values are 0 to 59.
	Minutes int64 `json:"minutes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Hours") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hours") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod TimeInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeZone: The timezone ID and offset from Coordinated Universal Time (UTC).
// Only supported for the event types `CARD_CLICKED`
// (https://developers.google.com/chat/api/reference/rest/v1/EventType#ENUM_VALUES.CARD_CLICKED)
// and `SUBMIT_DIALOG`
// (https://developers.google.com/chat/api/reference/rest/v1/DialogEventType#ENUM_VALUES.SUBMIT_DIALOG).
type TimeZone struct {
	// Id: The IANA TZ (https://www.iana.org/time-zones) time zone database code,
	// such as "America/Toronto".
	Id string `json:"id,omitempty"`
	// Offset: The user timezone offset, in milliseconds, from Coordinated
	// Universal Time (UTC).
	Offset int64 `json:"offset,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeZone) MarshalJSON() ([]byte, error) {
	type NoMethod TimeZone
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UpdatedWidget: For `selectionInput` widgets, returns autocomplete
// suggestions for a multiselect menu.
type UpdatedWidget struct {
	// Suggestions: List of widget autocomplete results
	Suggestions *SelectionItems `json:"suggestions,omitempty"`
	// Widget: The ID of the updated widget. The ID must match the one for the
	// widget that triggered the update request.
	Widget string `json:"widget,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Suggestions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Suggestions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UpdatedWidget) MarshalJSON() ([]byte, error) {
	type NoMethod UpdatedWidget
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UploadAttachmentRequest: Request to upload an attachment.
type UploadAttachmentRequest struct {
	// Filename: Required. The filename of the attachment, including the file
	// extension.
	Filename string `json:"filename,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Filename") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Filename") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UploadAttachmentRequest) MarshalJSON() ([]byte, error) {
	type NoMethod UploadAttachmentRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UploadAttachmentResponse: Response of uploading an attachment.
type UploadAttachmentResponse struct {
	// AttachmentDataRef: Reference to the uploaded attachment.
	AttachmentDataRef *AttachmentDataRef `json:"attachmentDataRef,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AttachmentDataRef") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttachmentDataRef") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UploadAttachmentResponse) MarshalJSON() ([]byte, error) {
	type NoMethod UploadAttachmentResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// User: A user in Google Chat. When returned as an output from a request, if
// your Chat app authenticates as a user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
// the output for a `User` resource only populates the user's `name` and
// `type`.
type User struct {
	// DisplayName: Output only. The user's display name.
	DisplayName string `json:"displayName,omitempty"`
	// DomainId: Unique identifier of the user's Google Workspace domain.
	DomainId string `json:"domainId,omitempty"`
	// IsAnonymous: Output only. When `true`, the user is deleted or their profile
	// is not visible.
	IsAnonymous bool `json:"isAnonymous,omitempty"`
	// Name: Resource name for a Google Chat user. Format: `users/{user}`.
	// `users/app` can be used as an alias for the calling app bot user. For human
	// users, `{user}` is the same user identifier as: - the `id` for the Person
	// (https://developers.google.com/people/api/rest/v1/people) in the People API.
	// For example, `users/123456789` in Chat API represents the same person as the
	// `123456789` Person profile ID in People API. - the `id` for a user
	// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users)
	// in the Admin SDK Directory API. - the user's email address can be used as an
	// alias for `{user}` in API requests. For example, if the People API Person
	// profile ID for `user@example.com` is `123456789`, you can use
	// `users/user@example.com` as an alias to reference `users/123456789`. Only
	// the canonical resource name (for example `users/123456789`) will be returned
	// from the API.
	Name string `json:"name,omitempty"`
	// Type: User type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "HUMAN" - Human user.
	//   "BOT" - Chat app user.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserMentionMetadata: Annotation metadata for user mentions (@).
type UserMentionMetadata struct {
	// Type: The type of user mention.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "ADD" - Add user to space.
	//   "MENTION" - Mention user in space.
	Type string `json:"type,omitempty"`
	// User: The user mentioned.
	User *User `json:"user,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Type") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Type") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserMentionMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod UserMentionMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WidgetMarkup: A widget is a UI element that presents text and images.
type WidgetMarkup struct {
	// Buttons: A list of buttons. Buttons is also `oneof data` and only one of
	// these fields should be set.
	Buttons []*Button `json:"buttons,omitempty"`
	// Image: Display an image in this widget.
	Image *Image `json:"image,omitempty"`
	// KeyValue: Display a key value item in this widget.
	KeyValue *KeyValue `json:"keyValue,omitempty"`
	// TextParagraph: Display a text paragraph in this widget.
	TextParagraph *TextParagraph `json:"textParagraph,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Buttons") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Buttons") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WidgetMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod WidgetMarkup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WorkflowDataSourceMarkup: * Only supported by Google Workspace Workflow, but
// not Google Chat apps or Google Workspace add-ons. In a `TextInput` or
// `SelectionInput` widget with MULTI_SELECT type or a `DateTimePicker`,
// provide data source from Google.
type WorkflowDataSourceMarkup struct {
	// IncludeVariables: Whether to include variables from the previous step in the
	// data source.
	IncludeVariables bool `json:"includeVariables,omitempty"`
	// Type: The type of data source.
	//
	// Possible values:
	//   "UNKNOWN" - Default value. Don't use.
	//   "USER" - Google Workspace users. The user can only view and select users
	// from their Google Workspace organization.
	//   "SPACE" - Google Chat spaces that the user is a member of.
	//   "USER_WITH_FREE_FORM" - Users can choose to view and select existing
	// members from their Google Workspace organization or manually enter an email
	// address or a valid domain.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IncludeVariables") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IncludeVariables") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WorkflowDataSourceMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod WorkflowDataSourceMarkup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type CustomEmojisCreateCall struct {
	s           *Service
	customemoji *CustomEmoji
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Create: Creates a custom emoji. Custom emojis are only available for Google
// Workspace accounts, and the administrator must turn custom emojis on for the
// organization. For more information, see Learn about custom emojis in Google
// Chat (https://support.google.com/chat/answer/12800149) and Manage custom
// emoji permissions (https://support.google.com/a/answer/12850085). Requires
// user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.customemojis`
func (r *CustomEmojisService) Create(customemoji *CustomEmoji) *CustomEmojisCreateCall {
	c := &CustomEmojisCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.customemoji = customemoji
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CustomEmojisCreateCall) Fields(s ...googleapi.Field) *CustomEmojisCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CustomEmojisCreateCall) Context(ctx context.Context) *CustomEmojisCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CustomEmojisCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CustomEmojisCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.customemoji)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/customEmojis")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.customEmojis.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.customEmojis.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *CustomEmoji.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CustomEmojisCreateCall) Do(opts ...googleapi.CallOption) (*CustomEmoji, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CustomEmoji{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.customEmojis.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CustomEmojisDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a custom emoji. By default, users can only delete custom
// emoji they created. Emoji managers
// (https://support.google.com/a/answer/12850085) assigned by the administrator
// can delete any custom emoji in the organization. See Learn about custom
// emojis in Google Chat (https://support.google.com/chat/answer/12800149).
// Custom emojis are only available for Google Workspace accounts, and the
// administrator must turn custom emojis on for the organization. For more
// information, see Learn about custom emojis in Google Chat
// (https://support.google.com/chat/answer/12800149) and Manage custom emoji
// permissions (https://support.google.com/a/answer/12850085). Requires user
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.customemojis`
//
//   - name: Resource name of the custom emoji to delete. Format:
//     `customEmojis/{customEmoji}` You can use the emoji name as an alias for
//     `{customEmoji}`. For example, `customEmojis/:example-emoji:` where
//     `:example-emoji:` is the emoji name for a custom emoji.
func (r *CustomEmojisService) Delete(name string) *CustomEmojisDeleteCall {
	c := &CustomEmojisDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CustomEmojisDeleteCall) Fields(s ...googleapi.Field) *CustomEmojisDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CustomEmojisDeleteCall) Context(ctx context.Context) *CustomEmojisDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CustomEmojisDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CustomEmojisDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.customEmojis.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.customEmojis.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CustomEmojisDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.customEmojis.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CustomEmojisGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a custom emoji. Custom emojis are only available
// for Google Workspace accounts, and the administrator must turn custom emojis
// on for the organization. For more information, see Learn about custom emojis
// in Google Chat (https://support.google.com/chat/answer/12800149) and Manage
// custom emoji permissions (https://support.google.com/a/answer/12850085).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.customemojis.readonly` -
// `https://www.googleapis.com/auth/chat.customemojis`
//
//   - name: Resource name of the custom emoji. Format:
//     `customEmojis/{customEmoji}` You can use the emoji name as an alias for
//     `{customEmoji}`. For example, `customEmojis/:example-emoji:` where
//     `:example-emoji:` is the emoji name for a custom emoji.
func (r *CustomEmojisService) Get(name string) *CustomEmojisGetCall {
	c := &CustomEmojisGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CustomEmojisGetCall) Fields(s ...googleapi.Field) *CustomEmojisGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CustomEmojisGetCall) IfNoneMatch(entityTag string) *CustomEmojisGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CustomEmojisGetCall) Context(ctx context.Context) *CustomEmojisGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CustomEmojisGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CustomEmojisGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.customEmojis.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.customEmojis.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *CustomEmoji.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CustomEmojisGetCall) Do(opts ...googleapi.CallOption) (*CustomEmoji, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CustomEmoji{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.customEmojis.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CustomEmojisListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists custom emojis visible to the authenticated user. Custom emojis
// are only available for Google Workspace accounts, and the administrator must
// turn custom emojis on for the organization. For more information, see Learn
// about custom emojis in Google Chat
// (https://support.google.com/chat/answer/12800149) and Manage custom emoji
// permissions (https://support.google.com/a/answer/12850085). Requires user
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.customemojis.readonly` -
// `https://www.googleapis.com/auth/chat.customemojis`
func (r *CustomEmojisService) List() *CustomEmojisListCall {
	c := &CustomEmojisListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": A query filter. Supports
// filtering by creator. To filter by creator, you must specify a valid value.
// Currently only `creator("users/me")` and `NOT creator("users/me")` are
// accepted to filter custom emojis by whether they were created by the calling
// user or not. For example, the following query returns custom emojis created
// by the caller: ``` creator("users/me") ``` Invalid queries are rejected with
// an `INVALID_ARGUMENT` error.
func (c *CustomEmojisListCall) Filter(filter string) *CustomEmojisListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// custom emojis returned. The service can return fewer custom emojis than this
// value. If unspecified, the default value is 25. The maximum value is 200;
// values above 200 are changed to 200.
func (c *CustomEmojisListCall) PageSize(pageSize int64) *CustomEmojisListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": (If resuming from a
// previous query.) A page token received from a previous list custom emoji
// call. Provide this to retrieve the subsequent page. When paginating, the
// filter value should match the call that provided the page token. Passing a
// different value might lead to unexpected results.
func (c *CustomEmojisListCall) PageToken(pageToken string) *CustomEmojisListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CustomEmojisListCall) Fields(s ...googleapi.Field) *CustomEmojisListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CustomEmojisListCall) IfNoneMatch(entityTag string) *CustomEmojisListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CustomEmojisListCall) Context(ctx context.Context) *CustomEmojisListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CustomEmojisListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CustomEmojisListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/customEmojis")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.customEmojis.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.customEmojis.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListCustomEmojisResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *CustomEmojisListCall) Do(opts ...googleapi.CallOption) (*ListCustomEmojisResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListCustomEmojisResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.customEmojis.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *CustomEmojisListCall) Pages(ctx context.Context, f func(*ListCustomEmojisResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type MediaDownloadCall struct {
	s            *Service
	resourceName string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Download: Downloads media. Download is supported on the URI
// `/v1/media/{+name}?alt=media`.
//
//   - resourceName: Name of the media that is being downloaded. See
//     ReadRequest.resource_name.
func (r *MediaService) Download(resourceName string) *MediaDownloadCall {
	c := &MediaDownloadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resourceName = resourceName
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MediaDownloadCall) Fields(s ...googleapi.Field) *MediaDownloadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *MediaDownloadCall) IfNoneMatch(entityTag string) *MediaDownloadCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *MediaDownloadCall) Context(ctx context.Context) *MediaDownloadCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MediaDownloadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaDownloadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/media/{+resourceName}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resourceName": c.resourceName,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.media.download", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *MediaDownloadCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "chat.media.download" call.
// Any non-2xx status code is an error. Response headers are in either
// *Media.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *MediaDownloadCall) Do(opts ...googleapi.CallOption) (*Media, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Media{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.media.download", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type MediaUploadCall struct {
	s                       *Service
	parent                  string
	uploadattachmentrequest *UploadAttachmentRequest
	urlParams_              gensupport.URLParams
	mediaInfo_              *gensupport.MediaInfo
	ctx_                    context.Context
	header_                 http.Header
}

// Upload: Uploads an attachment. For an example, see Upload media as a file
// attachment
// (https://developers.google.com/workspace/chat/upload-media-attachments).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.messages.create` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) You
// can upload attachments up to 200 MB. Certain file types aren't supported.
// For details, see File types blocked by Google Chat
// (https://support.google.com/chat/answer/7651457?&co=GENIE.Platform%3DDesktop#File%20types%20blocked%20in%20Google%20Chat).
//
//   - parent: Resource name of the Chat space in which the attachment is
//     uploaded. Format "spaces/{space}".
func (r *MediaService) Upload(parent string, uploadattachmentrequest *UploadAttachmentRequest) *MediaUploadCall {
	c := &MediaUploadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.uploadattachmentrequest = uploadattachmentrequest
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *MediaUploadCall) Media(r io.Reader, options ...googleapi.MediaOption) *MediaUploadCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *MediaUploadCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *MediaUploadCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *MediaUploadCall) ProgressUpdater(pu googleapi.ProgressUpdater) *MediaUploadCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MediaUploadCall) Fields(s ...googleapi.Field) *MediaUploadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *MediaUploadCall) Context(ctx context.Context) *MediaUploadCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MediaUploadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaUploadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.uploadattachmentrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/attachments:upload")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/v1/{+parent}/attachments:upload")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.media.upload", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.media.upload" call.
// Any non-2xx status code is an error. Response headers are in either
// *UploadAttachmentResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *MediaUploadCall) Do(opts ...googleapi.CallOption) (*UploadAttachmentResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &UploadAttachmentResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.media.upload", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesCompleteImportCall struct {
	s                          *Service
	name                       string
	completeimportspacerequest *CompleteImportSpaceRequest
	urlParams_                 gensupport.URLParams
	ctx_                       context.Context
	header_                    http.Header
}

// CompleteImport: Completes the import process
// (https://developers.google.com/workspace/chat/import-data) for the specified
// space and makes it visible to users. Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// and domain-wide delegation with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.import` For more information, see
// Authorize Google Chat apps to import data
// (https://developers.google.com/workspace/chat/authorize-import).
//
// - name: Resource name of the import mode space. Format: `spaces/{space}`.
func (r *SpacesService) CompleteImport(name string, completeimportspacerequest *CompleteImportSpaceRequest) *SpacesCompleteImportCall {
	c := &SpacesCompleteImportCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.completeimportspacerequest = completeimportspacerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesCompleteImportCall) Fields(s ...googleapi.Field) *SpacesCompleteImportCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesCompleteImportCall) Context(ctx context.Context) *SpacesCompleteImportCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesCompleteImportCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesCompleteImportCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.completeimportspacerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:completeImport")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.completeImport", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.completeImport" call.
// Any non-2xx status code is an error. Response headers are in either
// *CompleteImportSpaceResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesCompleteImportCall) Do(opts ...googleapi.CallOption) (*CompleteImportSpaceResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CompleteImportSpaceResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.completeImport", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesCreateCall struct {
	s          *Service
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a space. Can be used to create a named space, or a group
// chat in `Import mode`. For an example, see Create a space
// (https://developers.google.com/workspace/chat/create-spaces). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.app.spaces.create` -
// `https://www.googleapis.com/auth/chat.app.spaces` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.create` -
// `https://www.googleapis.com/auth/chat.spaces` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) When
// authenticating as an app, the `space.customer` field must be set in the
// request. When authenticating as an app, the Chat app is added as a member of
// the space. However, unlike human authentication, the Chat app is not added
// as a space manager. By default, the Chat app can be removed from the space
// by all space members. To allow only space managers to remove the app from a
// space, set `space.permission_settings.manage_apps` to `managers_allowed`.
// Space membership upon creation depends on whether the space is created in
// `Import mode`: * **Import mode:** No members are created. * **All other
// modes:** The calling user is added as a member. This is: * The app itself
// when using app authentication. * The human user when using user
// authentication. If you receive the error message `ALREADY_EXISTS` when
// creating a space, try a different `displayName`. An existing space within
// the Google Workspace organization might already use this display name.
func (r *SpacesService) Create(space *Space) *SpacesCreateCall {
	c := &SpacesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.space = space
	return c
}

// RequestId sets the optional parameter "requestId": A unique identifier for
// this request. A random UUID is recommended. Specifying an existing request
// ID returns the space created with that ID instead of creating a new space.
// Specifying an existing request ID from the same Chat app with a different
// authenticated user returns an error.
func (c *SpacesCreateCall) RequestId(requestId string) *SpacesCreateCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesCreateCall) Fields(s ...googleapi.Field) *SpacesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesCreateCall) Context(ctx context.Context) *SpacesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.space)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesCreateCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a named space. Always performs a cascading delete, which
// means that the space's child resources—like messages posted in the space
// and memberships in the space—are also deleted. For an example, see Delete
// a space (https://developers.google.com/workspace/chat/delete-spaces).
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and the authorization scope: -
// `https://www.googleapis.com/auth/chat.app.delete` (only in spaces the app
// created) - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.delete` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) -
// User authentication grants administrator privileges when an administrator
// account authenticates, `use_admin_access` is `true`, and the following
// authorization scope is used: -
// `https://www.googleapis.com/auth/chat.admin.delete`
//
// - name: Resource name of the space to delete. Format: `spaces/{space}`.
func (r *SpacesService) Delete(name string) *SpacesDeleteCall {
	c := &SpacesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.delete` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
func (c *SpacesDeleteCall) UseAdminAccess(useAdminAccess bool) *SpacesDeleteCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesDeleteCall) Fields(s ...googleapi.Field) *SpacesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesDeleteCall) Context(ctx context.Context) *SpacesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesFindDirectMessageCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// FindDirectMessage: Returns the existing direct message with the specified
// user. If no direct message space is found, returns a `404 NOT_FOUND` error.
// For an example, see Find a direct message
// (/chat/api/guides/v1/spaces/find-direct-message). With app authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app),
// returns the direct message space between the specified user and the calling
// Chat app. With user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
// returns the direct message space between the specified user and the
// authenticated user. Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.spaces`
func (r *SpacesService) FindDirectMessage() *SpacesFindDirectMessageCall {
	c := &SpacesFindDirectMessageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Name sets the optional parameter "name": Required. Resource name of the user
// to find direct message with. Format: `users/{user}`, where `{user}` is
// either the `id` for the person
// (https://developers.google.com/people/api/rest/v1/people) from the People
// API, or the `id` for the user
// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users)
// in the Directory API. For example, if the People API profile ID is
// `123456789`, you can find a direct message with that person by using
// `users/123456789` as the `name`. When authenticated as a user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
// you can use the email as an alias for `{user}`. For example,
// `users/example@gmail.com` where `example@gmail.com` is the email of the
// Google Chat user.
func (c *SpacesFindDirectMessageCall) Name(name string) *SpacesFindDirectMessageCall {
	c.urlParams_.Set("name", name)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesFindDirectMessageCall) Fields(s ...googleapi.Field) *SpacesFindDirectMessageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesFindDirectMessageCall) IfNoneMatch(entityTag string) *SpacesFindDirectMessageCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesFindDirectMessageCall) Context(ctx context.Context) *SpacesFindDirectMessageCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesFindDirectMessageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesFindDirectMessageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces:findDirectMessage")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.findDirectMessage", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.findDirectMessage" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesFindDirectMessageCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.findDirectMessage", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a space. For an example, see Get details about a
// space (https://developers.google.com/workspace/chat/get-spaces). Supports
// the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.bot` -
// `https://www.googleapis.com/auth/chat.app.spaces` with administrator
// approval (https://support.google.com/a?p=chat-app-auth) - User
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.spaces` - User authentication grants
// administrator privileges when an administrator account authenticates,
// `use_admin_access` is `true`, and one of the following authorization scopes
// is used: - `https://www.googleapis.com/auth/chat.admin.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.admin.spaces` App authentication has
// the following limitations: - `space.access_settings` is only populated when
// using the `chat.app.spaces` scope. - `space.predefind_permission_settings`
// and `space.permission_settings` are only populated when using the
// `chat.app.spaces` scope, and only for spaces the app created.
//
//   - name: Resource name of the space, in the form `spaces/{space}`. Format:
//     `spaces/{space}`.
func (r *SpacesService) Get(name string) *SpacesGetCall {
	c := &SpacesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.spaces` or `chat.admin.spaces.readonly` OAuth 2.0 scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
func (c *SpacesGetCall) UseAdminAccess(useAdminAccess bool) *SpacesGetCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesGetCall) Fields(s ...googleapi.Field) *SpacesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesGetCall) IfNoneMatch(entityTag string) *SpacesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesGetCall) Context(ctx context.Context) *SpacesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesGetCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists spaces the caller is a member of. Group chats and DMs aren't
// listed until the first message is sent. For an example, see List spaces
// (https://developers.google.com/workspace/chat/list-spaces). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.spaces` To list all named spaces by
// Google Workspace organization, use the `spaces.search()`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces/search)
// method using Workspace administrator privileges instead.
func (r *SpacesService) List() *SpacesListCall {
	c := &SpacesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can filter
// spaces by the space type (`space_type`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces#spacetype)).
// To filter by space type, you must specify valid enum value, such as `SPACE`
// or `GROUP_CHAT` (the `space_type` can't be `SPACE_TYPE_UNSPECIFIED`). To
// query for multiple space types, use the `OR` operator. For example, the
// following queries are valid: ``` space_type = "SPACE" spaceType =
// "GROUP_CHAT" OR spaceType = "DIRECT_MESSAGE" ``` Invalid queries are
// rejected by the server with an `INVALID_ARGUMENT` error.
func (c *SpacesListCall) Filter(filter string) *SpacesListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// spaces to return. The service might return fewer than this value. If
// unspecified, at most 100 spaces are returned. The maximum value is 1000. If
// you use a value more than 1000, it's automatically changed to 1000. Negative
// values return an `INVALID_ARGUMENT` error.
func (c *SpacesListCall) PageSize(pageSize int64) *SpacesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous list spaces call. Provide this parameter to retrieve the
// subsequent page. When paginating, the filter value should match the call
// that provided the page token. Passing a different value may lead to
// unexpected results.
func (c *SpacesListCall) PageToken(pageToken string) *SpacesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesListCall) Fields(s ...googleapi.Field) *SpacesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesListCall) IfNoneMatch(entityTag string) *SpacesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesListCall) Context(ctx context.Context) *SpacesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSpacesResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SpacesListCall) Do(opts ...googleapi.CallOption) (*ListSpacesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSpacesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesListCall) Pages(ctx context.Context, f func(*ListSpacesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesPatchCall struct {
	s          *Service
	name       string
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a space. For an example, see Update a space
// (https://developers.google.com/workspace/chat/update-spaces). If you're
// updating the `displayName` field and receive the error message
// `ALREADY_EXISTS`, try a different display name.. An existing space within
// the Google Workspace organization might already use this display name.
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.app.spaces` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) -
// User authentication grants administrator privileges when an administrator
// account authenticates, `use_admin_access` is `true`, and the following
// authorization scopes is used: -
// `https://www.googleapis.com/auth/chat.admin.spaces` App authentication has
// the following limitations: - To update either
// `space.predefined_permission_settings` or `space.permission_settings`, the
// app must be the space creator. - Updating the
// `space.access_settings.audience` is not supported for app authentication.
//
//   - name: Identifier. Resource name of the space. Format: `spaces/{space}`
//     Where `{space}` represents the system-assigned ID for the space. You can
//     obtain the space ID by calling the `spaces.list()`
//     (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces/list)
//     method or from the space URL. For example, if the space URL is
//     `https://mail.google.com/mail/u/0/#chat/space/AAAAAAAAA`, the space ID is
//     `AAAAAAAAA`.
func (r *SpacesService) Patch(name string, space *Space) *SpacesPatchCall {
	c := &SpacesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.space = space
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The updated
// field paths, comma separated if there are multiple. You can update the
// following fields for a space: `space_details`: Updates the space's
// description. Supports up to 150 characters. `display_name`: Only supports
// updating the display name for spaces where `spaceType` field is `SPACE`. If
// you receive the error message `ALREADY_EXISTS`, try a different value. An
// existing space within the Google Workspace organization might already use
// this display name. `space_type`: Only supports changing a `GROUP_CHAT` space
// type to `SPACE`. Include `display_name` together with `space_type` in the
// update mask and ensure that the specified space has a non-empty display name
// and the `SPACE` space type. Including the `space_type` mask and the `SPACE`
// type in the specified space when updating the display name is optional if
// the existing space already has the `SPACE` type. Trying to update the space
// type in other ways results in an invalid argument error. `space_type` is not
// supported with `useAdminAccess`. `space_history_state`: Updates space
// history settings (https://support.google.com/chat/answer/7664687) by turning
// history on or off for the space. Only supported if history settings are
// enabled for the Google Workspace organization. To update the space history
// state, you must omit all other field masks in your request.
// `space_history_state` is not supported with `useAdminAccess`.
// `access_settings.audience`: Updates the access setting
// (https://support.google.com/chat/answer/11971020) of who can discover the
// space, join the space, and preview the messages in named space where
// `spaceType` field is `SPACE`. If the existing space has a target audience,
// you can remove the audience and restrict space access by omitting a value
// for this field mask. To update access settings for a space, the
// authenticating user must be a space manager and omit all other field masks
// in your request. You can't update this field if the space is in import mode
// (https://developers.google.com/workspace/chat/import-data-overview). To
// learn more, see Make a space discoverable to specific users
// (https://developers.google.com/workspace/chat/space-target-audience).
// `access_settings.audience` is not supported with `useAdminAccess`.
// `permission_settings`: Supports changing the permission settings
// (https://support.google.com/chat/answer/13340792) of a space. When updating
// permission settings, you can only specify `permissionSettings` field masks;
// you cannot update other field masks at the same time. `permissionSettings`
// is not supported with `useAdminAccess`. The supported field masks include: -
// `permission_settings.manageMembersAndGroups` -
// `permission_settings.modifySpaceDetails` -
// `permission_settings.toggleHistory` - `permission_settings.useAtMentionAll`
// - `permission_settings.manageApps` - `permission_settings.manageWebhooks` -
// `permission_settings.replyMessages`
func (c *SpacesPatchCall) UpdateMask(updateMask string) *SpacesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.spaces` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// Some `FieldMask` values are not supported using admin access. For details,
// see the description of `update_mask`.
func (c *SpacesPatchCall) UseAdminAccess(useAdminAccess bool) *SpacesPatchCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesPatchCall) Fields(s ...googleapi.Field) *SpacesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesPatchCall) Context(ctx context.Context) *SpacesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.space)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesPatchCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesSearchCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Search: Returns a list of spaces in a Google Workspace organization based on
// an administrator's search. In the request, set `use_admin_access` to `true`.
// For an example, see Search for and manage spaces
// (https://developers.google.com/workspace/chat/search-manage-admin). Requires
// user authentication with administrator privileges
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user#admin-privileges)
// and one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.admin.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.admin.spaces`
func (r *SpacesService) Search() *SpacesSearchCall {
	c := &SpacesSearchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// OrderBy sets the optional parameter "orderBy": How the list of spaces is
// ordered. Supported attributes to order by are: -
// `membership_count.joined_direct_human_user_count` — Denotes the count of
// human users that have directly joined a space. - `last_active_time` —
// Denotes the time when last eligible item is added to any topic of this
// space. - `create_time` — Denotes the time of the space creation. Valid
// ordering operation values are: - `ASC` for ascending. Default value. -
// `DESC` for descending. The supported syntax are: -
// `membership_count.joined_direct_human_user_count DESC` -
// `membership_count.joined_direct_human_user_count ASC` - `last_active_time
// DESC` - `last_active_time ASC` - `create_time DESC` - `create_time ASC`
func (c *SpacesSearchCall) OrderBy(orderBy string) *SpacesSearchCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// spaces to return. The service may return fewer than this value. If
// unspecified, at most 100 spaces are returned. The maximum value is 1000. If
// you use a value more than 1000, it's automatically changed to 1000.
func (c *SpacesSearchCall) PageSize(pageSize int64) *SpacesSearchCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token, received from
// the previous search spaces call. Provide this parameter to retrieve the
// subsequent page. When paginating, all other parameters provided should match
// the call that provided the page token. Passing different values to the other
// parameters might lead to unexpected results.
func (c *SpacesSearchCall) PageToken(pageToken string) *SpacesSearchCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Query sets the optional parameter "query": Required. A search query. You can
// search by using the following parameters: - `create_time` - `customer` -
// `display_name` - `external_user_allowed` - `last_active_time` -
// `space_history_state` - `space_type` `create_time` and `last_active_time`
// accept a timestamp in RFC-3339 (https://www.rfc-editor.org/rfc/rfc3339)
// format and the supported comparison operators are: `=`, `<`, `>`, `<=`,
// `>=`. `customer` is required and is used to indicate which customer to fetch
// spaces from. `customers/my_customer` is the only supported value.
// `display_name` only accepts the `HAS` (`:`) operator. The text to match is
// first tokenized into tokens and each token is prefix-matched
// case-insensitively and independently as a substring anywhere in the space's
// `display_name`. For example, `Fun Eve` matches `Fun event` or `The evening
// was fun`, but not `notFun event` or `even`. `external_user_allowed` accepts
// either `true` or `false`. `space_history_state` only accepts values from the
// [`historyState`]
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces#Space.HistoryState)
// field of a `space` resource. `space_type` is required and the only valid
// value is `SPACE`. Across different fields, only `AND` operators are
// supported. A valid example is `space_type = "SPACE" AND
// display_name:"Hello" and an invalid example is `space_type = "SPACE" OR
// display_name:"Hello". Among the same field, `space_type` doesn't support
// `AND` or `OR` operators. `display_name`, 'space_history_state', and
// 'external_user_allowed' only support `OR` operators. `last_active_time` and
// `create_time` support both `AND` and `OR` operators. `AND` can only be used
// to represent an interval, such as `last_active_time <
// "2022-01-01T00:00:00+00:00" AND last_active_time >
// "2023-01-01T00:00:00+00:00". The following example queries are valid: ```
// customer = "customers/my_customer" AND space_type = "SPACE" customer =
// "customers/my_customer" AND space_type = "SPACE" AND display_name:"Hello
// World" customer = "customers/my_customer" AND space_type = "SPACE" AND
// (last_active_time < "2020-01-01T00:00:00+00:00" OR last_active_time >
// "2022-01-01T00:00:00+00:00") customer = "customers/my_customer" AND
// space_type = "SPACE" AND (display_name:"Hello World" OR display_name:"Fun
// event") AND (last_active_time > "2020-01-01T00:00:00+00:00" AND
// last_active_time < "2022-01-01T00:00:00+00:00") customer =
// "customers/my_customer" AND space_type = "SPACE" AND (create_time >
// "2019-01-01T00:00:00+00:00" AND create_time < "2020-01-01T00:00:00+00:00")
// AND (external_user_allowed = "true") AND (space_history_state = "HISTORY_ON"
// OR space_history_state = "HISTORY_OFF") ```
func (c *SpacesSearchCall) Query(query string) *SpacesSearchCall {
	c.urlParams_.Set("query", query)
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires either the
// `chat.admin.spaces.readonly` or `chat.admin.spaces` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// This method currently only supports admin access, thus only `true` is
// accepted for this field.
func (c *SpacesSearchCall) UseAdminAccess(useAdminAccess bool) *SpacesSearchCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesSearchCall) Fields(s ...googleapi.Field) *SpacesSearchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesSearchCall) IfNoneMatch(entityTag string) *SpacesSearchCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesSearchCall) Context(ctx context.Context) *SpacesSearchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesSearchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesSearchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces:search")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.search", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.search" call.
// Any non-2xx status code is an error. Response headers are in either
// *SearchSpacesResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SpacesSearchCall) Do(opts ...googleapi.CallOption) (*SearchSpacesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SearchSpacesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.search", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesSearchCall) Pages(ctx context.Context, f func(*SearchSpacesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesSetupCall struct {
	s                 *Service
	setupspacerequest *SetUpSpaceRequest
	urlParams_        gensupport.URLParams
	ctx_              context.Context
	header_           http.Header
}

// Setup: Creates a space and adds specified users to it. The calling user is
// automatically added to the space, and shouldn't be specified as a membership
// in the request. For an example, see Set up a space with initial members
// (https://developers.google.com/workspace/chat/set-up-spaces). To specify the
// human members to add, add memberships with the appropriate
// `membership.member.name`. To add a human user, use `users/{user}`, where
// `{user}` can be the email address for the user. For users in the same
// Workspace organization `{user}` can also be the `id` for the person from the
// People API, or the `id` for the user in the Directory API. For example, if
// the People API Person profile ID for `user@example.com` is `123456789`, you
// can add the user to the space by setting the `membership.member.name` to
// `users/user@example.com` or `users/123456789`. To specify the Google groups
// to add, add memberships with the appropriate `membership.group_member.name`.
// To add or invite a Google group, use `groups/{group}`, where `{group}` is
// the `id` for the group from the Cloud Identity Groups API. For example, you
// can use Cloud Identity Groups lookup API
// (https://cloud.google.com/identity/docs/reference/rest/v1/groups/lookup) to
// retrieve the ID `123456789` for group email `group@example.com`, then you
// can add the group to the space by setting the `membership.group_member.name`
// to `groups/123456789`. Group email is not supported, and Google groups can
// only be added as members in named spaces. For a named space or group chat,
// if the caller blocks, or is blocked by some members, or doesn't have
// permission to add some members, then those members aren't added to the
// created space. To create a direct message (DM) between the calling user and
// another human user, specify exactly one membership to represent the human
// user. If one user blocks the other, the request fails and the DM isn't
// created. To create a DM between the calling user and the calling app, set
// `Space.singleUserBotDm` to `true` and don't specify any memberships. You can
// only use this method to set up a DM with the calling app. To add the calling
// app as a member of a space or an existing DM between two human users, see
// Invite or add a user or app to a space
// (https://developers.google.com/workspace/chat/create-members). If a DM
// already exists between two users, even when one user blocks the other at the
// time a request is made, then the existing DM is returned. Spaces with
// threaded replies aren't supported. If you receive the error message
// `ALREADY_EXISTS` when setting up a space, try a different `displayName`. An
// existing space within the Google Workspace organization might already use
// this display name. Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.spaces.create` -
// `https://www.googleapis.com/auth/chat.spaces`
func (r *SpacesService) Setup(setupspacerequest *SetUpSpaceRequest) *SpacesSetupCall {
	c := &SpacesSetupCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.setupspacerequest = setupspacerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesSetupCall) Fields(s ...googleapi.Field) *SpacesSetupCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesSetupCall) Context(ctx context.Context) *SpacesSetupCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesSetupCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesSetupCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.setupspacerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces:setup")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.setup", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.setup" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesSetupCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.setup", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMembersCreateCall struct {
	s          *Service
	parent     string
	membership *Membership
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a membership for the calling Chat app, a user, or a Google
// Group. Creating memberships for other Chat apps isn't supported. When
// creating a membership, if the specified member has their auto-accept policy
// turned off, then they're invited, and must accept the space invitation
// before joining. Otherwise, creating a membership adds the member directly to
// the specified space. Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and the authorization scope: -
// `https://www.googleapis.com/auth/chat.app.memberships` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.memberships` -
// `https://www.googleapis.com/auth/chat.memberships.app` (to add the calling
// app to the space) - `https://www.googleapis.com/auth/chat.import` (import
// mode spaces only) - User authentication grants administrator privileges when
// an administrator account authenticates, `use_admin_access` is `true`, and
// the following authorization scope is used: -
// `https://www.googleapis.com/auth/chat.admin.memberships` App authentication
// is not supported for the following use cases: - Inviting users external to
// the Workspace organization that owns the space. - Adding a Google Group to a
// space. - Adding a Chat app to a space. For example usage, see: - Invite or
// add a user to a space
// (https://developers.google.com/workspace/chat/create-members#create-user-membership).
// - Invite or add a Google Group to a space
// (https://developers.google.com/workspace/chat/create-members#create-group-membership).
// - Add the Chat app to a space
// (https://developers.google.com/workspace/chat/create-members#create-membership-calling-api).
//
//   - parent: The resource name of the space for which to create the membership.
//     Format: spaces/{space}.
func (r *SpacesMembersService) Create(parent string, membership *Membership) *SpacesMembersCreateCall {
	c := &SpacesMembersCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.membership = membership
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.memberships` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// Creating app memberships or creating memberships for users outside the
// administrator's Google Workspace organization isn't supported using admin
// access.
func (c *SpacesMembersCreateCall) UseAdminAccess(useAdminAccess bool) *SpacesMembersCreateCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMembersCreateCall) Fields(s ...googleapi.Field) *SpacesMembersCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMembersCreateCall) Context(ctx context.Context) *SpacesMembersCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMembersCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.membership)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/members")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.members.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMembersCreateCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.members.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMembersDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a membership. For an example, see Remove a user or a Google
// Chat app from a space
// (https://developers.google.com/workspace/chat/delete-members). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and the authorization scope: -
// `https://www.googleapis.com/auth/chat.app.memberships` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.memberships` -
// `https://www.googleapis.com/auth/chat.memberships.app` (to remove the
// calling app from the space) - `https://www.googleapis.com/auth/chat.import`
// (import mode spaces only) - User authentication grants administrator
// privileges when an administrator account authenticates, `use_admin_access`
// is `true`, and the following authorization scope is used: -
// `https://www.googleapis.com/auth/chat.admin.memberships` App authentication
// is not supported for the following use cases: - Removing a Google Group from
// a space. - Removing a Chat app from a space. To delete memberships for space
// managers, the requester must be a space manager. If you're using app
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// the Chat app must be the space creator.
//
//   - name: Resource name of the membership to delete. Chat apps can delete
//     human users' or their own memberships. Chat apps can't delete other apps'
//     memberships. When deleting a human membership, requires the
//     `chat.memberships` scope with user authentication
//     (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
//     or the `chat.memberships.app` scope with app authentication
//     (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
//     and the `spaces/{space}/members/{member}` format. You can use the email as
//     an alias for `{member}`. For example,
//     `spaces/{space}/members/example@gmail.com` where `example@gmail.com` is
//     the email of the Google Chat user. When deleting an app membership,
//     requires the `chat.memberships.app` scope and `spaces/{space}/members/app`
//     format. Format: `spaces/{space}/members/{member}` or
//     `spaces/{space}/members/app`.
func (r *SpacesMembersService) Delete(name string) *SpacesMembersDeleteCall {
	c := &SpacesMembersDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.memberships` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// Deleting app memberships in a space isn't supported using admin access.
func (c *SpacesMembersDeleteCall) UseAdminAccess(useAdminAccess bool) *SpacesMembersDeleteCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMembersDeleteCall) Fields(s ...googleapi.Field) *SpacesMembersDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMembersDeleteCall) Context(ctx context.Context) *SpacesMembersDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMembersDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.members.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMembersDeleteCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.members.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMembersGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a membership. For an example, see Get details
// about a user's or Google Chat app's membership
// (https://developers.google.com/workspace/chat/get-members). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.bot` -
// `https://www.googleapis.com/auth/chat.app.memberships` (requires
// administrator approval (https://support.google.com/a?p=chat-app-auth)) -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.memberships.readonly` -
// `https://www.googleapis.com/auth/chat.memberships` - User authentication
// grants administrator privileges when an administrator account authenticates,
// `use_admin_access` is `true`, and one of the following authorization scopes
// is used: - `https://www.googleapis.com/auth/chat.admin.memberships.readonly`
// - `https://www.googleapis.com/auth/chat.admin.memberships`
//
//   - name: Resource name of the membership to retrieve. To get the app's own
//     membership by using user authentication
//     (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user),
//     you can optionally use `spaces/{space}/members/app`. Format:
//     `spaces/{space}/members/{member}` or `spaces/{space}/members/app` You can
//     use the user's email as an alias for `{member}`. For example,
//     `spaces/{space}/members/example@gmail.com` where `example@gmail.com` is
//     the email of the Google Chat user.
func (r *SpacesMembersService) Get(name string) *SpacesMembersGetCall {
	c := &SpacesMembersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.memberships` or `chat.admin.memberships.readonly` OAuth 2.0
// scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// Getting app memberships in a space isn't supported when using admin access.
func (c *SpacesMembersGetCall) UseAdminAccess(useAdminAccess bool) *SpacesMembersGetCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMembersGetCall) Fields(s ...googleapi.Field) *SpacesMembersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMembersGetCall) IfNoneMatch(entityTag string) *SpacesMembersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMembersGetCall) Context(ctx context.Context) *SpacesMembersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMembersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.members.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMembersGetCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.members.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMembersListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists memberships in a space. For an example, see List users and
// Google Chat apps in a space
// (https://developers.google.com/workspace/chat/list-members). Listing
// memberships with app authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// lists memberships in spaces that the Chat app has access to, but excludes
// Chat app memberships, including its own. Listing memberships with User
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// lists memberships in spaces that the authenticated user has access to.
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.bot` -
// `https://www.googleapis.com/auth/chat.app.memberships` (requires
// administrator approval (https://support.google.com/a?p=chat-app-auth)) -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.memberships.readonly` -
// `https://www.googleapis.com/auth/chat.memberships` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) -
// User authentication grants administrator privileges when an administrator
// account authenticates, `use_admin_access` is `true`, and one of the
// following authorization scopes is used: -
// `https://www.googleapis.com/auth/chat.admin.memberships.readonly` -
// `https://www.googleapis.com/auth/chat.admin.memberships`
//
//   - parent: The resource name of the space for which to fetch a membership
//     list. Format: spaces/{space}.
func (r *SpacesMembersService) List(parent string) *SpacesMembersListCall {
	c := &SpacesMembersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can filter
// memberships by a member's role (`role`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.members#membershiprole))
// and type (`member.type`
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/User#type)).
// To filter by role, set `role` to `ROLE_MEMBER` or `ROLE_MANAGER`. To filter
// by type, set `member.type` to `HUMAN` or `BOT`. You can also filter for
// `member.type` using the `!=` operator. To filter by both role and type, use
// the `AND` operator. To filter by either role or type, use the `OR` operator.
// Either `member.type = "HUMAN" or `member.type != "BOT" is required when
// `use_admin_access` is set to true. Other member type filters will be
// rejected. For example, the following queries are valid: ``` role =
// "ROLE_MANAGER" OR role = "ROLE_MEMBER" member.type = "HUMAN" AND role =
// "ROLE_MANAGER" member.type != "BOT" ``` The following queries are invalid:
// ``` member.type = "HUMAN" AND member.type = "BOT" role = "ROLE_MANAGER" AND
// role = "ROLE_MEMBER" ``` Invalid queries are rejected by the server with an
// `INVALID_ARGUMENT` error.
func (c *SpacesMembersListCall) Filter(filter string) *SpacesMembersListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// memberships to return. The service might return fewer than this value. If
// unspecified, at most 100 memberships are returned. The maximum value is
// 1000. If you use a value more than 1000, it's automatically changed to 1000.
// Negative values return an `INVALID_ARGUMENT` error.
func (c *SpacesMembersListCall) PageSize(pageSize int64) *SpacesMembersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous call to list memberships. Provide this parameter to retrieve
// the subsequent page. When paginating, all other parameters provided should
// match the call that provided the page token. Passing different values to the
// other parameters might lead to unexpected results.
func (c *SpacesMembersListCall) PageToken(pageToken string) *SpacesMembersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// ShowGroups sets the optional parameter "showGroups": When `true`, also
// returns memberships associated with a Google Group, in addition to other
// types of memberships. If a filter is set, Google Group memberships that
// don't match the filter criteria aren't returned.
func (c *SpacesMembersListCall) ShowGroups(showGroups bool) *SpacesMembersListCall {
	c.urlParams_.Set("showGroups", fmt.Sprint(showGroups))
	return c
}

// ShowInvited sets the optional parameter "showInvited": When `true`, also
// returns memberships associated with invited members, in addition to other
// types of memberships. If a filter is set, invited memberships that don't
// match the filter criteria aren't returned. Currently requires user
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
func (c *SpacesMembersListCall) ShowInvited(showInvited bool) *SpacesMembersListCall {
	c.urlParams_.Set("showInvited", fmt.Sprint(showInvited))
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires either the
// `chat.admin.memberships.readonly` or `chat.admin.memberships` OAuth 2.0
// scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
// Listing app memberships in a space isn't supported when using admin access.
func (c *SpacesMembersListCall) UseAdminAccess(useAdminAccess bool) *SpacesMembersListCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMembersListCall) Fields(s ...googleapi.Field) *SpacesMembersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMembersListCall) IfNoneMatch(entityTag string) *SpacesMembersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMembersListCall) Context(ctx context.Context) *SpacesMembersListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMembersListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/members")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.members.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListMembershipsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMembersListCall) Do(opts ...googleapi.CallOption) (*ListMembershipsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMembershipsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.members.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMembersListCall) Pages(ctx context.Context, f func(*ListMembershipsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesMembersPatchCall struct {
	s          *Service
	name       string
	membership *Membership
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a membership. For an example, see Update a user's membership
// in a space (https://developers.google.com/workspace/chat/update-members).
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// and the authorization scope: -
// `https://www.googleapis.com/auth/chat.app.memberships` (only in spaces the
// app created) - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.memberships` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) -
// User authentication grants administrator privileges when an administrator
// account authenticates, `use_admin_access` is `true`, and the following
// authorization scope is used: -
// `https://www.googleapis.com/auth/chat.admin.memberships`
//
//   - name: Identifier. Resource name of the membership, assigned by the server.
//     Format: `spaces/{space}/members/{member}`.
func (r *SpacesMembersService) Patch(name string, membership *Membership) *SpacesMembersPatchCall {
	c := &SpacesMembersPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.membership = membership
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The field
// paths to update. Separate multiple values with commas or use `*` to update
// all field paths. Currently supported field paths: - `role`
func (c *SpacesMembersPatchCall) UpdateMask(updateMask string) *SpacesMembersPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// UseAdminAccess sets the optional parameter "useAdminAccess": When `true`,
// the method runs using the user's Google Workspace administrator privileges.
// The calling user must be a Google Workspace administrator with the manage
// chat and spaces conversations privilege
// (https://support.google.com/a/answer/13369245). Requires the
// `chat.admin.memberships` OAuth 2.0 scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes).
func (c *SpacesMembersPatchCall) UseAdminAccess(useAdminAccess bool) *SpacesMembersPatchCall {
	c.urlParams_.Set("useAdminAccess", fmt.Sprint(useAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMembersPatchCall) Fields(s ...googleapi.Field) *SpacesMembersPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMembersPatchCall) Context(ctx context.Context) *SpacesMembersPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMembersPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.membership)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.members.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMembersPatchCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.members.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesCreateCall struct {
	s          *Service
	parent     string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a message in a Google Chat space. For an example, see Send a
// message (https://developers.google.com/workspace/chat/create-messages).
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages.create` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) Chat
// attributes the message sender differently depending on the type of
// authentication that you use in your request. The following image shows how
// Chat attributes a message when you use app authentication. Chat displays the
// Chat app as the message sender. The content of the message can contain text
// (`text`), cards (`cardsV2`), and accessory widgets (`accessoryWidgets`).
// !Message sent with app authentication
// (https://developers.google.com/workspace/chat/images/message-app-auth.svg)
// The following image shows how Chat attributes a message when you use user
// authentication. Chat displays the user as the message sender and attributes
// the Chat app to the message by displaying its name. The content of message
// can only contain text (`text`). !Message sent with user authentication
// (https://developers.google.com/workspace/chat/images/message-user-auth.svg)
// The maximum message size, including the message contents, is 32,000 bytes.
// For webhook
// (https://developers.google.com/workspace/chat/quickstart/webhooks) requests,
// the response doesn't contain the full message. The response only populates
// the `name` and `thread.name` fields in addition to the information that was
// in the request.
//
//   - parent: The resource name of the space in which to create a message.
//     Format: `spaces/{space}`.
func (r *SpacesMessagesService) Create(parent string, message *Message) *SpacesMessagesCreateCall {
	c := &SpacesMessagesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.message = message
	return c
}

// MessageId sets the optional parameter "messageId": A custom ID for a
// message. Lets Chat apps get, update, or delete a message without needing to
// store the system-assigned ID in the message's resource name (represented in
// the message `name` field). The value for this field must meet the following
// requirements: * Begins with `client-`. For example, `client-custom-name` is
// a valid custom ID, but `custom-name` is not. * Contains up to 63 characters
// and only lowercase letters, numbers, and hyphens. * Is unique within a
// space. A Chat app can't use the same custom ID for different messages. For
// details, see Name a message
// (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
func (c *SpacesMessagesCreateCall) MessageId(messageId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageId", messageId)
	return c
}

// MessageReplyOption sets the optional parameter "messageReplyOption":
// Specifies whether a message starts a thread or replies to one. Only
// supported in named spaces. When responding to user interactions
// (https://developers.google.com/workspace/chat/receive-respond-interactions),
// this field is ignored. For interactions within a thread, the reply is
// created in the same thread. Otherwise, the reply is created as a new thread.
//
// Possible values:
//
//	"MESSAGE_REPLY_OPTION_UNSPECIFIED" - Default. Starts a new thread. Using
//
// this option ignores any thread ID or `thread_key` that's included.
//
//	"REPLY_MESSAGE_FALLBACK_TO_NEW_THREAD" - Creates the message as a reply to
//
// the thread specified by thread ID or `thread_key`. If it fails, the message
// starts a new thread instead.
//
//	"REPLY_MESSAGE_OR_FAIL" - Creates the message as a reply to the thread
//
// specified by thread ID or `thread_key`. If a new `thread_key` is used, a new
// thread is created. If the message creation fails, a `NOT_FOUND` error is
// returned instead.
func (c *SpacesMessagesCreateCall) MessageReplyOption(messageReplyOption string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageReplyOption", messageReplyOption)
	return c
}

// RequestId sets the optional parameter "requestId": A unique request ID for
// this message. Specifying an existing request ID returns the message created
// with that ID instead of creating a new message.
func (c *SpacesMessagesCreateCall) RequestId(requestId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// ThreadKey sets the optional parameter "threadKey": Deprecated: Use
// thread.thread_key instead. ID for the thread. Supports up to 4000
// characters. To start or add to a thread, create a message and specify a
// `threadKey` or the thread.name. For example usage, see Start or reply to a
// message thread
// (https://developers.google.com/workspace/chat/create-messages#create-message-thread).
func (c *SpacesMessagesCreateCall) ThreadKey(threadKey string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("threadKey", threadKey)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesCreateCall) Fields(s ...googleapi.Field) *SpacesMessagesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesCreateCall) Context(ctx context.Context) *SpacesMessagesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.message)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/messages")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesCreateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a message. For an example, see Delete a message
// (https://developers.google.com/workspace/chat/delete-messages). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) When
// using app authentication, requests can only delete messages created by the
// calling Chat app.
//
//   - name: Resource name of the message. Format:
//     `spaces/{space}/messages/{message}` If you've set a custom ID for your
//     message, you can use the value from the `clientAssignedMessageId` field
//     for `{message}`. For details, see [Name a message]
//     (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
func (r *SpacesMessagesService) Delete(name string) *SpacesMessagesDeleteCall {
	c := &SpacesMessagesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Force sets the optional parameter "force": When `true`, deleting a message
// also deletes its threaded replies. When `false`, if a message has threaded
// replies, deletion fails. Only applies when authenticating as a user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
// Has no effect when [authenticating as a Chat app]
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
func (c *SpacesMessagesDeleteCall) Force(force bool) *SpacesMessagesDeleteCall {
	c.urlParams_.Set("force", fmt.Sprint(force))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesDeleteCall) Fields(s ...googleapi.Field) *SpacesMessagesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesDeleteCall) Context(ctx context.Context) *SpacesMessagesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a message. For an example, see Get details about
// a message (https://developers.google.com/workspace/chat/get-messages).
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.bot`: When using this authorization
// scope, this method returns details about a message the Chat app has access
// to, like direct messages and slash commands
// (https://developers.google.com/workspace/chat/slash-commands) that invoke
// the Chat app. - `https://www.googleapis.com/auth/chat.app.messages.readonly`
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// (available in Developer Preview
// (https://developers.google.com/workspace/preview)). When using this
// authentication scope, this method returns details about a public message in
// a space. - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages.readonly` -
// `https://www.googleapis.com/auth/chat.messages` Note: Might return a message
// from a blocked member or space.
//
//   - name: Resource name of the message. Format:
//     `spaces/{space}/messages/{message}` If you've set a custom ID for your
//     message, you can use the value from the `clientAssignedMessageId` field
//     for `{message}`. For details, see [Name a message]
//     (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
func (r *SpacesMessagesService) Get(name string) *SpacesMessagesGetCall {
	c := &SpacesMessagesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesGetCall) Fields(s ...googleapi.Field) *SpacesMessagesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMessagesGetCall) IfNoneMatch(entityTag string) *SpacesMessagesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesGetCall) Context(ctx context.Context) *SpacesMessagesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesGetCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists messages in a space that the caller is a member of, including
// messages from blocked members and spaces. System messages, like those
// announcing new space members, aren't included. If you list messages from a
// space with no messages, the response is an empty object. When using a
// REST/HTTP interface, the response contains an empty JSON object, `{}`. For
// an example, see List messages
// (https://developers.google.com/workspace/chat/api/guides/v1/messages/list).
// Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// in Developer Preview (https://developers.google.com/workspace/preview) with
// the authorization scope: -
// `https://www.googleapis.com/auth/chat.app.messages.readonly`. When using
// this authentication scope, this method only returns public messages in a
// space. It doesn't include private messages. - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages.readonly` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only)
//
//   - parent: The resource name of the space to list messages from. Format:
//     `spaces/{space}`.
func (r *SpacesMessagesService) List(parent string) *SpacesMessagesListCall {
	c := &SpacesMessagesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can filter
// messages by date (`create_time`) and thread (`thread.name`). To filter
// messages by the date they were created, specify the `create_time` with a
// timestamp in RFC-3339 (https://www.rfc-editor.org/rfc/rfc3339) format and
// double quotation marks. For example, "2023-04-21T11:30:00-04:00". You can
// use the greater than operator `>` to list messages that were created after a
// timestamp, or the less than operator `<` to list messages that were created
// before a timestamp. To filter messages within a time interval, use the `AND`
// operator between two timestamps. To filter by thread, specify the
// `thread.name`, formatted as `spaces/{space}/threads/{thread}`. You can only
// specify one `thread.name` per query. To filter by both thread and date, use
// the `AND` operator in your query. For example, the following queries are
// valid: ``` create_time > "2012-04-21T11:30:00-04:00" create_time >
// "2012-04-21T11:30:00-04:00" AND thread.name = spaces/AAAAAAAAAAA/threads/123
// create_time > "2012-04-21T11:30:00+00:00" AND create_time <
// "2013-01-01T00:00:00+00:00" AND thread.name = spaces/AAAAAAAAAAA/threads/123
// thread.name = spaces/AAAAAAAAAAA/threads/123 ``` Invalid queries are
// rejected by the server with an `INVALID_ARGUMENT` error.
func (c *SpacesMessagesListCall) Filter(filter string) *SpacesMessagesListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// OrderBy sets the optional parameter "orderBy": How the list of messages is
// ordered. Specify a value to order by an ordering operation. Valid ordering
// operation values are as follows: - `ASC` for ascending. - `DESC` for
// descending. The default ordering is `create_time ASC`.
func (c *SpacesMessagesListCall) OrderBy(orderBy string) *SpacesMessagesListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// messages returned. The service might return fewer messages than this value.
// If unspecified, at most 25 are returned. The maximum value is 1000. If you
// use a value more than 1000, it's automatically changed to 1000. Negative
// values return an `INVALID_ARGUMENT` error.
func (c *SpacesMessagesListCall) PageSize(pageSize int64) *SpacesMessagesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token received
// from a previous list messages call. Provide this parameter to retrieve the
// subsequent page. When paginating, all other parameters provided should match
// the call that provided the page token. Passing different values to the other
// parameters might lead to unexpected results.
func (c *SpacesMessagesListCall) PageToken(pageToken string) *SpacesMessagesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// ShowDeleted sets the optional parameter "showDeleted": Whether to include
// deleted messages. Deleted messages include deleted time and metadata about
// their deletion, but message content is unavailable.
func (c *SpacesMessagesListCall) ShowDeleted(showDeleted bool) *SpacesMessagesListCall {
	c.urlParams_.Set("showDeleted", fmt.Sprint(showDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesListCall) Fields(s ...googleapi.Field) *SpacesMessagesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMessagesListCall) IfNoneMatch(entityTag string) *SpacesMessagesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesListCall) Context(ctx context.Context) *SpacesMessagesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/messages")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListMessagesResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SpacesMessagesListCall) Do(opts ...googleapi.CallOption) (*ListMessagesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMessagesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMessagesListCall) Pages(ctx context.Context, f func(*ListMessagesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesMessagesPatchCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a message. There's a difference between the `patch` and
// `update` methods. The `patch` method uses a `patch` request while the
// `update` method uses a `put` request. We recommend using the `patch` method.
// For an example, see Update a message
// (https://developers.google.com/workspace/chat/update-messages). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) When
// using app authentication, requests can only update messages created by the
// calling Chat app.
//
//   - name: Identifier. Resource name of the message. Format:
//     `spaces/{space}/messages/{message}` Where `{space}` is the ID of the space
//     where the message is posted and `{message}` is a system-assigned ID for
//     the message. For example,
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`. If you set a custom
//     ID when you create a message, you can use this ID to specify the message
//     in a request by replacing `{message}` with the value from the
//     `clientAssignedMessageId` field. For example,
//     `spaces/AAAAAAAAAAA/messages/client-custom-name`. For details, see Name a
//     message
//     (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
func (r *SpacesMessagesService) Patch(name string, message *Message) *SpacesMessagesPatchCall {
	c := &SpacesMessagesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true` and the
// message isn't found, a new message is created and `updateMask` is ignored.
// The specified message ID must be client-assigned
// (https://developers.google.com/workspace/chat/create-messages#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesPatchCall) AllowMissing(allowMissing bool) *SpacesMessagesPatchCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The field
// paths to update. Separate multiple values with commas or use `*` to update
// all field paths. Currently supported field paths: - `text` - `attachment` -
// `cards` (Requires app authentication
// (/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires app
// authentication (/chat/api/guides/auth/service-accounts).) -
// `accessory_widgets` (Requires app authentication
// (/chat/api/guides/auth/service-accounts).) - `quoted_message_metadata` (Only
// allows removal of the quoted message.)
func (c *SpacesMessagesPatchCall) UpdateMask(updateMask string) *SpacesMessagesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesPatchCall) Fields(s ...googleapi.Field) *SpacesMessagesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesPatchCall) Context(ctx context.Context) *SpacesMessagesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.message)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesPatchCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesUpdateCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a message. There's a difference between the `patch` and
// `update` methods. The `patch` method uses a `patch` request while the
// `update` method uses a `put` request. We recommend using the `patch` method.
// For an example, see Update a message
// (https://developers.google.com/workspace/chat/update-messages). Supports the
// following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize): - App
// authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope: - `https://www.googleapis.com/auth/chat.bot` -
// User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only) When
// using app authentication, requests can only update messages created by the
// calling Chat app.
//
//   - name: Identifier. Resource name of the message. Format:
//     `spaces/{space}/messages/{message}` Where `{space}` is the ID of the space
//     where the message is posted and `{message}` is a system-assigned ID for
//     the message. For example,
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`. If you set a custom
//     ID when you create a message, you can use this ID to specify the message
//     in a request by replacing `{message}` with the value from the
//     `clientAssignedMessageId` field. For example,
//     `spaces/AAAAAAAAAAA/messages/client-custom-name`. For details, see Name a
//     message
//     (https://developers.google.com/workspace/chat/create-messages#name_a_created_message).
func (r *SpacesMessagesService) Update(name string, message *Message) *SpacesMessagesUpdateCall {
	c := &SpacesMessagesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true` and the
// message isn't found, a new message is created and `updateMask` is ignored.
// The specified message ID must be client-assigned
// (https://developers.google.com/workspace/chat/create-messages#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesUpdateCall) AllowMissing(allowMissing bool) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The field
// paths to update. Separate multiple values with commas or use `*` to update
// all field paths. Currently supported field paths: - `text` - `attachment` -
// `cards` (Requires app authentication
// (/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires app
// authentication (/chat/api/guides/auth/service-accounts).) -
// `accessory_widgets` (Requires app authentication
// (/chat/api/guides/auth/service-accounts).) - `quoted_message_metadata` (Only
// allows removal of the quoted message.)
func (c *SpacesMessagesUpdateCall) UpdateMask(updateMask string) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesUpdateCall) Fields(s ...googleapi.Field) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesUpdateCall) Context(ctx context.Context) *SpacesMessagesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.message)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesUpdateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesAttachmentsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the metadata of a message attachment. The attachment data is
// fetched using the media API
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/media/download).
// For an example, see Get metadata about a message attachment
// (https://developers.google.com/workspace/chat/get-media-attachments).
// Requires app authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.bot`
//
//   - name: Resource name of the attachment, in the form
//     `spaces/{space}/messages/{message}/attachments/{attachment}`.
func (r *SpacesMessagesAttachmentsService) Get(name string) *SpacesMessagesAttachmentsGetCall {
	c := &SpacesMessagesAttachmentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesAttachmentsGetCall) Fields(s ...googleapi.Field) *SpacesMessagesAttachmentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMessagesAttachmentsGetCall) IfNoneMatch(entityTag string) *SpacesMessagesAttachmentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesAttachmentsGetCall) Context(ctx context.Context) *SpacesMessagesAttachmentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesAttachmentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesAttachmentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.attachments.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.attachments.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Attachment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesAttachmentsGetCall) Do(opts ...googleapi.CallOption) (*Attachment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Attachment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.attachments.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesReactionsCreateCall struct {
	s          *Service
	parent     string
	reaction   *Reaction
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a reaction and adds it to a message. For an example, see Add
// a reaction to a message
// (https://developers.google.com/workspace/chat/create-reactions). Requires
// user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.messages.reactions.create` -
// `https://www.googleapis.com/auth/chat.messages.reactions` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only)
//
//   - parent: The message where the reaction is created. Format:
//     `spaces/{space}/messages/{message}`.
func (r *SpacesMessagesReactionsService) Create(parent string, reaction *Reaction) *SpacesMessagesReactionsCreateCall {
	c := &SpacesMessagesReactionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.reaction = reaction
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesReactionsCreateCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesReactionsCreateCall) Context(ctx context.Context) *SpacesMessagesReactionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesReactionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.reaction)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/reactions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Reaction.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesReactionsCreateCall) Do(opts ...googleapi.CallOption) (*Reaction, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Reaction{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesReactionsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a reaction to a message. For an example, see Delete a
// reaction (https://developers.google.com/workspace/chat/delete-reactions).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.messages.reactions` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.import` (import mode spaces only)
//
//   - name: Name of the reaction to delete. Format:
//     `spaces/{space}/messages/{message}/reactions/{reaction}`.
func (r *SpacesMessagesReactionsService) Delete(name string) *SpacesMessagesReactionsDeleteCall {
	c := &SpacesMessagesReactionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesReactionsDeleteCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesReactionsDeleteCall) Context(ctx context.Context) *SpacesMessagesReactionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesReactionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesMessagesReactionsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesMessagesReactionsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists reactions to a message. For an example, see List reactions for a
// message (https://developers.google.com/workspace/chat/list-reactions).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.messages.reactions.readonly` -
// `https://www.googleapis.com/auth/chat.messages.reactions` -
// `https://www.googleapis.com/auth/chat.messages.readonly` -
// `https://www.googleapis.com/auth/chat.messages`
//
//   - parent: The message users reacted to. Format:
//     `spaces/{space}/messages/{message}`.
func (r *SpacesMessagesReactionsService) List(parent string) *SpacesMessagesReactionsListCall {
	c := &SpacesMessagesReactionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can filter
// reactions by emoji
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/Emoji)
// (either `emoji.unicode` or `emoji.custom_emoji.uid`) and user
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/User)
// (`user.name`). To filter reactions for multiple emojis or users, join
// similar fields with the `OR` operator, such as `emoji.unicode = "🙂" OR
// emoji.unicode = "👍" and `user.name = "users/AAAAAA" OR user.name =
// "users/BBBBBB". To filter reactions by emoji and user, use the `AND`
// operator, such as `emoji.unicode = "🙂" AND user.name = "users/AAAAAA".
// If your query uses both `AND` and `OR`, group them with parentheses. For
// example, the following queries are valid: ``` user.name = "users/{user}"
// emoji.unicode = "🙂" emoji.custom_emoji.uid = "{uid}" emoji.unicode =
// "🙂" OR emoji.unicode = "👍" emoji.unicode = "🙂" OR
// emoji.custom_emoji.uid = "{uid}" emoji.unicode = "🙂" AND user.name =
// "users/{user}" (emoji.unicode = "🙂" OR emoji.custom_emoji.uid = "{uid}")
// AND user.name = "users/{user}" ``` The following queries are invalid: ```
// emoji.unicode = "🙂" AND emoji.unicode = "👍" emoji.unicode = "🙂" AND
// emoji.custom_emoji.uid = "{uid}" emoji.unicode = "🙂" OR user.name =
// "users/{user}" emoji.unicode = "🙂" OR emoji.custom_emoji.uid = "{uid}" OR
// user.name = "users/{user}" emoji.unicode = "🙂" OR emoji.custom_emoji.uid
// = "{uid}" AND user.name = "users/{user}" ``` Invalid queries are rejected
// with an `INVALID_ARGUMENT` error.
func (c *SpacesMessagesReactionsListCall) Filter(filter string) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// reactions returned. The service can return fewer reactions than this value.
// If unspecified, the default value is 25. The maximum value is 200; values
// above 200 are changed to 200.
func (c *SpacesMessagesReactionsListCall) PageSize(pageSize int64) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": (If resuming from a
// previous query.) A page token received from a previous list reactions call.
// Provide this to retrieve the subsequent page. When paginating, the filter
// value should match the call that provided the page token. Passing a
// different value might lead to unexpected results.
func (c *SpacesMessagesReactionsListCall) PageToken(pageToken string) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesMessagesReactionsListCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesMessagesReactionsListCall) IfNoneMatch(entityTag string) *SpacesMessagesReactionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesMessagesReactionsListCall) Context(ctx context.Context) *SpacesMessagesReactionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesMessagesReactionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/reactions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListReactionsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SpacesMessagesReactionsListCall) Do(opts ...googleapi.CallOption) (*ListReactionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListReactionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.messages.reactions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMessagesReactionsListCall) Pages(ctx context.Context, f func(*ListReactionsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesSpaceEventsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns an event from a Google Chat space. The event payload
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.spaceEvents#SpaceEvent.FIELDS.oneof_payload)
// contains the most recent version of the resource that changed. For example,
// if you request an event about a new message but the message was later
// updated, the server returns the updated `Message` resource in the event
// payload. Note: The `permissionSettings` field is not returned in the Space
// object of the Space event data for this request. Supports the following
// types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize) with
// an authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes)
// appropriate for reading the requested data: - App authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// in Developer Preview (https://developers.google.com/workspace/preview) with
// one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.app.spaces` -
// `https://www.googleapis.com/auth/chat.app.messages.readonly` -
// `https://www.googleapis.com/auth/chat.app.memberships` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.spaces` -
// `https://www.googleapis.com/auth/chat.messages.readonly` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.messages.reactions.readonly` -
// `https://www.googleapis.com/auth/chat.messages.reactions` -
// `https://www.googleapis.com/auth/chat.memberships.readonly` -
// `https://www.googleapis.com/auth/chat.memberships` To get an event, the
// authenticated caller must be a member of the space. For an example, see Get
// details about an event from a Google Chat space
// (https://developers.google.com/workspace/chat/get-space-event).
//
//   - name: The resource name of the space event. Format:
//     `spaces/{space}/spaceEvents/{spaceEvent}`.
func (r *SpacesSpaceEventsService) Get(name string) *SpacesSpaceEventsGetCall {
	c := &SpacesSpaceEventsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesSpaceEventsGetCall) Fields(s ...googleapi.Field) *SpacesSpaceEventsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesSpaceEventsGetCall) IfNoneMatch(entityTag string) *SpacesSpaceEventsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesSpaceEventsGetCall) Context(ctx context.Context) *SpacesSpaceEventsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesSpaceEventsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesSpaceEventsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.spaceEvents.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.spaceEvents.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *SpaceEvent.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesSpaceEventsGetCall) Do(opts ...googleapi.CallOption) (*SpaceEvent, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SpaceEvent{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.spaceEvents.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesSpaceEventsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists events from a Google Chat space. For each event, the payload
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.spaceEvents#SpaceEvent.FIELDS.oneof_payload)
// contains the most recent version of the Chat resource. For example, if you
// list events about new space members, the server returns `Membership`
// resources that contain the latest membership details. If new members were
// removed during the requested period, the event payload contains an empty
// `Membership` resource. Supports the following types of authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize) with
// an authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes)
// appropriate for reading the requested data: - App authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app)
// with administrator approval (https://support.google.com/a?p=chat-app-auth)
// in Developer Preview (https://developers.google.com/workspace/preview) with
// one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.app.spaces` -
// `https://www.googleapis.com/auth/chat.app.messages.readonly` -
// `https://www.googleapis.com/auth/chat.app.memberships` - User authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes: -
// `https://www.googleapis.com/auth/chat.spaces.readonly` -
// `https://www.googleapis.com/auth/chat.spaces` -
// `https://www.googleapis.com/auth/chat.messages.readonly` -
// `https://www.googleapis.com/auth/chat.messages` -
// `https://www.googleapis.com/auth/chat.messages.reactions.readonly` -
// `https://www.googleapis.com/auth/chat.messages.reactions` -
// `https://www.googleapis.com/auth/chat.memberships.readonly` -
// `https://www.googleapis.com/auth/chat.memberships` To list events, the
// authenticated caller must be a member of the space. For an example, see List
// events from a Google Chat space
// (https://developers.google.com/workspace/chat/list-space-events).
//
//   - parent: Resource name of the Google Chat space
//     (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces)
//     where the events occurred. Format: `spaces/{space}`.
func (r *SpacesSpaceEventsService) List(parent string) *SpacesSpaceEventsListCall {
	c := &SpacesSpaceEventsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": Required. A query filter. You
// must specify at least one event type (`event_type`) using the has `:`
// operator. To filter by multiple event types, use the `OR` operator. Omit
// batch event types in your filter. The request automatically returns any
// related batch events. For example, if you filter by new reactions
// (`google.workspace.chat.reaction.v1.created`), the server also returns batch
// new reactions events (`google.workspace.chat.reaction.v1.batchCreated`). For
// a list of supported event types, see the `SpaceEvents` reference
// documentation
// (https://developers.google.com/workspace/chat/api/reference/rest/v1/spaces.spaceEvents#SpaceEvent.FIELDS.event_type).
// Optionally, you can also filter by start time (`start_time`) and end time
// (`end_time`): * `start_time`: Exclusive timestamp from which to start
// listing space events. You can list events that occurred up to 28 days ago.
// If unspecified, lists space events from the past 28 days. * `end_time`:
// Inclusive timestamp until which space events are listed. If unspecified,
// lists events up to the time of the request. To specify a start or end time,
// use the equals `=` operator and format in RFC-3339
// (https://www.rfc-editor.org/rfc/rfc3339). To filter by both `start_time` and
// `end_time`, use the `AND` operator. For example, the following queries are
// valid: ``` start_time="2023-08-23T19:20:33+00:00" AND
// end_time="2023-08-23T19:21:54+00:00" ``` ```
// start_time="2023-08-23T19:20:33+00:00" AND
// (event_types:"google.workspace.chat.space.v1.updated" OR
// event_types:"google.workspace.chat.message.v1.created") ``` The following
// queries are invalid: ``` start_time="2023-08-23T19:20:33+00:00" OR
// end_time="2023-08-23T19:21:54+00:00" ``` ```
// event_types:"google.workspace.chat.space.v1.updated" AND
// event_types:"google.workspace.chat.message.v1.created" ``` Invalid queries
// are rejected by the server with an `INVALID_ARGUMENT` error.
func (c *SpacesSpaceEventsListCall) Filter(filter string) *SpacesSpaceEventsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of space
// events returned. The service might return fewer than this value. Negative
// values return an `INVALID_ARGUMENT` error.
func (c *SpacesSpaceEventsListCall) PageSize(pageSize int64) *SpacesSpaceEventsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous list space events call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to list
// space events must match the call that provided the page token. Passing
// different values to the other parameters might lead to unexpected results.
func (c *SpacesSpaceEventsListCall) PageToken(pageToken string) *SpacesSpaceEventsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesSpaceEventsListCall) Fields(s ...googleapi.Field) *SpacesSpaceEventsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesSpaceEventsListCall) IfNoneMatch(entityTag string) *SpacesSpaceEventsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesSpaceEventsListCall) Context(ctx context.Context) *SpacesSpaceEventsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesSpaceEventsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesSpaceEventsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/spaceEvents")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.spaces.spaceEvents.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.spaceEvents.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSpaceEventsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesSpaceEventsListCall) Do(opts ...googleapi.CallOption) (*ListSpaceEventsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSpaceEventsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.spaces.spaceEvents.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesSpaceEventsListCall) Pages(ctx context.Context, f func(*ListSpaceEventsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type UsersSpacesGetSpaceReadStateCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetSpaceReadState: Returns details about a user's read state within a space,
// used to identify read and unread messages. For an example, see Get details
// about a user's space read state
// (https://developers.google.com/workspace/chat/get-space-read-state).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.users.readstate.readonly` -
// `https://www.googleapis.com/auth/chat.users.readstate`
//
//   - name: Resource name of the space read state to retrieve. Only supports
//     getting read state for the calling user. To refer to the calling user, set
//     one of the following: - The `me` alias. For example,
//     `users/me/spaces/{space}/spaceReadState`. - Their Workspace email address.
//     For example, `users/user@example.com/spaces/{space}/spaceReadState`. -
//     Their user id. For example,
//     `users/123456789/spaces/{space}/spaceReadState`. Format:
//     users/{user}/spaces/{space}/spaceReadState.
func (r *UsersSpacesService) GetSpaceReadState(name string) *UsersSpacesGetSpaceReadStateCall {
	c := &UsersSpacesGetSpaceReadStateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UsersSpacesGetSpaceReadStateCall) Fields(s ...googleapi.Field) *UsersSpacesGetSpaceReadStateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *UsersSpacesGetSpaceReadStateCall) IfNoneMatch(entityTag string) *UsersSpacesGetSpaceReadStateCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UsersSpacesGetSpaceReadStateCall) Context(ctx context.Context) *UsersSpacesGetSpaceReadStateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UsersSpacesGetSpaceReadStateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UsersSpacesGetSpaceReadStateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.users.spaces.getSpaceReadState", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.users.spaces.getSpaceReadState" call.
// Any non-2xx status code is an error. Response headers are in either
// *SpaceReadState.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *UsersSpacesGetSpaceReadStateCall) Do(opts ...googleapi.CallOption) (*SpaceReadState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SpaceReadState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.users.spaces.getSpaceReadState", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UsersSpacesUpdateSpaceReadStateCall struct {
	s              *Service
	name           string
	spacereadstate *SpaceReadState
	urlParams_     gensupport.URLParams
	ctx_           context.Context
	header_        http.Header
}

// UpdateSpaceReadState: Updates a user's read state within a space, used to
// identify read and unread messages. For an example, see Update a user's space
// read state
// (https://developers.google.com/workspace/chat/update-space-read-state).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.users.readstate`
//
//   - name: Resource name of the space read state. Format:
//     `users/{user}/spaces/{space}/spaceReadState`.
func (r *UsersSpacesService) UpdateSpaceReadState(name string, spacereadstate *SpaceReadState) *UsersSpacesUpdateSpaceReadStateCall {
	c := &UsersSpacesUpdateSpaceReadStateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.spacereadstate = spacereadstate
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The field
// paths to update. Currently supported field paths: - `last_read_time` When
// the `last_read_time` is before the latest message create time, the space
// appears as unread in the UI. To mark the space as read, set `last_read_time`
// to any value later (larger) than the latest message create time. The
// `last_read_time` is coerced to match the latest message create time. Note
// that the space read state only affects the read state of messages that are
// visible in the space's top-level conversation. Replies in threads are
// unaffected by this timestamp, and instead rely on the thread read state.
func (c *UsersSpacesUpdateSpaceReadStateCall) UpdateMask(updateMask string) *UsersSpacesUpdateSpaceReadStateCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UsersSpacesUpdateSpaceReadStateCall) Fields(s ...googleapi.Field) *UsersSpacesUpdateSpaceReadStateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UsersSpacesUpdateSpaceReadStateCall) Context(ctx context.Context) *UsersSpacesUpdateSpaceReadStateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UsersSpacesUpdateSpaceReadStateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UsersSpacesUpdateSpaceReadStateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.spacereadstate)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.users.spaces.updateSpaceReadState", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.users.spaces.updateSpaceReadState" call.
// Any non-2xx status code is an error. Response headers are in either
// *SpaceReadState.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *UsersSpacesUpdateSpaceReadStateCall) Do(opts ...googleapi.CallOption) (*SpaceReadState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SpaceReadState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.users.spaces.updateSpaceReadState", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UsersSpacesSpaceNotificationSettingGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the space notification setting. For an example, see Get the
// caller's space notification setting
// (https://developers.google.com/workspace/chat/get-space-notification-setting).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.users.spacesettings`
//
//   - name: Format: users/{user}/spaces/{space}/spaceNotificationSetting -
//     `users/me/spaces/{space}/spaceNotificationSetting`, OR -
//     `users/user@example.com/spaces/{space}/spaceNotificationSetting`, OR -
//     `users/123456789/spaces/{space}/spaceNotificationSetting`. Note: Only the
//     caller's user id or email is allowed in the path.
func (r *UsersSpacesSpaceNotificationSettingService) Get(name string) *UsersSpacesSpaceNotificationSettingGetCall {
	c := &UsersSpacesSpaceNotificationSettingGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UsersSpacesSpaceNotificationSettingGetCall) Fields(s ...googleapi.Field) *UsersSpacesSpaceNotificationSettingGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *UsersSpacesSpaceNotificationSettingGetCall) IfNoneMatch(entityTag string) *UsersSpacesSpaceNotificationSettingGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UsersSpacesSpaceNotificationSettingGetCall) Context(ctx context.Context) *UsersSpacesSpaceNotificationSettingGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UsersSpacesSpaceNotificationSettingGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UsersSpacesSpaceNotificationSettingGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.users.spaces.spaceNotificationSetting.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.users.spaces.spaceNotificationSetting.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *SpaceNotificationSetting.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *UsersSpacesSpaceNotificationSettingGetCall) Do(opts ...googleapi.CallOption) (*SpaceNotificationSetting, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SpaceNotificationSetting{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.users.spaces.spaceNotificationSetting.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UsersSpacesSpaceNotificationSettingPatchCall struct {
	s                        *Service
	name                     string
	spacenotificationsetting *SpaceNotificationSetting
	urlParams_               gensupport.URLParams
	ctx_                     context.Context
	header_                  http.Header
}

// Patch: Updates the space notification setting. For an example, see Update
// the caller's space notification setting
// (https://developers.google.com/workspace/chat/update-space-notification-setting).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with the authorization scope
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.users.spacesettings`
//
//   - name: Identifier. The resource name of the space notification setting.
//     Format: `users/{user}/spaces/{space}/spaceNotificationSetting`.
func (r *UsersSpacesSpaceNotificationSettingService) Patch(name string, spacenotificationsetting *SpaceNotificationSetting) *UsersSpacesSpaceNotificationSettingPatchCall {
	c := &UsersSpacesSpaceNotificationSettingPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.spacenotificationsetting = spacenotificationsetting
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Supported
// field paths: - `notification_setting` - `mute_setting`
func (c *UsersSpacesSpaceNotificationSettingPatchCall) UpdateMask(updateMask string) *UsersSpacesSpaceNotificationSettingPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UsersSpacesSpaceNotificationSettingPatchCall) Fields(s ...googleapi.Field) *UsersSpacesSpaceNotificationSettingPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UsersSpacesSpaceNotificationSettingPatchCall) Context(ctx context.Context) *UsersSpacesSpaceNotificationSettingPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UsersSpacesSpaceNotificationSettingPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UsersSpacesSpaceNotificationSettingPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.spacenotificationsetting)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.users.spaces.spaceNotificationSetting.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.users.spaces.spaceNotificationSetting.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *SpaceNotificationSetting.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *UsersSpacesSpaceNotificationSettingPatchCall) Do(opts ...googleapi.CallOption) (*SpaceNotificationSetting, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SpaceNotificationSetting{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.users.spaces.spaceNotificationSetting.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UsersSpacesThreadsGetThreadReadStateCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetThreadReadState: Returns details about a user's read state within a
// thread, used to identify read and unread messages. For an example, see Get
// details about a user's thread read state
// (https://developers.google.com/workspace/chat/get-thread-read-state).
// Requires user authentication
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user)
// with one of the following authorization scopes
// (https://developers.google.com/workspace/chat/authenticate-authorize#chat-api-scopes):
// - `https://www.googleapis.com/auth/chat.users.readstate.readonly` -
// `https://www.googleapis.com/auth/chat.users.readstate`
//
//   - name: Resource name of the thread read state to retrieve. Only supports
//     getting read state for the calling user. To refer to the calling user, set
//     one of the following: - The `me` alias. For example,
//     `users/me/spaces/{space}/threads/{thread}/threadReadState`. - Their
//     Workspace email address. For example,
//     `users/user@example.com/spaces/{space}/threads/{thread}/threadReadState`.
//   - Their user id. For example,
//     `users/123456789/spaces/{space}/threads/{thread}/threadReadState`. Format:
//     users/{user}/spaces/{space}/threads/{thread}/threadReadState.
func (r *UsersSpacesThreadsService) GetThreadReadState(name string) *UsersSpacesThreadsGetThreadReadStateCall {
	c := &UsersSpacesThreadsGetThreadReadStateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UsersSpacesThreadsGetThreadReadStateCall) Fields(s ...googleapi.Field) *UsersSpacesThreadsGetThreadReadStateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *UsersSpacesThreadsGetThreadReadStateCall) IfNoneMatch(entityTag string) *UsersSpacesThreadsGetThreadReadStateCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UsersSpacesThreadsGetThreadReadStateCall) Context(ctx context.Context) *UsersSpacesThreadsGetThreadReadStateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UsersSpacesThreadsGetThreadReadStateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UsersSpacesThreadsGetThreadReadStateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chat.users.spaces.threads.getThreadReadState", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.users.spaces.threads.getThreadReadState" call.
// Any non-2xx status code is an error. Response headers are in either
// *ThreadReadState.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *UsersSpacesThreadsGetThreadReadStateCall) Do(opts ...googleapi.CallOption) (*ThreadReadState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ThreadReadState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chat.users.spaces.threads.getThreadReadState", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
