// Package line implements the OAuth2 protocol for authenticating users through line.
// This package can be used as a reference implementation of an OAuth2 provider for Goth.
package line

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"net/http"

	"github.com/golang-jwt/jwt/v5"
	"github.com/markbates/goth"
	"golang.org/x/oauth2"
)

const (
	authURL      string = "https://access.line.me/oauth2/v2.1/authorize"
	tokenURL     string = "https://api.line.me/oauth2/v2.1/token"
	endpointUser string = "https://api.line.me/v2/profile"
	issuerURL    string = "https://access.line.me"
)

type IDTokenClaims struct {
	jwt.RegisteredClaims
	Email string `json:"email"`
}

// Provider is the implementation of `goth.Provider` for accessing Line.me.
type Provider struct {
	ClientKey       string
	Secret          string
	CallbackURL     string
	HTTPClient      *http.Client
	config          *oauth2.Config
	authCodeOptions []oauth2.AuthCodeOption
	providerName    string
}

// New creates a new Line provider and sets up important connection details.
// You should always call `line.New` to get a new provider.  Never try to
// create one manually.
func New(clientKey, secret, callbackURL string, scopes ...string) *Provider {
	p := &Provider{
		ClientKey:    clientKey,
		Secret:       secret,
		CallbackURL:  callbackURL,
		providerName: "line",
	}
	p.config = newConfig(p, scopes)
	return p
}

// Name is the name used to retrieve this provider later.
func (p *Provider) Name() string {
	return p.providerName
}

// SetName is to update the name of the provider (needed in case of multiple providers of 1 type)
func (p *Provider) SetName(name string) {
	p.providerName = name
}

// Client returns a pointer to http.Client setting some client fallback.
func (p *Provider) Client() *http.Client {
	return goth.HTTPClientWithFallBack(p.HTTPClient)
}

// Debug is a no-op for the line package.
func (p *Provider) Debug(debug bool) {}

// BeginAuth asks line.me for an authentication end-point.
func (p *Provider) BeginAuth(state string) (goth.Session, error) {
	return &Session{
		AuthURL: p.config.AuthCodeURL(state, p.authCodeOptions...),
	}, nil
}

// FetchUser will go to line.me and access basic information about the user.
func (p *Provider) FetchUser(session goth.Session) (goth.User, error) {
	sess := session.(*Session)
	user := goth.User{
		AccessToken:  sess.AccessToken,
		Provider:     p.Name(),
		RefreshToken: sess.RefreshToken,
		ExpiresAt:    sess.ExpiresAt,
	}

	if user.AccessToken == "" {
		// data is not yet retrieved since accessToken is still empty
		return user, fmt.Errorf("%s cannot get user information without accessToken", p.providerName)
	}

	// Get the userID, line needs userID in order to get user profile info
	c := p.Client()
	req, err := http.NewRequest("GET", endpointUser, nil)
	if err != nil {
		return user, err
	}

	req.Header.Add("Authorization", "Bearer "+sess.AccessToken)

	response, err := c.Do(req)
	if err != nil {
		return user, err
	}
	defer response.Body.Close()

	if response.StatusCode != http.StatusOK {
		return user, fmt.Errorf("%s responded with a %d trying to fetch user information", p.providerName, response.StatusCode)
	}

	bits, err := io.ReadAll(response.Body)
	if err != nil {
		return user, err
	}

	u := struct {
		Name          string `json:"name"`
		UserID        string `json:"userId"`
		DisplayName   string `json:"displayName"`
		PictureURL    string `json:"pictureUrl"`
		StatusMessage string `json:"statusMessage"`
	}{}

	if err = json.NewDecoder(bytes.NewReader(bits)).Decode(&u); err != nil {
		return user, err
	}

	user.NickName = u.DisplayName
	user.AvatarURL = u.PictureURL
	user.UserID = u.UserID

	if sess.IDToken != "" {
		if err = p.addDataFromIdToken(sess.IDToken, &user); err != nil {
			return user, err
		}
	}

	return user, err
}

func newConfig(provider *Provider, scopes []string) *oauth2.Config {
	c := &oauth2.Config{
		ClientID:     provider.ClientKey,
		ClientSecret: provider.Secret,
		RedirectURL:  provider.CallbackURL,
		Endpoint: oauth2.Endpoint{
			AuthURL:  authURL,
			TokenURL: tokenURL,
		},
		Scopes: []string{},
	}

	if len(scopes) > 0 {
		c.Scopes = append(c.Scopes, scopes...)
	}
	return c
}

// RefreshTokenAvailable refresh token is provided by auth provider or not
func (p *Provider) RefreshTokenAvailable() bool {
	return false
}

// RefreshToken get new access token based on the refresh token
func (p *Provider) RefreshToken(refreshToken string) (*oauth2.Token, error) {
	return nil, nil
}

// SetBotPrompt sets the bot_prompt parameter for the line OAuth call.
// Use this to display the option to add your LINE Official Account as a friend.
// See https://developers.line.biz/en/docs/line-login/link-a-bot/#redirect-users
func (p *Provider) SetBotPrompt(botPrompt string) {
	if botPrompt == "" {
		return
	}
	p.authCodeOptions = append(p.authCodeOptions, oauth2.SetAuthURLParam("bot_prompt", botPrompt))
}

func (p *Provider) addDataFromIdToken(idToken string, user *goth.User) error {
	token, err := jwt.ParseWithClaims(idToken, &IDTokenClaims{}, func(t *jwt.Token) (interface{}, error) {
		return []byte(p.Secret), nil
	},
		jwt.WithAudience(p.ClientKey),
		jwt.WithIssuer(issuerURL),
		jwt.WithExpirationRequired(),
	)
	if err != nil {
		return err
	}

	user.Email = token.Claims.(*IDTokenClaims).Email

	return nil
}
